7900 '---------------------------------------------------------------------
7901 'NAME:      MD2BI1S.BAS
7902 'DESC:      MD-2 LEVEL 1 MOTION CONTROL SUBROUTINES FOR BASIC INTERPRETERS.
7903 'USAGE:     USE WHEN CREATING A CUSTOM MOTION CONTROL PROGRAM
7904 '           THAT DOESN'T REQUIRE COMPLEX FEATURES SUCH AS RAMPING, ETC.
7905 '           MOTOR PARAMETERS ARE PASSED VIA GLOBAL VARIABLES.
7906 '           ALL VARIABLES AND SUBS BEGIN WITH MD2 TO AVOID CONFLICTS.
7907 'BY:        COPYRIGHT (C) 1993 ARRICK ROBOTICS, ROGER ARRICK.
7908 'DATE:      11/6/93
7909 'EDIT:      12
7910 '
7911 'SUBROUTINES ---------------------------------------------------------
7912 '
7913 '    NAME           DESCRIPTION
7914 '    ----           -----------
7915 ' 8000 MD2SETUP     Used at the beginning of a program to initialize
7916 '                   motor parameters to their default values.
7917 '                   Use this subroutine before any other.
7918 ' 8400 MD2ON        Turns on an MD-2 system.
7919 ' 8600 MD2OFF       Turns off an MD-2 system.
7920 ' 8800 MD2HOME      Moves a motor to the Home position by watching the
7921 '                   home switch.  Current position is set to zero.
7922 ' 9500 MD2MOVE      Moves a motor the number of steps in MD2TARGET(M)
7923 '                   parameter.
7924 '
7925 'MOTOR PARAMETERS AND VARIABLES --------------------------------------
7926 '
7927 '    NAME            DESCRIPTION
7928 '    ----            -----------
7929 '
7930 '    MD2HOLD         -1=Leaves the motor energized after a move
7931 '                    to cause the motor to hold its position.
7932 '                    0 causes the motor to turn off after a move
7933 '                    which conserves power and reduces heat.
7934 '    MD2MOTOR        The selected motor to act upon. 1,2,3,4,5 or 6.
7935 '    MD2SPEED(M)     Delay count between steps. Determines the
7936 '                    motor's.  0=fast, 32766=slow.
7937 '                    Actual motor speed depends on the computer.
7938 '                    M is the motor number 1,2,3,4,5 or 6.
7939 '    MD2POSITION(M)  Current motor position for each motor in steps
7940 '                    relative to home. Negative = reverse from home
7941 '                    and positive = forward from home.
7942 '                    M is the motor number 1,2,3,4,5 or 6.
7943 '    MD2STATUS$      Completion status. O=motion completed OK,
7944 '                    K=a keypress stopped the motion, B=Bad parameter.
7945 '    MD2TARGET(M)    The number of steps to move. Positive #'s are
7946 '                    forward, negative are reverse.
7947 '                    M is the motor number 1,2,3,4,5 or 6.
7948 '
7949 'VARIABLES NEEDED BY SUBROUTINES BUT NOT BY PROGRAMMER ---------------
7950 '
7951 '    NAME            DESCRIPTION
7952 '    ----            -----------
7953 '
7954 '    MD2DELAY        Used for delay counter.
7955 '    MD2TIMER        Used for timer.
7956 '    MD2STEPS        Used for step counter.
7957 '    MD2DIR          Used as direction storage.
7958 '    MD2PMASK        Bit mask for selected motor.
7959 '    MD2OMASK        Bit mask for other motor.
7960 '    MD2SMASK        Bit mask for switch.
7961 '    MD2MTRADR       Selected port address.
7962 '    MD2MTRADR12     Port address for motors 1 & 2.
7963 '    MD2MTRADR34     Port address for motors 3 & 4.
7964 '    MD2MTRADR56     Port address for motors 5 & 6.
7965 '    MD2STPPAT(7)    Step pattern array.
7966 '    MD2PATPTR(6)    Pattern pointer for each motor.
7967 '    MD2PATTERN      Selected step pattern.
7968 '---------------------------------------------------------------------

100 'THESE STATEMENTS NEED TO BE AT THE BEGINNING OF A PROGRAM.
110 DIM MD2STPPAT(7)    'STEP PATTERN ARRAY.
120 DIM MD2PATPTR(6)    'STEP PATTERN POINTERS.
130 DIM MD2POSITION(6)  'MOTOR POSITIONS.
140 DIM MD2SPEED(6)     'MOTOR SPEED.
150 DIM MD2TARGET(6)    'TARGET DISTANCE AND DIRECTION.

8000 '---------------------------------------------------------------------
8010 'NAME:      MD2SETUP
8020 'DESC:      THE MD2SETUP PROCEDURE SETS DEFAULT MD-2 SYSTEM
8030 '           PARAMETERS SUCH AS MOTOR SPEED, CURRENT POSITION, ETC.
8040 'USAGE:     USE AT THE BEGINNING OF A MOTION CONTROL PROGRAM.
8050 '           MUST BE USED -BEFORE- ANY OTHER MOTION CONTROL SUBROUTINE!
8060 'INPUTS:    NONE.
8070 'OUTPUTS:   DEFAULT MOTOR PARAMETERS.
8080 '---------------------------------------------------------------------
8090 'SETUP PORT ADDRESSES.
8100 MD2MTRADR12 = &H3BC
8110 MD2MTRADR34 = &H378
8120 MD2MTRADR56 = &H278
8130 'SETUP MOTOR PARAMETER DEFAULTS.
8140 FOR MD2MOTOR = 1 TO 6
8150 MD2POSITION(MD2MOTOR) = 0  'POSITIONS = 0.
8160 MD2PATPTR(MD2MOTOR) = 0    'STEP PATTERN POINTERS.
8170 MD2TARGET(MD2MOTOR) = 0    'TARGET POSITION,STEPS.
8180 MD2SPEED(MD2MOTOR) = 5000  'SPEED.
8190 NEXT MD2MOTOR
8200 MD2MOTOR = 1               'MOTOR NUMBER.
8210 MD2STATUS$ = "O"           'STATUS=OK.
8220 MD2HOLD = 0                'DON'T HOLD MOTORS.
8230 'SET HALF STEP PHASE PATTERNS.
8240 MD2STPPAT(0) = &H66
8250 MD2STPPAT(1) = &H77
8260 MD2STPPAT(2) = &H33
8270 MD2STPPAT(3) = &HBB
8280 MD2STPPAT(4) = &H99
8290 MD2STPPAT(5) = &HDD
8300 MD2STPPAT(6) = &HCC
8310 MD2STPPAT(7) = &HEE
8320 RETURN

8400 '---------------------------------------------------------------------
8410 'NAME:      MD2ON
8420 'DESC:      THE MD2ON PROCEDURE INITIALIZES A PARALLEL PRINTER PORT
8430 '           AND TURNS ON AN MD-2.
8440 'USAGE:     USE AT THE BEGINNING OF A MOTION CONTROL PROGRAM BUT
8450 '           AFTER THE MD2SETUP SUBROUTINE.
8460 'INPUTS:    MOTOR # DETERMINES PORT.
8470 'OUTPUTS:   NONE.
8480 '---------------------------------------------------------------------
8490 IF MD2MOTOR = 1 OR MD2MOTOR = 2 THEN OUT MD2MTRADR12, &HFF:OUT MD2MTRADR12 + 2, &H5
8500 IF MD2MOTOR = 3 OR MD2MOTOR = 4 THEN OUT MD2MTRADR34, &HFF:OUT MD2MTRADR34 + 2, &H5
8510 IF MD2MOTOR = 5 OR MD2MOTOR = 6 THEN OUT MD2MTRADR56, &HFF:OUT MD2MTRADR56 + 2, &H5
8520 RETURN

8600 '---------------------------------------------------------------------
8610 'NAME:      MD2OFF
8620 'DESC:      THE MD2OFF PROCEDURE RETURNS A PARALLEL PRINTER PORT
8630 '           REFERENCED BY THE MOTOR # TO ITS PREVIOUS STATE READY
8640 '           FOR USE WITH A PRINTER AND DISABLES THE MD-2.
8650 'USAGE:     USE AT THE END OF A MOTION CONTROL PROGRAM.
8660 'INPUTS:    MOTOR # DETERMINES PORT.
8670 'OUTPUTS:   NONE.
8680 '---------------------------------------------------------------------
8690 '-STROBE PIN HIGH, -ALF PIN HIGH, -INIT PIN LOW, -SELIN PIN LOW, IRQ OFF.
8700 IF MD2MOTOR = 1 OR MD2MOTOR = 2 THEN OUT MD2MTRADR12 + 2, &H4
8710 IF MD2MOTOR = 3 OR MD2MOTOR = 4 THEN OUT MD2MTRADR34 + 2, &H4
8720 IF MD2MOTOR = 5 OR MD2MOTOR = 6 THEN OUT MD2MTRADR56 + 2, &H4
8730 'DELAY FOR .2 SECONDS.
8740 MD2TIMER = TIMER: IF TIMER > MD2TIMER + .2 THEN GOTO 8740
8750 'TURN -INIT PIN HIGH.
8760 IF MD2MOTOR = 1 OR MD2MOTOR = 2 THEN OUT MD2MTRADR12 + 2, &HC
8770 IF MD2MOTOR = 3 OR MD2MOTOR = 4 THEN OUT MD2MTRADR34 + 2, &HC
8780 IF MD2MOTOR = 5 OR MD2MOTOR = 6 THEN OUT MD2MTRADR56 + 2, &HC
8790 RETURN

8800 '---------------------------------------------------------------------
8810 'NAME:      MD2HOME
8820 'DESC:      THE MD2HOME PROCEDURE IS USED TO MOVE THE STEPPER MOTOR
8830 '           TO A KNOWN HOME POSITION.  ALL OTHER MOVES ARE RELATIVE
8840 '           TO THIS HOME (ZERO) POSITION.  THE SELECTED MOTOR IS
8850 '           MOVED REVERSE UNTIL THE SWITCH IS ACTIVATED, THEN FORWARD
8860 '           UNTIL DEACTIVATED. THE CURRENT POSITION IS THEN SET TO
8870 '           ZERO - THIS IS THE HOME POSITION.
8880 '           ONLY 1 MOTOR (1-6) CAN BE HOME'D AT A TIME.
8890 'USAGE:     SET THE DESIRED MOTOR #, MOTOR PARAMETERS AND CALL.
8900 'INPUTS:    MOTOR # AND MOTOR PARAMETERS.
8910 'OUTPUTS:   CURRENT POSITION SET TO ZERO, MD2STATUS$.
8920 '---------------------------------------------------------------------
8930 'SET DEFAULT RETURN STATUS.
8340 MD2STATUS$ = "O"
8950 'IF BAD MOTOR # THEN BAIL OUT.
8960 IF MD2MOTOR < 1 OR MD2MOTOR > 6 THEN MD2STATUS$ = "B": RETURN
8970 'SET UP ADDRESS.
8980 IF MD2MOTOR = 1 OR MD2MOTOR = 2 THEN MD2MTRADR = MD2MTRADR12
8990 IF MD2MOTOR = 3 OR MD2MOTOR = 4 THEN MD2MTRADR = MD2MTRADR34
9000 IF MD2MOTOR = 5 OR MD2MOTOR = 6 THEN MD2MTRADR = MD2MTRADR56
9010 'SET UP PATTERN MASK, OTHER MOTOR'S MASK AND SWITCH MASK.
9020 IF MD2MOTOR=1 OR MD2MOTOR=3 OR MD2MOTOR=5 THEN MD2PMASK=&HF:MD2OMASK=&HF0:MD2SMASK=&H20
9030 IF MD2MOTOR=2 OR MD2MOTOR=4 OR MD2MOTOR=6 THEN MD2PMASK=&HF0:MD2OMASK=&HF:MD2SMASK=&H10
9040 '------------------------------------------
9050 'MOVE MOTOR REVERSE UNTIL SWITCH ACTIVATED.
9060 '------------------------------------------
9070 'LOOP UNTIL SWITCH IS ACTIVATED.
9080 IF (INP(MD2MTRADR + 1) AND MD2SMASK) = 0 THEN GOTO 9220
9090 'QUIT MOVING IF KEY PRESSED.
9100 IF INKEY$ <> "" THEN MD2STATUS$ = "K": GOTO 9220
9110 'POINT TO THE NEXT STEP PATTERN.
9120 MD2PATPTR(MD2MOTOR) = (MD2PATPTR(MD2MOTOR) - 1) AND &H7
9130 'GET STEP PATTERN AND MASK OFF UNNEEDED BITS.
9140 MD2PATTERN = MD2STPPAT(MD2PATPTR(MD2MOTOR)) AND MD2PMASK
9150 'DON'T DISTURB OTHER MOTORS BITS.
9160 MD2PATTERN = MD2PATTERN OR (INP(MD2MTRADR) AND MD2OMASK)
9170 'OUTPUT THE STEP PATTERN TO MOVE THE MOTOR.
9180 OUT MD2MTRADR, MD2PATTERN
9190 'DELAY BETWEEN STEPS.
9200 FOR MD2DELAY = 1 TO MD2SPEED(MD2MOTOR): NEXT MD2DELAY
9210 GOTO 9080
9220 '------------------------------------------
9230 'MOVE MOTOR FORWARD UNTIL SWITCH DEACTIVATED.
9240 '------------------------------------------
9250 'LOOP UNTIL SWITCH IS DEACTIVATED.
9260 IF (INP(MD2MTRADR + 1) AND MD2SMASK) <> 0 THEN GOTO 9400
9270 'QUIT MOVING IF KEY PRESSED.
9280 IF INKEY$ <> "" THEN MD2STATUS$ = "K": GOTO 9400
9290 'POINT TO THE NEXT STEP PATTERN.
9300 MD2PATPTR(MD2MOTOR) = (MD2PATPTR(MD2MOTOR) + 1) AND &H7
9310 'GET STEP PATTERN AND MASK OFF UNNEEDED BITS.
9320 MD2PATTERN = MD2STPPAT(MD2PATPTR(MD2MOTOR)) AND MD2PMASK
9330 'DON'T DISTURB OTHER MOTORS BITS.
9340 MD2PATTERN = MD2PATTERN OR (INP(MD2MTRADR) AND MD2OMASK)
9350 'OUTPUT THE STEP PATTERN TO MOVE THE MOTOR.
9360 OUT MD2MTRADR, MD2PATTERN
9370 'DELAY BETWEEN STEPS.
9380 FOR MD2DELAY = 1 TO MD2SPEED(MD2MOTOR): NEXT MD2DELAY
9390 GOTO 9260
9400 'SET POSITION TO 0.
9410 MD2POSITION(MD2MOTOR) = 0
9420 'POWER OFF MOTOR IF DESIRED.
9430 IF MD2HOLD = 0 THEN OUT MD2MTRADR, &HFF
9440 RETURN

9500 '---------------------------------------------------------------------
9510 'NAME:      MD2MOVE
9520 'DESC:      THIS PROCEDURE IS USED TO MOVE A MOTOR.
9530 '           THE HOME SWITCH IS IGNORED.  ANY KEYPRESS WILL STOP MOTION.
9540 'USAGE:     SET MOTOR #, MOTOR PARAMETERS AND CALL.
9550 'INPUTS:    MOTOR # (1,2,3,4,5 OR 6) AND MOTOR PARAMETERS.
9560 'OUTPUTS:   CURRENT POSITION, MD2STATUS$.
9570 '---------------------------------------------------------------------
9580 'SET DEFAULT RETURN STATUS.
9590 MD2STATUS$ = "O"
9600 'IF BAD MOTOR # THEN BAIL OUT.
9610 IF MD2MOTOR < 1 OR MD2MOTOR > 6 THEN MD2STATUS$ = "B": RETURN
9620 'SET UP ADDRESS.
9630 IF MD2MOTOR = 1 OR MD2MOTOR = 2 THEN MD2MTRADR = MD2MTRADR12
9640 IF MD2MOTOR = 3 OR MD2MOTOR = 4 THEN MD2MTRADR = MD2MTRADR34
9650 IF MD2MOTOR = 5 OR MD2MOTOR = 6 THEN MD2MTRADR = MD2MTRADR56
9660 'SET UP PATTERN MASK AND OTHER MOTOR'S MASK.
9670 IF MD2MOTOR=1 OR MD2MOTOR=3 OR MD2MOTOR=5 THEN MD2PMASK=&HF:MD2OMASK=&HF0
9680 IF MD2MOTOR=2 OR MD2MOTOR=4 OR MD2MOTOR=6 THEN MD2PMASK=&HF0:MD2OMASK=&HF
9690 'SET THE # OF STEPS.
9700 MD2STEPS = ABS(MD2TARGET(MD2MOTOR))
9710 'SET THE DIRECTION.
9720 IF MD2TARGET(MD2MOTOR) < 0 THEN MD2DIR = -1 ELSE MD2DIR = 1
9730 'SET THE FINAL MOTOR POSITION.
9740 MD2POSITION(MD2MOTOR) = MD2POSITION(MD2MOTOR) + MD2TARGET(MD2MOTOR)
9750 'MOVE LOOP.
9760 IF MD2STEPS = 0 THEN GOTO 9920
9770 'QUIT MOVING IF KEY PRESSED.
9780 IF INKEY$ <> "" THEN MD2STATUS$ = "K": GOTO 9920
9790 'POINT TO THE NEXT STEP PATTERN.
9800 MD2PATPTR(MD2MOTOR) = (MD2PATPTR(MD2MOTOR) + MD2DIR) AND &H7
9810 'GET STEP PATTERN AND MASK OFF UNNEEDED BITS.
9820 MD2PATTERN = MD2STPPAT(MD2PATPTR(MD2MOTOR)) AND MD2PMASK
9830 'DON'T DISTURB OTHER MOTORS BITS.
9840 MD2PATTERN = MD2PATTERN OR (INP(MD2MTRADR) AND MD2OMASK)
9850 'OUTPUT THE STEP PATTERN TO MOVE THE MOTOR.
9860 OUT MD2MTRADR, MD2PATTERN
9870 'DELAY BETWEEN STEPS.
9880 FOR MD2DELAY = 1 TO MD2SPEED(MD2MOTOR): NEXT MD2DELAY
9890 'DECREMENT STEP COUNT.
9900 MD2STEPS = MD2STEPS - 1
9910 GOTO 9760
9920 'UPDATE POSITION.
9930 MD2POSITION(MD2MOTOR) = MD2POSITION(MD2MOTOR) - (MD2STEPS * MD2DIR)
9940 'POWER OFF MOTOR IF DESIRED.
9950 IF MD2HOLD = 0 THEN OUT MD2MTRADR, &HFF
9960 RETURN
