(*  MD2PA1S.PAS  MD-2 Level 1 Pascal Subroutines with Example Program  *)
					
Program MD2PA1S;

Uses  CRT;   (* Need CRT for KeyPressed Statement *)

(*****************************************************************************
NAME:      MD2PA1S.PAS
DESC:      MD-2 LEVEL 1 MOTION CONTROL SUBROUTINES FOR PASCAL.
USAGE:     USE WHEN CREATING A CUSTOM MOTION CONTROL PROGRAM
		   THAT DOESN'T REQUIRE COMPLEX FEATURES SUCH AS RAMPING, ETC.
		   MOTOR PARAMETERS ARE PASSED VIA GLOBAL VARIABLES.
		   ALL VARIABLES AND LABELS BEGIN WITH 'MD2' TO AVOID CONFLICTS.
BY:        COPYRIGHT (C) 1994 ARRICK ROBOTICS, ROGER ARRICK.
DATE:      1/2/94
EDIT:      7

SUBROUTINES ---------------------------------------------------------

	NAME           DESCRIPTION
	----           -----------
	MD2Setup       Used at the beginning of a program to initialize
				   Motor parameters to their default values.
				   Use this subroutine beFore any other.
	MD2On          Turns on an MD-2 system.
	MD2Off         Turns off an MD-2 system.
	MD2Home        Moves a Motor to the Home position by watching the
				   home switch.  Current position is set to zero.
	MD2Move        Moves a Motor the number of steps in MD2Target[M]
				   parameter.

MOTOR PARAMETERS AND VARIABLES --------------------------------------

	NAME            TYPE    DESCRIPTION
	----            ----    -----------

	MD2Hold         Integer -1=Leaves the Motor energized after a move
							to cause the Motor to hold its position.
							0 causes the Motor to turn off after a move
							which conserves power and reduces heat.
	MD2Motor        Integer The selected Motor to act upon. 1,2,3,4,5 Or 6.
	MD2Speed[M]     Integer Delay count between steps. Determines the
							Motor's.  0=fast, 32766=slow.
							Actual Motor speed depends on the computer.
							M is the Motor number 1,2,3,4,5 Or 6.
	MD2Position[M]  Longint Current Motor position For each Motor in steps
							relative to home. Negative = reverse from home
							and positive = Forward from home.
							M is the Motor number 1,2,3,4,5 Or 6.
	MD2Status       Char    Completion status. O=motion completed OK,
							K=a keypress stopped the motion, B=Bad parameter.
	MD2Target[M]    Longint The number of steps to move. Positive #'s are
							Forward, negative are reverse.
							M is the Motor number 1,2,3,4,5 Or 6.

VARIABLES NEEDED BY SUBROUTINES BUT NOT BY PROGRAMMER ---------------

	NAME            TYPE    DESCRIPTION
	----            ----    -----------

	MD2Delay        Integer Used For delay counter.
	MD2Steps        Longint Used For step counter.
	MD2Dir          Integer Used as direction stOrage.
	MD2PMask        Integer Bit mask For selected Motor.
	MD2OMask        Integer Bit mask For other Motor.
	MD2SMask        Integer Bit mask For switch.
	MD2MtrAdr       Integer Selected Port Address.
	MD2MtrAdr12     Integer Port address For Motors 1 & 2.
	MD2MtrAdr34     Integer Port address For Motors 3 & 4.
	MD2MtrAdr56     Integer Port address For Motors 5 & 6.
	MD2StpPat[7]    Integer Step pattern array.
	MD2PatPtr[6]    Integer Pattern pointer For each Motor.
	MD2Pattern      Integer Selected step pattern.
****************************************************************************)

Var

	MD2Delay : Integer;                     (* DELAY COUNTER *)
	MD2Steps : LongInt;                     (* STEP COUNTER *)
	MD2Dir : Integer;                       (* DIRECTION *)
	MD2PMask : Integer;                     (* SELECTED MOTOR PATTERN BIT MASK *)
	MD2OMask : Integer;                     (* OTHER MOTOR PATTERN BIT MASK *)
	MD2SMask : Integer;                     (* SWITCH BIT MASK *)
	MD2MtrAdr : Integer;                    (* SELECTED PORT *)
	MD2MtrAdr12 : Integer;                  (* PORT FOR MOTORS 1 & 2 *)
	MD2MtrAdr34 : Integer;                  (* PORT FOR MOTORS 3 & 4 *)
	MD2MtrAdr56 : Integer;                  (* PORT FOR MOTORS 5 & 6 *)
	MD2StpPat : Array [0..7] of Integer;    (* STEP PATTERN ARRAY *)
	MD2PatPtr : Array [1..6] of Integer;    (* STEP PATTERN POINTERS *)
	MD2Pattern : Integer;                   (* SELECTED STEP PATTERN *)
	MD2Status : Char;                       (* COMPLETION STATUS *)
	MD2Hold : Integer;                      (* HOLD MOTORS WHEN DONE *)
	MD2Position: Array [1..6] of LongInt;   (* MOTOR POSITIONS *)
	MD2Motor : Integer;                     (* SELECTED MOTOR NUMBER *)
	MD2Speed : Array [1..6] of Integer;     (* MOTOR SPEED *)
	MD2Target: Array [1..6] of LongInt;     (* TARGET DISTANCE AND DIRECTION *)


Procedure MD2Setup;

	(*********************************************************************
	NAME:      MD2Setup
	DESC:      THE MD2Setup PROCEDURE SETS DEFAULT MD-2 SYSTEM
			   PARAMETERS SUCH AS MOTOR SPEED, CURRENT POSITION, ETC.
	USAGE:     USE AT THE BEGINNING OF A MOTION CONTROL PROGRAM.
			   MUST BE USED -BEFORE- ANY OTHER MOTION CONTROL SUBROUTINE!
	INPUTS:    NONE.
	OUTPUTS:   DEFAULT MOTOR PARAMETERS.
	**********************************************************************)

	Begin

		(* SETUP PORT ADDRESSES *)
		MD2MtrAdr12 := $03BC;
		MD2MtrAdr34 := $0378;
		MD2MtrAdr56 := $0278;

		(* SETUP MOTOR PARAMETER DEFAULTS *)
		For MD2Motor := 1 To 6 Do
		Begin
			MD2Position[MD2Motor] := 0;       (* POSITIONS = 0. *)
			MD2PatPtr[MD2Motor] := 0;         (* STEP PATTERN POINTERS. *)
			MD2Target[MD2Motor] := 0;         (* TARGET POSITION,STEPS. *)
			MD2Speed[MD2Motor] := 5000;       (* SPEED. *)
		End;
		MD2Motor := 1;                        (* MOTOR NUMBER. *)
		MD2Status := 'O';                     (* STATUS=OK. *)
		MD2Hold := 0;                         (* DON'T HOLD MOTORS. *)

		(* SET HALF STEP PHASE PATTERNS *)
		MD2StpPat[0] := $66;
		MD2StpPat[1] := $77;
		MD2StpPat[2] := $33;
		MD2StpPat[3] := $BB;
		MD2StpPat[4] := $99;
		MD2StpPat[5] := $DD;
		MD2StpPat[6] := $CC;
		MD2StpPat[7] := $EE;

	End;


Procedure MD2On;

	(*********************************************************************
	NAME:      MD2On
	DESC:      THE MD2On PROCEDURE INITIALIZES A PARALLEL PRINTER PORT
			   AND TURNS ON AN MD-2.
	USAGE:     USE AT THE BEGINNING OF A MOTION CONTROL PROGRAM BUT
			   AFTER THE MD2Setup SUBROUTINE.
	INPUTS:    MOTOR # DETERMINES PORT.
	OUTPUTS:   NONE.
	**********************************************************************)

	Begin

		If (MD2Motor = 1) Or (MD2Motor = 2) Then
		Begin
			Port[MD2MtrAdr12] := $FF;
			Port[MD2MtrAdr12 + 2] := $05;
		End;

		If (MD2Motor = 3) Or (MD2Motor = 4) Then
		Begin
			Port[MD2MtrAdr34] := $FF;
			Port[MD2MtrAdr34 + 2] := $05;
		End;

		If (MD2Motor = 5) Or (MD2Motor = 6) Then
		Begin
			Port[MD2MtrAdr56] := $FF;
			Port[MD2MtrAdr56 + 2] := $05;
		End;

	End;


Procedure MD2Off;

	(*********************************************************************
	NAME:      MD2Off
	DESC:      THE MD2Off PROCEDURE RETURNS A PARALLEL PRINTER PORT
			   REFERENCED BY THE MOTOR # TO ITS PREVIOUS STATE READY
			   For USE WITH A PRINTER AND DISABLES THE MD-2.
	USAGE:     USE AT THE END OF A MOTION CONTROL PROGRAM.
	INPUTS:    MOTOR # DETERMINES Port.
	OUTPUTS:   NONE.
	**********************************************************************)

	Begin

		(* -STROBE PIN HIGH, -ALF PIN HIGH,
		   -INIT PIN LOW, -SELIN PIN LOW, IRQ OFF.  *)
		If (MD2Motor = 1) Or (MD2Motor = 2) Then Port[MD2MtrAdr12 + 2] := $04;
		If (MD2Motor = 3) Or (MD2Motor = 4) Then Port[MD2MtrAdr34 + 2] := $04;
		If (MD2Motor = 5) Or (MD2Motor = 6) Then Port[MD2MtrAdr56 + 2] := $04;

		(* DELAY. *)
		For MD2Delay := 1 To 500 Do
		Begin
		End;

		(* TURN -INIT PIN HIGH. *)
		If (MD2Motor = 1) Or (MD2Motor = 2) Then Port[MD2MtrAdr12 + 2] := $0C;
		If (MD2Motor = 3) Or (MD2Motor = 4) Then Port[MD2MtrAdr34 + 2] := $0C;
		If (MD2Motor = 5) Or (MD2Motor = 6) Then Port[MD2MtrAdr56 + 2] := $0C;

	End;


Procedure MD2Home;

	(*********************************************************************
	NAME:      MD2Home
	DESC:      THE MD2Home PROCEDURE IS USED TO MOVE THE STEPPER MOTOR
			   TO A KNOWN HOME POSITION.  ALL OTHER MOVES ARE RELATIVE
			   TO THIS HOME (ZERO) POSITION.  THE SELECTED MOTOR IS
			   MOVED REVERSE UNTIL THE SWITCH IS ACTIVATED, THEN FORWARD
			   UNTIL DEACTIVATED. THE CURRENT POSITION IS THEN SET TO
			   ZERO - THIS IS THE HOME POSITION.
			   ONLY 1 MOTOR (1-6) CAN BE HOME'D AT A TIME.
	USAGE:     SET THE DESIRED MOTOR #, MOTOR PARAMETERS AND CALL.
	INPUTS:    MOTOR # AND MOTOR PARAMETERS.
	OUTPUTS:   CURRENT POSITION SET TO ZERO, MD2Status.
	*********************************************************************)

	Begin

		(* SET DEFAULT RETURN STATUS. *)
		MD2Status := 'O';

		(* IF BAD MOTOR # THEN BAIL OUT. *)
		If (MD2Motor < 1) Or (MD2Motor > 6) Then
		Begin
			MD2Status := 'B';
			Exit;
		End;

		(* SET UP ADDRESS. *)
		If (MD2Motor = 1) Or (MD2Motor = 2) Then MD2MtrAdr := MD2MtrAdr12;
		If (MD2Motor = 3) Or (MD2Motor = 4) Then MD2MtrAdr := MD2MtrAdr34;
		If (MD2Motor = 5) Or (MD2Motor = 6) Then MD2MtrAdr := MD2MtrAdr56;

		(* SET UP PATTERN MASK, OTHER MOTOR'S MASK AND SWITCH MASK. *)
		If (MD2Motor = 1) Or (MD2Motor = 3) Or (MD2Motor = 5) Then
		Begin
			MD2PMask := $0F;
			MD2OMask := $F0;
			MD2SMask := $20;
		End
		Else
		Begin
			MD2PMask := $F0;
			MD2OMask := $0F;
			MD2SMask := $10;
		End;

		(* ------------------------------------------ *)
		(* MOVE MOTOR REVERSE UNTIL SWITCH ACTIVATED. *)
		(* ------------------------------------------ *)

		(* LOOP UNTIL SWITCH IS ACTIVATED. *)
		While (Port[MD2MtrAdr + 1] And MD2SMask) <> 0 Do

		Begin

			(* QUIT MOVING IF KEY PRESSED. *)
			If KeyPressed Then
			Begin
				MD2Status := ReadKey;
				MD2Status := 'K';
				Exit;
			End;

			(* POINT TO THE NEXT STEP PATTERN. *)
			MD2PatPtr[MD2Motor] := (MD2PatPtr[MD2Motor] - 1) AND $7;

			(* GET STEP PATTERN AND MASK OFF UNNEEDED BITS. *)
			MD2Pattern := MD2StpPat[MD2PatPtr[MD2Motor]] And MD2PMask;

			(* DON'T DISTURB OTHER MOTORS BITS. *)
			MD2Pattern := MD2Pattern Or (Port[MD2MtrAdr] And MD2OMask);

			(* OUTPUT THE STEP PATTERN TO MOVE THE MOTOR. *)
			Port[MD2MtrAdr] := MD2Pattern;

			(* DELAY BETWEEN STEPS. *)
			For MD2Delay := 1 To MD2Speed[MD2Motor] Do
			Begin
			End;

		End;

		(* --------------------------------------------  *)
		(* MOVE MOTOR FORWARD UNTIL SWITCH DEACTIVATED.  *)
		(* --------------------------------------------  *)

		(* LOOP UNTIL SWITCH IS DEACTIVATED. *)
		While (Port[MD2MtrAdr + 1] And MD2SMask) = 0 Do

		Begin

			(* QUIT MOVING IF KEY PRESSED. *)
			If KeyPressed Then
			Begin
				MD2Status := ReadKey;
				MD2Status := 'K';
				Exit;
			End;

			(* POINT TO THE NEXT STEP PATTERN. *)
			MD2PatPtr[MD2Motor] := (MD2PatPtr[MD2Motor] + 1) AND $7;

			(* GET STEP PATTERN AND MASK OFF UNNEEDED BITS. *)
			MD2Pattern := MD2StpPat[MD2PatPtr[MD2Motor]] AND MD2PMask;

			(* DON'T DISTURB OTHER MOTORS BITS. *)
			MD2Pattern := MD2Pattern Or (Port[MD2MtrAdr] AND MD2OMask);

			(* OUTPUT THE STEP PATTERN TO MOVE THE MOTOR. *)
			Port[MD2MtrAdr] := MD2Pattern;

			(* DELAY BETWEEN STEPS. *)
			For MD2Delay := 1 To MD2Speed[MD2Motor] Do
			Begin
			End;

		End;

		(* SET POSITION TO 0. *)
		MD2Position[Md2Motor] := 0;

		(* POWER OFF MOTOR IF DESIRED. *)
		If MD2Hold = 0 Then Port[MD2MtrAdr] := $FF;

	End;


Procedure MD2Move;

	(********************************************************************
	NAME:      MD2Move
	DESC:      THIS PROCEDURE IS USED TO MOVE A Motor.
			   THE HOME SWITCH IS IGNOrED.  ANY KEYPRESS WILL STOP MOTION.
	USAGE:     SET Motor #, Motor PARAMETERS AND CALL.
	INPUTS:    Motor # (1,2,3,4,5 Or 6) AND Motor PARAMETERS.
	OUTPUTS:   CURRENT POSITION, MD2Status.
	*********************************************************************)

	Begin

		(* SET DEFAULT RETURN STATUS *)
		MD2Status := 'O';

		(* IF BAD MOTOR # THEN BAIL OUT. *)
		If (MD2Motor < 1) Or (MD2Motor > 6) Then
		Begin
			MD2Status := 'B';
			Exit;
		End;

		(* SET UP ADDRESS. *)
		If (MD2Motor = 1) Or (MD2Motor = 2) Then MD2MtrAdr := MD2MtrAdr12;
		If (MD2Motor = 3) Or (MD2Motor = 4) Then MD2MtrAdr := MD2MtrAdr34;
		If (MD2Motor = 5) Or (MD2Motor = 6) Then MD2MtrAdr := MD2MtrAdr56;

		(* SET UP PATTERN MASK AND OTHER MOTOR'S MASK. *)
		If (MD2Motor = 1) Or (MD2Motor = 3) Or (MD2Motor = 5) Then
		Begin
			MD2PMask := $0F;
			MD2OMask := $F0;
		End
		Else
		Begin
			MD2PMask := $F0;
			MD2OMask := $0F;
		End;

		(* SET THE # OF STEPS. *)
		MD2Steps := ABS(MD2Target[MD2Motor]);

		(* SET THE DIRECTION. *)
		If MD2Target[MD2Motor] < 0 Then MD2Dir := -1 Else MD2Dir := 1;

		(* SET THE FINAL MOTOR POSITION. *)
		MD2Position[Md2Motor] := MD2Position[Md2Motor] + MD2Target[MD2Motor];

		(* MOVE LOOP. *)
		While MD2Steps <> 0 Do

		Begin

			(* QUIT MOVING IF KEY PRESSED. *)
			If KeyPressed Then
			Begin
				MD2Status := ReadKey;
				MD2Status := 'K';
				Exit;
			End;

			(* POINT TO THE NEXT STEP PATTERN. *)
			MD2PatPtr[MD2Motor] := (MD2PatPtr[MD2Motor] + MD2Dir) And $7;

			(* GET STEP PATTERN AND MASK OFF UNNEEDED BITS. *)
			MD2Pattern := MD2StpPat[MD2PatPtr[MD2Motor]] And MD2PMask;

			(* DON'T DISTURB OTHER MOTORS BITS. *)
			MD2Pattern := MD2Pattern Or (Port[MD2MtrAdr] And MD2OMask);

			(* OUTPUT THE STEP PATTERN TO MOVE THE MOTOR. *)
			Port[MD2MtrAdr] := MD2Pattern;

			(* DELAY BETWEEN STEPS. *)
			For MD2Delay := 1 To MD2Speed[MD2Motor] Do
			Begin
			End;

			(* DECREMENT STEP COUNT. *)
			MD2Steps := MD2Steps - 1;

		End;

		(* UPDATE POSITION. *)
		MD2Position[Md2Motor] := MD2Position[Md2Motor] - (MD2Steps * MD2Dir);

		(* POWER OFF MOTOR IF DESIRED. *)
		If MD2Hold = 0 Then Port[MD2MtrAdr] := $FF;

	End;

(**************** End of MD2PA1S.PAS ******************)
					

(*                Small Example Program                     *)
(*  It may be necessary to change motor numbers and speeds  *)
(*  for this example program to work on your computer       *)

Begin

	Writeln('MD2PA1E1.PAS  Example Program ');

	(*  Setup MD-2 Variables  *)
	MD2Setup;

	(*  Turn on MD-2  *)
	MD2Motor := 3;
	MD2On;

	(*  Home motor 3  *)
	Writeln('Homing motor 3 ');
	MD2Motor := 3;
	MD2Speed[3] := 6000;
	MD2Hold := -1;
	MD2Home;

	(*  Home motor 4  *)
	Writeln('Homing motor 4 ');
	MD2Motor := 4;
	MD2Speed[4] := 7000;
	MD2Hold := -1;
	MD2Home;

	(*  Move motor 3  *)
	Writeln('Moving motor 3 400 steps forward ');
	MD2Motor := 3;
	MD2Target[3] := 400;
	MD2Speed[3] := 2000;
	MD2Move;

	(*  Move motor 4  *)
	Writeln('Moving motor 4 200 steps forward ');
	MD2Motor := 4;
	MD2Target[4] := 200;
	MD2Speed[4] := 3000;
	MD2Move;

	(*  Turn Off MD-2  *)
	MD2Off;

End.
