'---------------------------------------------------------------------
'NAME:      MD2QB2S.BAS
'DESC:      MD-2 LEVEL 2 QUICK-BASIC MOTION CONTROL SUBROUTINES
'           WITHOUT MD2PAREDIT FOR SPACE SAVINGS.
'USAGE:     USE WHEN CREATING COMPLEX MOTION CONTROL PROGRAMS.
'           USE LEVEL 1 LIBRARY WHEN CREATING SIMPLE PROGRAMS.
'
'           FEATURES:
'               DUAL MOTOR MOVES.
'               RELATIVE/ABSOLUTE MOVES.
'               ADJUSTABLE ACCELERATION/DECELERATION.
'               2-AXIS LINEAR INTERPOLATION.
'               2-AXIS CIRCULAR INTERPOLATION.
'               ARC AND ELLIPSE MOVES.
'               2-AXIS GRID MOVES.
'               USER DEFINED UNITS.
'               BACKLASH COMPENSATION.
'               SOFT LIMITS.
'               SPEED GIVEN IN UNITS PER SECOND.
'               PARAMETER EDITOR WITH SAVE & LOAD.
'               MOTION SEQUENCE PROGRAM INTERPRETER.
'               INPUT/OUTPUT PORT CONTROL.
'
'           ALL MOTOR PARAMETERS ARE PASSED TO SUBROUTINES VIA
'           GLOBAL VARIABLES.  ALL VARIABLES AND SUBS BEGIN WITH
'           MD2 TO AVOID CONFLICTS.
'           SEE .TXT FILE FOR COMPLETE DESCRIPTION.
'BY:        COPYRIGHT (C) 1994 ARRICK ROBOTICS, ROGER ARRICK.
'DATE:      5/14/95
'EDIT:      162
'
'PUBLIC SUBROUTINES ---------------------------------------------------
'
'    NAME           DESCRIPTION
'    ----           -----------
'    MD2SETUP       Used at the beginning of a program to initialize
'                   motor parameters to their default values and checks
'                   for parallel port availability. Loads calibration file.
'                   Use this SUB before any other.
'    MD2CALIBRATE   Calibrates motor speeds with computer speeds and saves
'                   to disk.  Use this SUB once on your cpu before moving.
'    MD2ON          Turns on an MD-2 system. Use before any moves.
'    MD2OFF         Turns off an MD-2 system. Use at end of program.
'    MD2HOME        Moves a motor to the Home position by watching the
'                   home switch.  Current position is set to zero.
'    MD2MOVE        Moves a motor to the desired target position using
'                   the various motor parameters.  Also moves two motors
'                   together providing linear interpolation.
'    MD2CIRCLE      Performs 2-axis circular, arc and ellipise moves.
'    MD2GRID        Moves 2 motors using grid coordinates.
'    MD2OUTPUTS     Turns on and off output pins to control tools.
'    MD2INPUTS      Reads condition of input pins and home switches.
'    MD2STANDBYON   Puts MD-2 into standby mode which reduces motor current
'                   and heat while maintaining some holding torque.
'    MD2STANDBYOFF  Turns off standby mode giving full current to motors.
'    MD2PARLOAD     Loads motor parameters from a disk file.
'    MD2PARSAVE     Saves motor parameters to a disk file.
'    MD2SEQLOAD     Loads a sequence of motion commands from a disk file.
'    MD2SEQSAVE     Saves a sequence of motion commands to a disk file.
'    MD2SEQRUN      Runs a sequence of motion commands.
'
'
'PUBLIC MOTOR PARAMETERS AND VARIABLES ----------------------------------
'
'    (M) indicates motor # 1,2,3,4,5 or 6.
'
'    NAME            TYPE    DESCRIPTION
'    ----            ----    -----------
'
'    MD2BACKLASH(M)  SINGLE  Backlash compensation in units.  Adds this
'                            amount to every move that changes direction.
'                            Uses MD2LASTDIR(M).
'                            Value is in Units.
'    MD2ENABLED12    INTEGER Port 3BC for motors 1 & 2 has been enabled
'                            by the MD2ON sub. 0=not enabled, -1=available.
'    MD2ENABLED34    INTEGER Port 3BC for motors 3 & 4 has been enabled
'                            by the MD2ON sub. 0=not enabled, -1=available.
'    MD2ENABLED56    INTEGER Port 3BC for motors 5 & 6 has been enabled
'                            by the MD2ON sub. 0=not enabled, -1=available.
'    MD2HOLD         INTEGER -1=Leaves the motor energized after a move
'                            to cause the motor to hold its position.
'                            0 causes the motor to turn off after a move
'                            which conserves power and reduces heat.
'    MD2HOMEOFFSET(M)SINGLE  Distance from home switch to call home.
'                            Value is in units.
'    MD2HOMEDIR(M)   INTEGER Direction to home switch, 0=rev, -1=fwd.
'                            Reverses meaning of forward and reverse
'                            target directions.  All directions are reversed.
'    MD2INTERRUPTS   INTEGER 0=turns COM port interrupts off during moves
'                            which prevents mouse & modem interference.
'                            This can be important under WINDOWS.
'                            -1 leaves interrupts enabled.
'    MD2LIMITF(M)    SINGLE  Forward soft limit in units.  Prevents moves
'                            beyond this value in the forward direction.
'                            Value is in Units.
'    MD2LIMITR(M)    SINGLE  Reverse soft limit in units.  Prevents moves
'                            beyond this value in the reverse direction
'                            Value is in Units.
'    MD2MOTOR        INTEGER The selected motor to act upon. 1,2,3,4,5 or 6.
'                            Set to 12, 34, 56 for dual motor moves.
'    MD2MOTORNAME(M) STRING  User definable motor name.
'    MD2MAXSPEED(M)  SINGLE  Motor fastest speed after accelerating.
'                            In Units per Second.
'    MD2MINSPEED(M)  SINGLE  Motor speed at the beginning and end of ramping.
'                            In Units per Second.
'    MD2MOVETYPE     STRING  Tells the MD2MOVE sub what type of move.
'                            R=relative, A=absolute. Affects all motors.
'                            With relative moves, the MD2TARGET parameter
'                            tells how many steps to move from the current
'                            position. Negative numbers are reverse, positive
'                            numbers are forward.
'                            With absolute moves, the MD2TARGET parameter
'                            tells what position to move to.  The number of
'                            steps and direction are determined by the
'                            current position. Negative numbers are reverse
'                            from home and positive numbers are forward.
'                            negative then reverse, positive then forward.
'    MD2AVAILABLE12  INTEGER Port 3BC availability for motors 1 & 2 set by
'                            the MD2SETUP sub. 0=not available, -1=available.
'    MD2AVAILABLE34  INTEGER Port 378 availability for motors 3 & 4 set by
'                            the MD2SETUP sub. 0=not available, -1=available.
'    MD2AVAILABLE56  INTEGER Port 278 availability for motors 5 & 6 set by
'                            the MD2SETUP sub. 0=not available, -1=available.
'    MD2POSITION(M)  SINGLE  Current motor position for each motor (M=motor #)
'                            relative to home. Negative = reverse from home
'                            and positive = forward from home.
'                            Value is in units.
'    MD2SLOPE(M)     SINGLE  The slope of the ramp (acceleration/deceleration)
'                            in number Units.  A 2 would mean that the
'                            motor would accelerate from the MD2MINSPEED to
'                            the MD2MAXSPEED in 2 units.
'    MD2STATUS       STRING  Completion status.
'                            O=motion completed OK,
'                            K=a keypress stopped the motion,
'                            B=bad parameter,
'                            L=soft limit was exceeded.
'                            E=MD-2 not enabled.
'                            P=Port not available.
'                            F=file error.
'    MD2STEPTYPE     STRING  H=Half step, D=Double phase full step,
'                            S=Single phase full step.
'                            Affects all motors.
'    MD2TARGET(M)    SINGLE  The number of units to move in relative moves,
'                            the position to move to in absolute moves.
'                            Value is in units which is converted to steps.
'    MD2UNITS(M)     LONG    Converts user definable units to steps
'                            for MD2MOVE subroutine.
'                            This is number of steps in each unit.
'    MD2UNITNAME(M)  STRING  This is the name of the units such as
'                            steps, inches, degrees, etc.
'
'PUBLIC INPUT/OUTPUT PORT PARAMETERS----------------------------------
'
'    MD2OUTPUTCODE   INTEGER Tells the MD2OUTPUTS sub which MD-2, which
'                            output pin and on/off action.
'    MD2INPUT(CODE)  INTEGER Contains status of input pins and home switches
'                            after using MD2INPUTS sub.
'
'PUBLIC PARAMETER & SEQUENCE FILE VARIABLES---------------------------
'
'    MD2COMMAND      STRING  Single command to execute.
'    MD2SEQUENCE     STRING  Contains a motion sequence.
'    MD2SEQFILE      STRING  Sequence file name.
'    MD2PARFILE      STRING  Parameter file name.
'    MD2POINTER      STRING  Sequence file pointer.
'
'PUBLIC GRID MOVE PARAMETERS-------------------------------------------
'
'    MD2GRIDBEGINX   SINGLE  Grid X start position in units.
'    MD2GRIDBEGINY   SINGLE  Grid Y start position in units.
'    MD2GRIDSPACEX   SINGLE  Grid X spacing in units.
'    MD2GRIDSPACEY   SINGLE  Grid Y spacing in units.
'    MD2GRIDTARGETX  INTEGER Grid X target.
'    MD2GRIDTARGETY  INTEGER Grid Y target.
'
'PUBLIC CIRCLE/ARC MOVE PARAMETERS------------------------------------
'
'    MD2CIRCLERADIUSX   SINGLE  Circle X radius in units.
'    MD2CIRCLERADIUSY   SINGLE  Circle Y radius in units.
'    MD2CIRCLECENTERX   SINGLE  Circle X center in units.
'    MD2CIRCLECENTERY   SINGLE  Circle Y center in units.
'    MD2CIRCLESTART     INTEGER Circle start angle in degrees.
'    MD2CIRCLEARC       INTEGER Circle arc angle in degrees. -=CW,+=CCW.
'    MD2CIRCLECHORD     INTEGER Circle chord angle in degrees.
'
'PUBLIC MOTOR SPEED CALIBRATION PARAMETERS--------------------------------
'
'    MD2VELOCITY(24)    SINGLE  Fastest possible motor speed in steps/sec.
'    MD2DELAY(24)       SINGLE  Slowest possible motor speed in steps/sec.
'    MD2CALFILE         STRING  Calibration file name.
'    MD2CALIBRATED      INTEGER Calibrated status, 0=false, -1=true.
'
'---------------------------------------------------------------------

'THIS INCLUDE FILE CONTAINS PROCEDURE DECLARATIONS AND GLOBAL VARIABLES.
'$INCLUDE: 'MD2QB2S.BI'

'THIS IS NEEDED FOR INTERRUPT CALLS FOR KEYBOARD CHECKING
'BECAUSE YOU CAN'T USE INKEY$ WHILE FORMS ARE SHOWING.
TYPE RegType
	 AX    AS INTEGER
	 bx    AS INTEGER
	 cx    AS INTEGER
	 dx    AS INTEGER
	 bp    AS INTEGER
	 si    AS INTEGER
	 di    AS INTEGER
	 flags AS INTEGER
END TYPE
DIM SHARED INREG AS RegType
DIM SHARED OUTREG AS RegType
DECLARE SUB INTERRUPT (intnum AS INTEGER, INREG AS RegType, OUTREG AS RegType)

'PRIVATE VARIABLES USED ONLY BY SUBS IN THIS MODULE-------------------
DIM SHARED MD2MTRADR12 AS INTEGER       'PORT FOR MOTORS 1 & 2.
DIM SHARED MD2MTRADR34 AS INTEGER       'PORT FOR MOTORS 3 & 4.
DIM SHARED MD2MTRADR56 AS INTEGER       'PORT FOR MOTORS 5 & 6.
DIM SHARED MD2STPPAT(7) AS INTEGER      'STEP PATTERN ARRAY.
DIM SHARED MD2PATPTR(6) AS INTEGER      'STEP PATTERN POINTERS.
DIM SHARED MD2LASTDIR(6) AS INTEGER     'LAST DIRECTION -1=REV,1=FWD.
DIM SHARED MD2CALMODE AS INTEGER        'CALIBRATION MODE. 0=OFF,-1=ON.


'ERROR HANDLER USED FOR ALL FILE I/O SUBROUTINES.
'
MD2ERROR:
	MD2STATUS = "F"
	RESUME NEXT

SUB MD2CALIBRATE

	'---------------------------------------------------------------------
	'NAME:      MD2CALIBRATE
	'DESC:      THIS PROCEDURE CALIBRATES MOTOR SPEEDS ON THIS PARTICULAR
	'           COMPUTER.  THE CALIBRATION INFORMATION IS STORED IN THE
	'           ARRAYS MD2VELOCITY(24) AND MD2DELAY(24) THEN SAVED TO DISK
	'           IN THE FILE NAME STORED IN MD2CALFILE.  CALIBRATION
	'           INFORMATION IS READ BY THE MD2SETUP SUB AT THE BEGINNING
	'           OF EACH MOTION PROGRAM AND IS USED WHEN MOVING MOTORS.
	'           THIS IS TRANSPARENT TO THE USER AND PROGRAMMER.
	'           WITHOUT CALIBRATION, MOTOR SPEEDS WOULD VARY DEPENDING ON
	'           COMPUTER SPEED AND OTHER FACTORS.
	'           IT IS ONLY NECESSARY TO CALIBRATE ON A PARTICULAR COMPUTER
	'           ONCE.  MD2SETUP MUST BE USED FIRST.
	'           MD2VELOCITY(1)=FASTEST POSSIBLE SPEED IN STEPS PER SECOND.
	'           MD2VELOCITY(24)=SLOWEST POSSIBLE SPEED IN STEPS PER SECOND
	'USAGE:     SIMPLY CALL AND WAIT SEVERAL MINUITES.
	'INPUTS:    MD2CALFILE = CALIBRATION FILE NAME USUALLY "MD2.CAL"
	'OUTPUTS:   FILE CONTAINING CALIBRATION PARAMETERS,ARRAYS,MD2STATUS.
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	DIM MD2ELEMENT AS INTEGER       'ELEMENT COUNTER.
	DIM MD2STIME AS SINGLE          'BEGINNING TIME.
	DIM MD2ETIME AS SINGLE          'ENDING TIME.
	DIM MD2DELAY AS LONG            'DELAY.
	DIM MD2FILENUM AS INTEGER       'FREE FILE NUMBER.
	DIM MD2TMPMOTOR AS INTEGER      'TEMP VARIABLE STORAGE.
	DIM MD2TMPMOVETYPE AS STRING
	DIM MD2TMPSLOPE AS SINGLE
	DIM MD2TMPBACKLASH AS SINGLE
	DIM MD2TMPUNITS AS LONG
	DIM MD2TMPLIMITF AS SINGLE
	DIM MD2TMPLIMITR AS SINGLE
	DIM MD2TMPTARGET AS SINGLE
	DIM MD2TMPPOSITION AS SINGLE
					  
	'CHECK FOR SETUP.
	IF MD2MTRADR12 <> &H3BC THEN MD2STATUS = "S": EXIT SUB
	
	'SAVE PARAMETERS.
	MD2TMPMOTOR = MD2MOTOR
	MD2TMPMOVETYPE = MD2MOVETYPE
	MD2TMPSLOPE = MD2SLOPE(1)
	MD2TMPBACKLASH = MD2BACKLASH(1)
	MD2TMPUNITS = MD2UNITS(1)
	MD2TMPLIMITF = MD2LIMITF(1)
	MD2TMPLIMITR = MD2LIMITR(1)
	MD2TMPTARGET = MD2TARGET(1)
	MD2TMPPOSITION = MD2POSITION(1)
   
	'INITIALIZE MOTOR PARAMETERS AND VARIABLES.
	MD2MOTOR = 1
	MD2MOVETYPE = "R"
	MD2SLOPE(1) = 0
	MD2BACKLASH(1) = 0
	MD2UNITS(1) = 1
	MD2LIMITF(1) = 2000000000
	MD2LIMITR(1) = -2000000000
	MD2DELAY = 1
	MD2CALMODE = -1
	MD2STATUS = "O"
	MD2OFF

	'FOR EACH DELAY (1,2,4,8,ETC), FIND VELOCITY AND PLACE IN ARRAY.
	FOR MD2ELEMENT = 1 TO 24
	   
		'PRINT ".";

		'INITIALIZE VARIABLES.
		MD2TARGET(1) = 1
		MD2MAXSPEED(1) = MD2DELAY
		MD2DELAY(MD2ELEMENT) = MD2DELAY

MD2CALOOP:
	   
		'TIME THE MOVE TO DETERMINE SPEED.
		MD2STIME = TIMER: DO: LOOP UNTIL MD2STIME <> TIMER: MD2STIME = TIMER
		MD2MOVE
		MD2ETIME = TIMER

		'IF MOVE ERROR THEN BAIL OUT.
		IF MD2STATUS <> "O" THEN EXIT FOR

		'TRY AGAIN IF NOT 10 SECONDS OR MORE OF TEST.
		IF (MD2ETIME - MD2STIME) <= 10 THEN
			MD2TARGET(1) = MD2TARGET(1) + MD2TARGET(1)
			GOTO MD2CALOOP
		END IF

		'PUT VELOCITY IN ARRAY.
		MD2VELOCITY(MD2ELEMENT) = MD2TARGET(1) / (MD2ETIME - MD2STIME)
	   
		'QUIT IF LAST MEASUREMENT WAS SLOWER THAN .01SPS.
		IF MD2VELOCITY(MD2ELEMENT) <= .01 THEN
			'FILL REMAINING ELEMENTS WITH LAST ENTRY.
			DO
				MD2DELAY(MD2ELEMENT) = MD2DELAY
				MD2VELOCITY(MD2ELEMENT) = MD2TARGET(1) / (MD2ETIME - MD2STIME)
				MD2ELEMENT = MD2ELEMENT + 1
			LOOP UNTIL MD2ELEMENT = 25
			EXIT FOR
		END IF
	   
		'DO NEXT DELAY.
		MD2DELAY = MD2DELAY + MD2DELAY

	NEXT MD2ELEMENT
   
	MD2CALMODE = 0
  
	IF MD2STATUS = "O" THEN
		'SAVE CALIBRATION PARAMETERS IF NO FAILURE.
		ON ERROR GOTO MD2ERROR
		MD2FILENUM = FREEFILE
		OPEN MD2CALFILE FOR OUTPUT AS MD2FILENUM
		FOR MD2ELEMENT = 1 TO 24
			IF MD2STATUS = "F" THEN EXIT FOR
			PRINT #MD2FILENUM, MD2DELAY(MD2ELEMENT); ","; MD2VELOCITY(MD2ELEMENT)
		NEXT MD2ELEMENT
		CLOSE MD2FILENUM
		ON ERROR GOTO 0
		MD2CALIBRATED = -1
	ELSE
		MD2CALIBRATED = 0
	END IF

	'RESTORE PARAMETERS.
	MD2MOVETYPE = MD2TMPMOVETYPE
	MD2MOTOR = MD2TMPMOTOR
	MD2SLOPE(1) = MD2TMPSLOPE
	MD2BACKLASH(1) = MD2TMPBACKLASH
	MD2UNITS(1) = MD2TMPUNITS
	MD2LIMITF(1) = MD2TMPLIMITF
	MD2LIMITR(1) = MD2TMPLIMITR
	MD2TARGET(1) = MD2TMPTARGET
	MD2POSITION(1) = MD2TMPPOSITION
				  
END SUB

SUB MD2CIRCLE

	'---------------------------------------------------------------------
	'NAME:      MD2CIRCLE
	'DESC:      THE MD2CIRCLE WILL MOVE 2 MOTORS WHICH ARE ATTACHED TO AN
	'           XY POSITIONING TABLE IN A CIRCULAR, ARC OR ELLIPSE PATTERN.
	'           THE 2 MOTORS MUST BE 1 & 2, 3 & 4 OR 5 & 6.
	'           THE USER CAN SET THE CHORD ANGLE IN DEGREES WHICH DETERMINES
	'           THE RESOLUTION OF THE CIRCLE.  A CHORD ANGLE OF 45 WILL
	'           PRODUCE AN OCTAGON.  A CHORD ANGLE OF 1 DEGREE WILL PRODUCE
	'           A CIRCLE WITH 360 SIDES.
	'           VARIOUS ARC ANGLES CAN ALSO BE GIVEN WHICH WILL PRODUCE
	'           PARTS OF A CIRCLE. AN ARC ANGLE OF 90 WILL PRODUCE AN ARC
	'           OF 90 DEGREES.  360 WILL PRODUCE A COMPLETE CIRCLE.
	'           X & Y RADIUS VALUES ARE PROVIDED TO PRODUCE AN ELLIPSE.
	'           IF BOTH X & Y RADIUS VALUES ARE THE SAME, A CIRCLE WILL
	'           RESULT.
	'           POSITIVE ARC ANGLES WILL PRODUCE COUNTER-CLOCKWISE ROTATION,
	'           NEGATIVE VALUES CLOCKWISE.
	'           MOVES ARE PERFORMED AT THE MINIMUM SPEED WITH NO RAMPING.
	'           TO MAKE A CIRCLE OR ARC AT AN ABSOLUTE POSITION, SET THE
	'           CENTERX,Y PARAMETERS TO THE ABSOLUTE CENTER POSITIONS AND
	'           SET MD2MOVETYPE TO "A".
	'           TO MAKE A CIRCLE OR ARC RELATIVE TO THE CURRENT POSITION,
	'           SET CENTERX,Y TO OFFSETS FROM CURRENT MOTOR POSITIONS AND
	'           SET MD2MOVETYPE TO "R".
	'           RAMPING IS DISABLED.  MOTORS WILL MOVE AT MINIMUM SPEED.
	'
	'              PARAMETER            DESCRIPTION
	'           ----------------    -----------------------
	'           MD2CIRCLERADIUSX    X-AXIS RADIUS IN UNITS.
	'           MD2CIRCLERADIUSY    Y-AXIS RADIUS IN UNITS.
	'           MD2CIRCLECENTERX    X-AXIS CENTER IN UNITS, REL OR ABS.
	'           MD2CIRCLECENTERY    Y-AXIS CENTER IN UNITS, REL OR ABS.
	'           MD2CIRCLESTART      STARTING ANGLE IN DEGREES.
	'           MD2CIRCLEARC        ARC ANGLE IN DEGREES. 360=CIRCLE.
	'                               +=CCW, -=CW ROTATION DIRECTION.
	'           MD2CIRCLECHORD      CHORD ANGLE IN DEGREES.
	'           MD2MOTOR            12, 34 OR 45.
	'           MD2MOVETYPE         "A"=ABSOLUTE, "R"=RELATIVE.
	'
	'USAGE:     SET CIRCLE PARAMETERS AND MOTOR PARAMETERS THEN CALL.
	'INPUTS:    ALL MOTOR PARAMETERS, CIRCLE PARAMETERS.
	'OUTPUTS:   MD2STATUS, MD2TARGET(M), MD2POSITION(M)
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	DIM MD2ENDANGLE AS INTEGER      'END ANGLE.
	DIM MD2ANGLE AS INTEGER         'CURRENT ANGLE.
	DIM MD2ANGLER AS SINGLE         'CURRENT ANGLE IN RADIANS.
	DIM MD2MOTOR1 AS INTEGER        '1ST MOTOR #.
	DIM MD2MOTOR2 AS INTEGER        '2ND MOTOR #.
	DIM MD2TMPMOVETYPE AS STRING    'SAVE MOVETYPE
	DIM MD2TMPSLOPE1 AS SINGLE      'SAVE SLOPE 1.
	DIM MD2TMPSLOPE2 AS SINGLE      'SAVE SLOPE 2.
	DIM MD2TMPSPEED1 AS SINGLE      'SAVE MAX SPEED 1.
	DIM MD2TMPSPEED2 AS SINGLE      'SAVE MAX SPEED 2.
	DIM MD2TMPHOLD AS INTEGER       'SAVE HOLD.
	DIM MD2CNTRX AS SINGLE          'CENTER X.
	DIM MD2CNTRY AS SINGLE          'CENTER Y.

	'CHECK FOR BAD PARAMETERS.
	MD2STATUS = "B"
	IF MD2MOTOR <> 12 AND MD2MOTOR <> 34 AND MD2MOTOR <> 56 THEN EXIT SUB
	IF MD2CIRCLEARC = 0 OR ABS(MD2CIRCLEARC) > 360 THEN EXIT SUB
	IF MD2CIRCLECHORD <= 0 OR MD2CIRCLECHORD > 180 THEN EXIT SUB
	IF MD2CIRCLESTART > 360 OR MD2CIRCLESTART < 0 THEN EXIT SUB
	MD2MOVETYPE = UCASE$(MD2MOVETYPE)
	IF MD2MOVETYPE <> "A" AND MD2MOVETYPE <> "R" THEN EXIT SUB
	MD2STATUS = "O"

	'INITIALIZE VARIABLES.
	IF MD2MOTOR = 12 THEN MD2MOTOR1 = 1: MD2MOTOR2 = 2
	IF MD2MOTOR = 34 THEN MD2MOTOR1 = 3: MD2MOTOR2 = 4
	IF MD2MOTOR = 56 THEN MD2MOTOR1 = 5: MD2MOTOR2 = 6
   
	'SAVE PARAMETERS.
	MD2TMPMOVETYPE = MD2MOVETYPE
	MD2TMPSLOPE1 = MD2SLOPE(MD2MOTOR1)
	MD2TMPSLOPE2 = MD2SLOPE(MD2MOTOR2)
	MD2TMPSPEED1 = MD2MAXSPEED(MD2MOTOR1)
	MD2TMPSPEED2 = MD2MAXSPEED(MD2MOTOR2)
	MD2TMPHOLD = MD2HOLD

	'SETUP MOTOR PARAMETERS.
	IF MD2MOVETYPE = "R" THEN
		'RELATIVE CENTER POSITIONS.
		MD2CNTRX = MD2POSITION(MD2MOTOR1) + MD2CIRCLECENTERX
		MD2CNTRY = MD2POSITION(MD2MOTOR2) + MD2CIRCLECENTERY
	ELSE
		'ABSOLUTE CENTER POSITIONS.
		MD2CNTRX = MD2CIRCLECENTERX
		MD2CNTRY = MD2CIRCLECENTERY
	END IF
	MD2MOVETYPE = "A"
   
	MD2SLOPE(MD2MOTOR1) = 0
	MD2SLOPE(MD2MOTOR2) = 0
	MD2MAXSPEED(MD2MOTOR1) = MD2MINSPEED(MD2MOTOR1)
	MD2MAXSPEED(MD2MOTOR2) = MD2MINSPEED(MD2MOTOR2)
	MD2HOLD = -1
   
	'SET ANGLE AND END ANGLE.
	MD2ENDANGLE = MD2CIRCLESTART + MD2CIRCLEARC
	MD2ANGLE = MD2CIRCLESTART
   
	'PLOT BASED ON DIRECTION.
	IF MD2CIRCLEARC > 0 THEN
	   
		'CCW PLOT LOOP.
		DO
			GOSUB MD2CIRCLEDO
			IF MD2STATUS <> "O" THEN EXIT DO
			MD2ANGLE = MD2ANGLE + MD2CIRCLECHORD
		LOOP UNTIL MD2ANGLE >= MD2ENDANGLE
   
	ELSE
	   
		'CW PLOT LOOP.
		DO
			GOSUB MD2CIRCLEDO
			IF MD2STATUS <> "O" THEN EXIT DO
			MD2ANGLE = MD2ANGLE - MD2CIRCLECHORD
		LOOP UNTIL MD2ANGLE <= MD2ENDANGLE
   
	END IF

	'PLOT LAST ANGLE UNLESS PREVIOUS ERROR.
	IF MD2STATUS = "O" THEN
		MD2ANGLE = MD2ENDANGLE
		GOSUB MD2CIRCLEDO
	END IF
   
	'RESTORE PARAMETERS.
	MD2MOVETYPE = MD2TMPMOVETYPE
	MD2SLOPE(MD2MOTOR1) = MD2TMPSLOPE1
	MD2SLOPE(MD2MOTOR2) = MD2TMPSLOPE2
	MD2MAXSPEED(MD2MOTOR1) = MD2TMPSPEED1
	MD2MAXSPEED(MD2MOTOR2) = MD2TMPSPEED2
	MD2HOLD = MD2TMPHOLD
  
	'POWER OFF MOTOR IF DESIRED.
	IF MD2HOLD = 0 THEN
		IF MD2MOTOR = 12 THEN OUT MD2MTRADR12, &HFF
		IF MD2MOTOR = 34 THEN OUT MD2MTRADR34, &HFF
		IF MD2MOTOR = 56 THEN OUT MD2MTRADR56, &HFF
	END IF
		
	EXIT SUB

'MOVE TO ANGLE DESIRED ANGLE.
'
MD2CIRCLEDO:
	'CONVERT TO RADIANS.
	MD2ANGLER = MD2ANGLE * (3.1416 / 180)

	'CALCULATE X & Y AND MOVE.
	MD2TARGET(MD2MOTOR1) = (MD2CIRCLERADIUSX * COS(MD2ANGLER)) + MD2CNTRX
	MD2TARGET(MD2MOTOR2) = (MD2CIRCLERADIUSY * SIN(MD2ANGLER)) + MD2CNTRY

	'MOVE THE MOTORS.
	MD2MOVE
   
	'DEBUG CODE.
	'LINE -((MD2TARGET(MD2MOTOR1) * 30) + 160, ABS((MD2TARGET(MD2MOTOR2) * 30) - 199) - 99)
	'PRESET ((MD2TARGET(MD2MOTOR1) * 30) + 160, ABS((MD2TARGET(MD2MOTOR2) * 30) - 199) - 99)
	'PRINT MD2TARGET(MD2MOTOR1), MD2TARGET(MD2MOTOR2): INPUT "", A$
	'DO: LOOP WHILE INKEY$ = ""

	RETURN

END SUB

SUB MD2GRID

	'---------------------------------------------------------------------
	'NAME:      MD2GRID
	'DESC:      THE MD2GRID PROCEDURE IS USED TO MOVE MOTORS TO A LOCATION
	'           OF A GRID.  A GRID IS AN X Y MATRIX OF LOCATIONS LIKE
	'           A CHECKER BOARD.  THIS ALLOWS A PROGRAMMER TO MOVE TO
	'           A ROW & COLUMN OF A GRID INSTEAD OF MOVING BY PASSING
	'           MD2GRID WORKS WITH ABSOLUTE TARGET POSITIONS ONLY.
	'           THE SPECIFIC TARGET LOCATION IN STEPS OR INCHES, ETC.
	'           THE FOLLOWING PARAMETERS DEFINE THE GRID.
	'
	'               PARAMETER       DESCRIPTION
	'               ---------       -----------
	'               MD2GRIDBEGINX   X BEGIN ABSOLUTE POSITION IN UNITS
	'               MD2GRIDBEGINY   Y BEGIN ABSOLUTE POSITION IN UNITS
	'               MD2GRIDSPACEX   X SPACING IN UNITS.
	'               MD2GRIDSPACEY   Y SPACING IN UNITS.
	'               MD2GRIDTARGETX  X TARGET, ABSOLUTE.
	'               MD2GRIDTARGETY  Y TARGET, ABSOLUTE.
	'
	'USAGE:     SET GRID PARAMETERS TO DEFINE THE GRID, SET MD2MOTOR
	'           TO A VALID MOTOR PAIR (12,34 OR 56), SET MOTOR PARAMETERS
	'           THEN CALL.
	'INPUTS:    ALL GRID AND MOTOR PARAMETERS.
	'OUTPUTS:   MD2STATUS,MD2TARGET(M),MD2POSITION(M).
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	DIM MD2X AS INTEGER        'X MOTOR #.
	DIM MD2Y AS INTEGER        'Y MOTOR #.
	DIM MD2MT AS STRING        'MOVETYPE STORAGE.

	'CHECK FOR BAD PARAMETERS.
	MD2STATUS = "B"
	IF MD2MOTOR <> 12 AND MD2MOTOR <> 34 AND MD2MOTOR <> 56 THEN EXIT SUB
	MD2STATUS = "O"

	'SET MOTOR #.
	IF MD2MOTOR = 12 THEN MD2X = 1: MD2Y = 2
	IF MD2MOTOR = 34 THEN MD2X = 3: MD2Y = 4
	IF MD2MOTOR = 56 THEN MD2X = 5: MD2Y = 6

	'SET TARGET POSITIONS.
	MD2TARGET(MD2X) = MD2GRIDBEGINX + ((MD2GRIDTARGETX - 1) * MD2GRIDSPACEX)
	MD2TARGET(MD2Y) = MD2GRIDBEGINY + ((MD2GRIDTARGETY - 1) * MD2GRIDSPACEY)

	'GRID MOVE.
	MD2MT = MD2MOVETYPE     'SAVE MOVETYPE.
	MD2MOVETYPE = "A"       'ABSOLUTE MOVES.
	MD2MOVE                 'MOVE TO THE GRID COORDINATES.
	MD2MOVETYPE = MD2MT     'RESTORE MOVETYPE.

END SUB

SUB MD2HOME

	'---------------------------------------------------------------------
	'NAME:      MD2HOME
	'DESC:      THE MD2HOME PROCEDURE IS USED TO MOVE THE STEPPER MOTOR
	'           TO A KNOWN HOME POSITION.  ALL OTHER MOVES ARE RELATIVE
	'           TO THIS HOME (ZERO) POSITION.  THE SELECTED MOTOR IS
	'           MOVED REVERSE UNTIL THE SWITCH IS ACTIVATED, THEN FORWARD
	'           UNTIL DEACTIVATED, THEN MOVED FORWARD ACCORDING TO THE
	'           MD2HOMEOFFSET(M) PARAMETER. THE CURRENT POSITION IS THEN
	'           SET TO ZERO - THIS IS THE HOME POSITION.  THE MD2MINSPEED
	'           PARAMETER IS USED FOR THE MOTOR SPEED - NO RAMPING IS DONE.
	'           ONLY 1 MOTOR (1-6) CAN BE HOME'D AT A TIME.
	'USAGE:     SET THE DESIRED MOTOR #, MOTOR PARAMETERS AND CALL.
	'INPUTS:    MOTOR # AND MOTOR PARAMETERS.
	'OUTPUTS:   CURRENT POSITION SET TO ZERO, MD2STATUS.
	'---------------------------------------------------------------------
   
	'LOCAL VARIABLES.
	DIM MD2DELAY AS LONG        'DELAY COUNTER BETWEEN STEPS.
	DIM MD2DIR AS INTEGER       'DIRECTION. -1=REV, 1=FWD.
	DIM MD2MTRADR AS INTEGER    'MOTOR PORT ADDRESS.
	DIM MD2PATS AS INTEGER      'CALCULATED STEP PATTERN.
	DIM MD2PMASK AS INTEGER     'SELECTED STEP PATTERN MASK.
	DIM MD2OMASK AS INTEGER     'OTHER MOTOR PATTERN MASK.
	DIM MD2SMASK AS INTEGER     'SWITCH MASK.
	DIM MD2INTREG AS INTEGER    'INTERRUPT REGISTER.
	DIM MD2I AS LONG            'TEMPORARY.
	DIM MD2WAY AS SINGLE        'TEMP FOR VELOCITY INTEROLATION.
	DIM MD2MINDELAY AS LONG     'CALCULATED DELAY COUNTER.

	'CHECK FOR SETUP.
	IF MD2MTRADR12 <> &H3BC THEN MD2STATUS = "S": EXIT SUB

	'CHECK FOR CALIBRATION.
	IF NOT MD2CALIBRATED THEN MD2STATUS = "C": EXIT SUB

	'CHECK FOR PORT AVAILABILITY.
	MD2STATUS = "P"
	IF (MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 12) AND NOT MD2AVAILABLE12 THEN EXIT SUB
	IF (MD2MOTOR = 3 OR MD2MOTOR = 4 OR MD2MOTOR = 34) AND NOT MD2AVAILABLE34 THEN EXIT SUB
	IF (MD2MOTOR = 5 OR MD2MOTOR = 6 OR MD2MOTOR = 56) AND NOT MD2AVAILABLE56 THEN EXIT SUB

	'CHECK FOR ENABLED MD-2.
	MD2STATUS = "E"
	IF (MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 12) AND NOT MD2ENABLED12 THEN EXIT SUB
	IF (MD2MOTOR = 3 OR MD2MOTOR = 4 OR MD2MOTOR = 34) AND NOT MD2ENABLED34 THEN EXIT SUB
	IF (MD2MOTOR = 5 OR MD2MOTOR = 6 OR MD2MOTOR = 56) AND NOT MD2ENABLED56 THEN EXIT SUB

	'CHECK FOR VALID MOTOR #.
	MD2STATUS = "B"
	IF MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 3 THEN MD2STATUS = "O"
	IF MD2MOTOR = 4 OR MD2MOTOR = 5 OR MD2MOTOR = 6 THEN MD2STATUS = "O"
	IF MD2STATUS <> "O" THEN EXIT SUB

	'CHECK FOR VALID STEP TYPE.
	MD2STEPTYPE = UCASE$(MD2STEPTYPE)
	IF MD2STEPTYPE <> "H" AND MD2STEPTYPE <> "D" AND MD2STEPTYPE <> "S" THEN MD2STATUS = "B": EXIT SUB

	'INITIALIZE STATUS.
	MD2STATUS = "O"

	'SET STEP PATTERNS.
	IF MD2STEPTYPE = "H" THEN   'HALF STEP MODE.
		MD2STPPAT(0) = &H66: MD2STPPAT(1) = &H77
		MD2STPPAT(2) = &H33: MD2STPPAT(3) = &HBB
		MD2STPPAT(4) = &H99: MD2STPPAT(5) = &HDD
		MD2STPPAT(6) = &HCC: MD2STPPAT(7) = &HEE
	END IF
	IF MD2STEPTYPE = "S" THEN   'SINGLE PHASE FULL STEP MODE.
		MD2STPPAT(0) = &H77: MD2STPPAT(1) = &HBB
		MD2STPPAT(2) = &HDD: MD2STPPAT(3) = &HEE
		MD2STPPAT(4) = &H77: MD2STPPAT(5) = &HBB
		MD2STPPAT(6) = &HDD: MD2STPPAT(7) = &HEE
	END IF
	IF MD2STEPTYPE = "D" THEN   'DOUBLE PHASE FULL STEP MODE.
		MD2STPPAT(0) = &H66: MD2STPPAT(1) = &H33
		MD2STPPAT(2) = &H99: MD2STPPAT(3) = &HCC
		MD2STPPAT(4) = &H66: MD2STPPAT(5) = &H33
		MD2STPPAT(6) = &H99: MD2STPPAT(7) = &HCC
	END IF

	'SET UP ADDRESS.
	IF MD2MOTOR = 1 OR MD2MOTOR = 2 THEN MD2MTRADR = MD2MTRADR12
	IF MD2MOTOR = 3 OR MD2MOTOR = 4 THEN MD2MTRADR = MD2MTRADR34
	IF MD2MOTOR = 5 OR MD2MOTOR = 6 THEN MD2MTRADR = MD2MTRADR56

	'SET UP PATTERN MASK, OTHER MOTOR'S MASK AND SWITCH MASK.
	IF MD2MOTOR = 1 OR MD2MOTOR = 3 OR MD2MOTOR = 5 THEN
		MD2PMASK = &HF
		MD2OMASK = &HF0
		MD2SMASK = &H20
	ELSE
		MD2PMASK = &HF0
		MD2OMASK = &HF
		MD2SMASK = &H10
	END IF

	'CONVERT MINSPEED VELOCITY TO DELAY COUNTS.
	FOR MD2I = 2 TO 23     'SEARCH FOR CORRECT ARRAY ELEMENTS.
		IF (MD2MINSPEED(MD2MOTOR) * MD2UNITS(MD2MOTOR)) >= MD2VELOCITY(MD2I) THEN EXIT FOR
	NEXT MD2I
	'INTERPOLATE.
	MD2WAY = ((MD2MINSPEED(MD2MOTOR) * MD2UNITS(MD2MOTOR)) - MD2VELOCITY(MD2I - 1)) / (MD2VELOCITY(MD2I) - MD2VELOCITY(MD2I - 1))
	MD2MINDELAY = INT((((MD2DELAY(MD2I) - MD2DELAY(MD2I - 1)) * MD2WAY) + MD2DELAY(MD2I - 1)) + .5)

	'SELECT DIRECTION TO HOME.
	IF MD2HOMEDIR(MD2MOTOR) = 0 THEN MD2DIR = -1 ELSE MD2DIR = 1
   
	'TURN OFF COM INTERRUPTS?
	IF MD2INTERRUPTS = 0 THEN
		MD2INTREG = INP(&H21)
		OUT &H21, MD2INTREG OR &H98
	END IF

	'---------------------------------------------
	'MOVE MOTOR REVERSE UNTIL SWITCH ACTIVATED.
	'---------------------------------------------

	'LOOP UNTIL SWITCH IS ACTIVATED.
	DO UNTIL (INP(MD2MTRADR + 1) AND MD2SMASK) = 0

		'QUIT MOVING IF KEY PRESSED.
		'IF INKEY$ <> "" THEN MD2STATUS = "K": EXIT DO
		INREG.AX = &HB00: INTERRUPT &H21, INREG, OUTREG
		IF OUTREG.AX > &HB00 THEN
			INREG.AX = &HC00: INTERRUPT &H21, INREG, OUTREG
			MD2STATUS = "K"
			EXIT DO
		END IF

		'POINT TO THE NEXT STEP PATTERN.
		MD2PATPTR(MD2MOTOR) = (MD2PATPTR(MD2MOTOR) + MD2DIR) AND &H7

		'GET STEP PATTERN AND MASK OFF UNNEEDED BITS.
		MD2PATS = MD2STPPAT(MD2PATPTR(MD2MOTOR)) AND MD2PMASK

		'DON'T DISTURB OTHER MOTORS BITS.
		MD2PATS = MD2PATS OR (INP(MD2MTRADR) AND MD2OMASK)

		'OUTPUT THE STEP PATTERN TO MOVE THE MOTOR.
		OUT MD2MTRADR, MD2PATS

		'DELAY BETWEEN STEPS.
		FOR MD2DELAY = 1 TO MD2MINDELAY: NEXT MD2DELAY
	
	LOOP

	'----------------------------------------------
	'MOVE MOTOR FORWARD UNTIL SWITCH DEACTIVATED.
	'----------------------------------------------

	'LOOP UNTIL SWITCH IS DEACTIVATED.
	DO UNTIL (INP(MD2MTRADR + 1) AND MD2SMASK) <> 0

		'QUIT MOVING IF KEY PRESSED.
		'IF INKEY$ <> "" THEN MD2STATUS = "K": EXIT DO
		IF MD2STATUS = "K" THEN EXIT DO
		INREG.AX = &HB00: INTERRUPT &H21, INREG, OUTREG
		IF OUTREG.AX > &HB00 THEN
			INREG.AX = &HC00: INTERRUPT &H21, INREG, OUTREG
			MD2STATUS = "K"
			EXIT DO
		END IF

		'POINT TO THE NEXT STEP PATTERN.
		MD2PATPTR(MD2MOTOR) = (MD2PATPTR(MD2MOTOR) - MD2DIR) AND &H7
	   
		'GET STEP PATTERN AND MASK OFF UNNEEDED BITS.
		MD2PATS = MD2STPPAT(MD2PATPTR(MD2MOTOR)) AND MD2PMASK

		'DON'T DISTURB OTHER MOTORS BITS.
		MD2PATS = MD2PATS OR (INP(MD2MTRADR) AND MD2OMASK)
	   
		'OUTPUT THE STEP PATTERN TO MOVE THE MOTOR.
		OUT MD2MTRADR, MD2PATS

		'DELAY BETWEEN STEPS.
		FOR MD2DELAY = 1 TO MD2MINDELAY: NEXT MD2DELAY
	
	LOOP
   
	'-----------------------------------------------
	'OVER STEP HOME POSITION USING MD2HOMEOFFSET(M).
	'-----------------------------------------------

	'STEP UNTIL COUNT DEPLETED.
	MD2I = INT(MD2HOMEOFFSET(MD2MOTOR) * MD2UNITS(MD2MOTOR))
	DO UNTIL MD2I = 0
	   
		'QUIT MOVING IF KEY PRESSED.
		'IF INKEY$ <> "" THEN MD2STATUS = "K": EXIT DO
		IF MD2STATUS = "K" THEN EXIT DO
		INREG.AX = &HB00: INTERRUPT &H21, INREG, OUTREG
		IF OUTREG.AX > &HB00 THEN
			INREG.AX = &HC00: INTERRUPT &H21, INREG, OUTREG
			MD2STATUS = "K"
			EXIT DO
		END IF

		'POINT TO THE NEXT STEP PATTERN.
		MD2PATPTR(MD2MOTOR) = (MD2PATPTR(MD2MOTOR) - MD2DIR) AND &H7
	  
		'GET STEP PATTERN AND MASK OFF UNNEEDED BITS.
		MD2PATS = MD2STPPAT(MD2PATPTR(MD2MOTOR)) AND MD2PMASK

		'DON'T DISTURB OTHER MOTORS BITS.
		MD2PATS = MD2PATS OR (INP(MD2MTRADR) AND MD2OMASK)
	  
		'OUTPUT THE STEP PATTERN TO MOVE THE MOTOR.
		OUT MD2MTRADR, MD2PATS

		'DELAY BETWEEN STEPS.
		FOR MD2DELAY = 1 TO MD2MINDELAY: NEXT MD2DELAY

		'UPDATE STEP COUNT.
		MD2I = MD2I - 1
   
	LOOP

	'TURN COM INTERRUPTS BACK ON?
	IF MD2INTERRUPTS = 0 THEN OUT &H21, MD2INTREG

	'SET LAST DIRECTION ACCORDING TO HOMEDIR.
	IF MD2HOMEDIR(MD2MOTOR) = 0 THEN
		MD2LASTDIR(MD2MOTOR) = 1    'HOMEDIR=REV, SET LASTDIR TO FORWARD.
	ELSE
		MD2LASTDIR(MD2MOTOR) = -1   'HOMEDIR=FWD, SET LASTDIR TO REVERSE.
	END IF

	'SET POSITION TO 0.
	MD2POSITION(MD2MOTOR) = 0

	'POWER OFF MOTOR IF DESIRED.
	IF MD2HOLD = 0 THEN OUT MD2MTRADR, &HFF

END SUB

SUB MD2INPUTS

	'---------------------------------------------------------------------
	'NAME:      MD2INPUTS
	'DESC:      THE MD2INPUTS PROCEDURE READS ALL INPUTS FROM ALL MD-2
	'           SYSTEMS INCLUDING HOME SWITCHES AND GENERAL PURPOSE INPUTS
	'           AND SETS VARIABLES ACCORDINGLY.  VARIABLES FOR PORTS THAT
	'           ARE NOT CONNECTED TO MD-2 SYSTEMS WILL CONTAIN USELESS
	'           INFORMATION.  EACH MD-2 SYSTEM HAS 2 HOME SWITCHES (ONE
	'           ASSOCIATED WITH EACH MOTOR) WHICH ARE INPUTS AND SOME
	'           MD-2 SYSTEMS HAVE 3 GENERAL PURPOSE INPUTS WHICH ARE
	'           USER DEFINED.  CHECK YOUR MANUAL TO SEE IF YOU HAVE
	'           GENERAL PURPOSE INPUTS.
	'USAGE:     USE TO READ THE CONDITION OF HOME SWITCHES, SENSORS
	'           AND OTHER INPUTS.
	'INPUTS:    NONE.
	'OUTPUTS:   INPUT VARIABLES -1=ACTIVE(LOGIC 0), 0=NOT ACTIVE(LOGIC 1).
	'
	'           VARIABLES  MD-2 PORT  INPUT           CONDITION
	'           ---------  ---------  -----           ---------
	'           MD2INPUT(11)  3BC     HOME SWITCH #1  -1=SWITCH ACTIVE (LOW).
	'           MD2INPUT(12)  3BC     HOME SWITCH #2  -1=SWITCH ACTIVE (LOW).
	'           MD2INPUT(13)  3BC     GEN PURP IN #1  -1=INPUT ACTIVE (LOW).
	'           MD2INPUT(14)  3BC     GEN PURP IN #2  -1=INPUT ACTIVE (LOW).
	'           MD2INPUT(15)  3BC     GEN PURP IN #3  -1=INPUT ACTIVE (LOW).
	'
	'           MD2INPUT(21)  378     HOME SWITCH #1  -1=SWITCH ACTIVE (LOW).
	'           MD2INPUT(22)  378     HOME SWITCH #2  -1=SWITCH ACTIVE (LOW).
	'           MD2INPUT(23)  378     GEN PURP IN #1  -1=INPUT ACTIVE (LOW).
	'           MD2INPUT(24)  378     GEN PURP IN #2  -1=INPUT ACTIVE (LOW).
	'           MD2INPUT(25)  378     GEN PURP IN #3  -1=INPUT ACTIVE (LOW).
	'
	'           MD2INPUT(31)  278     HOME SWITCH #1  -1=SWITCH ACTIVE (LOW).
	'           MD2INPUT(32)  278     HOME SWITCH #2  -1=SWITCH ACTIVE (LOW).
	'           MD2INPUT(33)  278     GEN PURP IN #1  -1=INPUT ACTIVE (LOW).
	'           MD2INPUT(34)  278     GEN PURP IN #2  -1=INPUT ACTIVE (LOW).
	'           MD2INPUT(35)  278     GEN PURP IN #3  -1=INPUT ACTIVE (LOW).
	'---------------------------------------------------------------------

	'FIRST CLEAR ALL VARIABLES.
	MD2INPUT(11) = 0
	MD2INPUT(12) = 0
	MD2INPUT(13) = 0
	MD2INPUT(14) = 0
	MD2INPUT(15) = 0
	MD2INPUT(21) = 0
	MD2INPUT(22) = 0
	MD2INPUT(23) = 0
	MD2INPUT(24) = 0
	MD2INPUT(25) = 0
	MD2INPUT(31) = 0
	MD2INPUT(32) = 0
	MD2INPUT(33) = 0
	MD2INPUT(34) = 0
	MD2INPUT(35) = 0

	'READ INPUTS AND SET VARIABLES IF INPUTS ARE ACTIVE (LOGIC ZERO).

	'MD-2 AT 3BC.
	IF (INP(&H3BC + 1) AND &H20) = 0 THEN MD2INPUT(11) = -1
	IF (INP(&H3BC + 1) AND &H10) = 0 THEN MD2INPUT(12) = -1
	IF (INP(&H3BC + 1) AND &H80) <> 0 THEN MD2INPUT(13) = -1
	IF (INP(&H3BC + 1) AND &H40) <> 0 THEN MD2INPUT(14) = -1
	IF (INP(&H3BC + 1) AND &H8) <> 0 THEN MD2INPUT(15) = -1

	'MD-2 AT 378.
	IF (INP(&H378 + 1) AND &H20) = 0 THEN MD2INPUT(21) = -1
	IF (INP(&H378 + 1) AND &H10) = 0 THEN MD2INPUT(22) = -1
	IF (INP(&H378 + 1) AND &H80) <> 0 THEN MD2INPUT(23) = -1
	IF (INP(&H378 + 1) AND &H40) <> 0 THEN MD2INPUT(24) = -1
	IF (INP(&H378 + 1) AND &H8) <> 0 THEN MD2INPUT(25) = -1
   
	'MD-2 AT 278.
	IF (INP(&H278 + 1) AND &H20) = 0 THEN MD2INPUT(31) = -1
	IF (INP(&H278 + 1) AND &H10) = 0 THEN MD2INPUT(32) = -1
	IF (INP(&H278 + 1) AND &H80) <> 0 THEN MD2INPUT(33) = -1
	IF (INP(&H278 + 1) AND &H40) <> 0 THEN MD2INPUT(34) = -1
	IF (INP(&H278 + 1) AND &H8) <> 0 THEN MD2INPUT(35) = -1
												   
END SUB

SUB MD2MOVE

	'---------------------------------------------------------------------
	'NAME:      MD2MOVE
	'DESC:      THIS PROCEDURE IS USED TO MOVE MOTORS.
	'           SET MD2MOTOR TO 1,2,3,4,5 OR 6 FOR SINGLE MOTOR MOVES OR
	'           TO 12, 34 OR 56 FOR DUAL MOTOR MOVES.
	'           WHEN MOVING 2 MOTORS, BOTH MOTORS START AND STOP AT THE
	'           SAME TIME RESULTING IN STRAIGHT LINE MOTION ALSO KNOWN
	'           AS LINEAR INTERPOLATION.
	'           SPEED AND RAMPING OF THE MOTOR WITH THE LONGEST
	'           TRAVEL WILL BE USED.
	'           SOFT LIMIT VALUES WILL PREVENT OVERTRAVEL.
	'           THE HOME SWITCH IS IGNORED.
	'           A KEYPRESS WILL STOP MOTION.
	'           IF SLOPE=0 THEN THE MOTOR WILL MOVE AT MD2MAXSPEED
	'USAGE:     SET MOTOR #, MOTOR PARAMETERS AND CALL.
	'INPUTS:    MOTOR # (1-6,12,34 OR 56) AND MOTOR PARAMETERS.
	'OUTPUTS:   CURRENT POSITION, MD2STATUS.
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	DIM MD2MTRADR AS INTEGER    'MOTOR PORT ADDRESS.
	DIM MD2PATS AS INTEGER      'CALCULATED STEP PATTERN.
	DIM MD2INTREG AS INTEGER    'INTERRUPT REGISTER.
	DIM MD2DIRSWAPL AS INTEGER  'LONG MOTOR SWAP DIR. -1=YES, 1=NO.
	DIM MD2DIRSWAPS AS INTEGER  'LONG MOTOR SWAP DIR. -1=YES, 1=NO.

	'RAMPING VARIABLES.
	DIM MD2DELAYTMP AS LONG     'TEMPORARY DELAY LOOP COUNTER.
	DIM MD2DELAYCUR AS LONG     'CURRENT SPEED DELAY COUNT.
	DIM MD2DELAYFLAG AS LONG    'FLAG INDICATING DELAY ADJUSTMENT.
	DIM MD2DELAYEACH AS LONG    'DELAY ADJUSTMENT EACH STEP.
	DIM MD2DELAYCHANGE AS LONG  'AMOUNT OF CHANGE IN DELAY.
	DIM MD2DELAYCHANGE2 AS LONG 'DELAYCHANGE * 2.
	DIM MD2DECEL AS LONG        'STEP COUNT TO BEGIN DECELERATION MODE.
	DIM MD2DELAYSLOPE AS LONG   'SLOPE.
	DIM MD2DELAYSLOPE2 AS LONG  'SLOPE * 2.
	DIM MD2MODE AS INTEGER      'SPEED MODE. 1=DECEL,0=CONSTANT,-1=ACCEL.
	DIM MD2MINDELAY AS LONG     'MINIMUM SPEED DELAY COUNT.
	DIM MD2MAXDELAY AS LONG     'MAXIMUM SPEED DELAY COUNT.
	DIM MD2WAY AS SINGLE        'TEMP FOR VELOCITY INTEROLATION.
	DIM MD2I AS INTEGER         'ARRAY ELEMENT COUNTER.

	'LINEAR INTERPOLATION VARIABLES.
	DIM MD2LIERROR AS LONG      'ERROR VALUE FOR LINEAR INTERPOLATION.
	DIM MD2MOTORL AS INTEGER    'MOTOR # TRAVELING THE LONGEST.
	DIM MD2MOTORS AS INTEGER    'MOTOR # TRAVELING THE SHORTEST.
	DIM MD2DIRL AS INTEGER      'LONG MOTOR DIRECTION. -1=REV,1=FWD.
	DIM MD2DIRS AS INTEGER      'SHORT MOTOR DIRECTION.
	DIM MD2MASKL AS INTEGER     'LONG MOTOR PATTERN MASK.
	DIM MD2MASKS AS INTEGER     'SHORT MOTOR PATTERN MASK.
	DIM MD2STEPSL AS LONG       '# OF STEPS FOR LONG MOTOR.
	DIM MD2STEPSS AS LONG       '# OF STEPS FOR SHORT MOTOR.
	DIM MD2STEPSL2 AS LONG      'MD2STEPSL * 2.
	DIM MD2STEPSS2 AS LONG      'MD2STEPSS * 2.
	DIM MD2SWAPL AS LONG        'SWAP TEMPORARY VARIABLE.

	'CHECK FOR SETUP.
	IF MD2MTRADR12 <> &H3BC THEN MD2STATUS = "S": EXIT SUB

	'IGNORE SOME CHECKS IF IN CALIBRATION MODE.
	IF NOT MD2CALMODE THEN

	  'CHECK FOR CALIBRATION.
	  IF NOT MD2CALIBRATED THEN MD2STATUS = "C": EXIT SUB

	  'CHECK FOR PORT AVAILABILITY.
	  MD2STATUS = "P"
	  IF (MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 12) AND NOT MD2AVAILABLE12 THEN EXIT SUB
	  IF (MD2MOTOR = 3 OR MD2MOTOR = 4 OR MD2MOTOR = 34) AND NOT MD2AVAILABLE34 THEN EXIT SUB
	  IF (MD2MOTOR = 5 OR MD2MOTOR = 6 OR MD2MOTOR = 56) AND NOT MD2AVAILABLE56 THEN EXIT SUB

	  'CHECK FOR ENABLED MD-2.
	  MD2STATUS = "E"
	  IF (MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 12) AND NOT MD2ENABLED12 THEN EXIT SUB
	  IF (MD2MOTOR = 3 OR MD2MOTOR = 4 OR MD2MOTOR = 34) AND NOT MD2ENABLED34 THEN EXIT SUB
	  IF (MD2MOTOR = 5 OR MD2MOTOR = 6 OR MD2MOTOR = 56) AND NOT MD2ENABLED56 THEN EXIT SUB

	END IF

	'CHECK FOR VALID MOTOR #.
	MD2STATUS = "B"
	IF MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 3 THEN MD2STATUS = "O"
	IF MD2MOTOR = 4 OR MD2MOTOR = 5 OR MD2MOTOR = 6 THEN MD2STATUS = "O"
	IF MD2MOTOR = 12 OR MD2MOTOR = 34 OR MD2MOTOR = 56 THEN MD2STATUS = "O"
	IF MD2STATUS <> "O" THEN EXIT SUB

	'CHECK FOR VALID STEP TYPE.
	MD2STEPTYPE = UCASE$(MD2STEPTYPE)
	IF MD2STEPTYPE <> "H" AND MD2STEPTYPE <> "D" AND MD2STEPTYPE <> "S" THEN MD2STATUS = "B": EXIT SUB

	'CHECK FOR VALID MOVE TYPE.
	MD2MOVETYPE = UCASE$(MD2MOVETYPE)
	IF MD2MOVETYPE <> "A" AND MD2MOVETYPE <> "R" THEN MD2STATUS = "B": EXIT SUB
												   
	'SET PORT ADDRESSES AND MOTOR NUMBERS.
	IF MD2MOTOR = 1 THEN
		MD2MTRADR = MD2MTRADR12
		MD2MOTORL = 1
		MD2MOTORS = 0
	END IF
	IF MD2MOTOR = 2 THEN
		MD2MTRADR = MD2MTRADR12
		MD2MOTORL = 2
		MD2MOTORS = 0
	END IF
	IF MD2MOTOR = 3 THEN
		MD2MTRADR = MD2MTRADR34
		MD2MOTORL = 3
		MD2MOTORS = 0
	END IF
	IF MD2MOTOR = 4 THEN
		MD2MTRADR = MD2MTRADR34
		MD2MOTORL = 4
		MD2MOTORS = 0
	END IF
	IF MD2MOTOR = 5 THEN
		MD2MTRADR = MD2MTRADR56
		MD2MOTORL = 5
		MD2MOTORS = 0
	END IF
	IF MD2MOTOR = 6 THEN
		MD2MTRADR = MD2MTRADR56
		MD2MOTORL = 6
		MD2MOTORS = 0
	END IF
	IF MD2MOTOR = 12 THEN
		MD2MTRADR = MD2MTRADR12
		MD2MOTORL = 1
		MD2MOTORS = 2
	END IF
	IF MD2MOTOR = 34 THEN
		MD2MTRADR = MD2MTRADR34
		MD2MOTORL = 3
		MD2MOTORS = 4
	END IF
	IF MD2MOTOR = 56 THEN
		MD2MTRADR = MD2MTRADR56
		MD2MOTORL = 5
		MD2MOTORS = 6
	END IF

	'SET STEP PATTERNS.
	IF MD2STEPTYPE = "H" THEN   'HALF STEP MODE.
		MD2STPPAT(0) = &H66: MD2STPPAT(1) = &H77
		MD2STPPAT(2) = &H33: MD2STPPAT(3) = &HBB
		MD2STPPAT(4) = &H99: MD2STPPAT(5) = &HDD
		MD2STPPAT(6) = &HCC: MD2STPPAT(7) = &HEE
	END IF
	IF MD2STEPTYPE = "S" THEN   'SINGLE PHASE FULL STEP MODE.
		MD2STPPAT(0) = &H77: MD2STPPAT(1) = &HBB
		MD2STPPAT(2) = &HDD: MD2STPPAT(3) = &HEE
		MD2STPPAT(4) = &H77: MD2STPPAT(5) = &HBB
		MD2STPPAT(6) = &HDD: MD2STPPAT(7) = &HEE
	END IF
	IF MD2STEPTYPE = "D" THEN   'DOUBLE PHASE FULL STEP MODE.
		MD2STPPAT(0) = &H66: MD2STPPAT(1) = &H33
		MD2STPPAT(2) = &H99: MD2STPPAT(3) = &HCC
		MD2STPPAT(4) = &H66: MD2STPPAT(5) = &H33
		MD2STPPAT(6) = &H99: MD2STPPAT(7) = &HCC
	END IF
						  
	'FIX VARIABLES FOR 0.
	MD2TARGET(0) = 0    'MOTOR 0 IS NO MOTOR, NO STEPS, ETC.
	MD2POSITION(0) = 0
	MD2UNITS(0) = 1
  
	'SET STEP COUNT AND DIRECTION FOR RELATIVE MOVES.
	IF MD2MOVETYPE = "R" THEN   'RELATIVE MOVE?
		'IF LIMITS EXCEEDED THEN QUIT.
		IF (MD2TARGET(MD2MOTORL) + MD2POSITION(MD2MOTORL)) > MD2LIMITF(MD2MOTORL) THEN MD2STATUS = "L"
		IF (MD2TARGET(MD2MOTORL) + MD2POSITION(MD2MOTORL)) < MD2LIMITR(MD2MOTORL) THEN MD2STATUS = "L"
		IF (MD2TARGET(MD2MOTORS) + MD2POSITION(MD2MOTORS)) > MD2LIMITF(MD2MOTORS) THEN MD2STATUS = "L"
		IF (MD2TARGET(MD2MOTORS) + MD2POSITION(MD2MOTORS)) < MD2LIMITR(MD2MOTORS) THEN MD2STATUS = "L"
		IF MD2STATUS = "L" THEN EXIT SUB
		'SET THE # OF STEPS.
		MD2STEPSL = INT((MD2TARGET(MD2MOTORL) * MD2UNITS(MD2MOTORL)) + .5)
		MD2STEPSS = INT((MD2TARGET(MD2MOTORS) * MD2UNITS(MD2MOTORS)) + .5)
		'SET THE DIRECTION.
		IF MD2STEPSL < 0 THEN MD2DIRL = -1 ELSE MD2DIRL = 1
		IF MD2STEPSS < 0 THEN MD2DIRS = -1 ELSE MD2DIRS = 1
		'MAKE STEP COUNTS POSITIVE.
		MD2STEPSL = ABS(MD2STEPSL)
		MD2STEPSS = ABS(MD2STEPSS)
		'SET COMPLETED POSITIONS AND ROUND.
		MD2POSITION(MD2MOTORL) = MD2POSITION(MD2MOTORL) + ((MD2STEPSL * (1 / MD2UNITS(MD2MOTORL))) * MD2DIRL)
		MD2POSITION(MD2MOTORL) = INT((MD2POSITION(MD2MOTORL) * MD2UNITS(MD2MOTORL)) + .5)
		MD2POSITION(MD2MOTORL) = MD2POSITION(MD2MOTORL) / MD2UNITS(MD2MOTORL)

		MD2POSITION(MD2MOTORS) = MD2POSITION(MD2MOTORS) + ((MD2STEPSS * (1 / MD2UNITS(MD2MOTORS))) * MD2DIRS)
		MD2POSITION(MD2MOTORS) = INT((MD2POSITION(MD2MOTORS) * MD2UNITS(MD2MOTORS)) + .5)
		MD2POSITION(MD2MOTORS) = MD2POSITION(MD2MOTORS) / MD2UNITS(MD2MOTORS)
	END IF
  
	'SET STEP COUNT AND DIRECTION FOR ABSOLUTE MOVES.
	IF MD2MOVETYPE = "A" THEN   'ABSOLUTE MOVE?
		'IF LIMITS EXCEEDED THEN QUIT.
		IF MD2TARGET(MD2MOTORL) > MD2LIMITF(MD2MOTORL) THEN MD2STATUS = "L"
		IF MD2TARGET(MD2MOTORL) < MD2LIMITR(MD2MOTORL) THEN MD2STATUS = "L"
		IF MD2TARGET(MD2MOTORS) > MD2LIMITF(MD2MOTORS) THEN MD2STATUS = "L"
		IF MD2TARGET(MD2MOTORS) < MD2LIMITR(MD2MOTORS) THEN MD2STATUS = "L"
		IF MD2STATUS = "L" THEN EXIT SUB
		'SET THE # OF STEPS.
		MD2STEPSL = INT(((MD2TARGET(MD2MOTORL) - MD2POSITION(MD2MOTORL)) * MD2UNITS(MD2MOTORL)) + .5)
		MD2STEPSS = INT(((MD2TARGET(MD2MOTORS) - MD2POSITION(MD2MOTORS)) * MD2UNITS(MD2MOTORS)) + .5)
		'SET THE DIRECTION.
		IF MD2STEPSL < 0 THEN MD2DIRL = -1 ELSE MD2DIRL = 1
		IF MD2STEPSS < 0 THEN MD2DIRS = -1 ELSE MD2DIRS = 1
		'MAKE STEP COUNTS POSITIVE.
		MD2STEPSL = ABS(MD2STEPSL)
		MD2STEPSS = ABS(MD2STEPSS)
		'SET COMPLETED POSITIONS AND ROUND.
		MD2POSITION(MD2MOTORL) = MD2TARGET(MD2MOTORL)
		MD2POSITION(MD2MOTORL) = INT((MD2POSITION(MD2MOTORL) * MD2UNITS(MD2MOTORL)) + .5)
		MD2POSITION(MD2MOTORL) = MD2POSITION(MD2MOTORL) / MD2UNITS(MD2MOTORL)

		MD2POSITION(MD2MOTORS) = MD2TARGET(MD2MOTORS)
		MD2POSITION(MD2MOTORS) = INT((MD2POSITION(MD2MOTORS) * MD2UNITS(MD2MOTORS)) + .5)
		MD2POSITION(MD2MOTORS) = MD2POSITION(MD2MOTORS) / MD2UNITS(MD2MOTORS)
	END IF

	'SWAP DIRECTIONS IF NEEDED.
	IF MD2HOMEDIR(MD2MOTORL) = -1 THEN MD2DIRL = MD2DIRL * -1
	IF MD2HOMEDIR(MD2MOTORS) = -1 THEN MD2DIRS = MD2DIRS * -1

	'BACKLASH COMPENSATION, ADJUST STEP COUNTS.
	IF MD2DIRL <> MD2LASTDIR(MD2MOTORL) THEN
		MD2STEPSL = MD2STEPSL + INT((MD2BACKLASH(MD2MOTORL) * MD2UNITS(MD2MOTORL)) + .5)
		MD2LASTDIR(MD2MOTORL) = MD2DIRL
	END IF
	IF MD2DIRS <> MD2LASTDIR(MD2MOTORS) THEN
		MD2STEPSS = MD2STEPSS + INT((MD2BACKLASH(MD2MOTORS) * MD2UNITS(MD2MOTORS)) + .5)
		MD2LASTDIR(MD2MOTORS) = MD2DIRS
	END IF

	'SWAP PARAMETERS DEPENDING ON LONGEST/SHORTEST STEP COUNT.
	IF MD2STEPSL < MD2STEPSS THEN
		MD2SWAPL = MD2STEPSL: MD2STEPSL = MD2STEPSS: MD2STEPSS = MD2SWAPL
		MD2SWAPL = MD2MOTORL: MD2MOTORL = MD2MOTORS: MD2MOTORS = MD2SWAPL
		MD2SWAPL = MD2DIRL: MD2DIRL = MD2DIRS: MD2DIRS = MD2SWAPL
	END IF
   
	'INITIALIZE ERROR ACCUMULATOR AND PRECALCULATE VARS FOR INTERPOLATION.
	MD2STEPSL2 = MD2STEPSL + MD2STEPSL
	MD2STEPSS2 = MD2STEPSS + MD2STEPSS
	MD2LIERROR = MD2STEPSS2 - MD2STEPSL
									
	'SET UP PATTERN MASKS.
	IF MD2MOTORL = 1 OR MD2MOTORL = 3 OR MD2MOTORL = 5 THEN
		MD2MASKL = &HF
		MD2MASKS = &HF0
	ELSE
		MD2MASKL = &HF0
		MD2MASKS = &HF
	END IF

	'SETUP RAMPING----------------.

	'IF IN CALIBRATION MODE SET RAMPING TO MAXSPEED OTHERWISE
	'SET RAMPING PARAMETERS NORMALLY.
	IF MD2CALMODE THEN

		MD2MODE = 0     'CONSTANT SPEED.
		MD2DECEL = 0    'NEVER BEGIN DECELERATION.
		MD2DELAYCUR = MD2MAXSPEED(MD2MOTORL)

	ELSE
		'NORMAL RAMPING.
		'FIND MINDELAY AND MAXDELAY BY SEARCHING VELOCITY ARRAY AND
		'INTERPOLATING BETWEEN ELEMENTS.
		'SETUP DELAY COUNT ADJUSMENTS TO BE LINEARIZE RAMP WHILE STEPPING.
	   
		'CONVERT MINSPEED VELOCITY TO DELAY COUNTS.
		FOR MD2I = 2 TO 23     'SEARCH FOR CORRECT ARRAY ELEMENTS.
			IF (MD2MINSPEED(MD2MOTORL) * MD2UNITS(MD2MOTORL)) >= MD2VELOCITY(MD2I) THEN EXIT FOR
		NEXT MD2I
		'INTERPOLATE BETWEEN ELEMENTS.
		MD2WAY = ((MD2MINSPEED(MD2MOTORL) * MD2UNITS(MD2MOTORL)) - MD2VELOCITY(MD2I - 1)) / (MD2VELOCITY(MD2I) - MD2VELOCITY(MD2I - 1))
		MD2MINDELAY = INT((((MD2DELAY(MD2I) - MD2DELAY(MD2I - 1)) * MD2WAY) + MD2DELAY(MD2I - 1)) + .5)

		'CONVERT MAXSPEED VELOCITY TO DELAY COUNTS.
		FOR MD2I = 2 TO 23     'SEARCH FOR CORRECT ARRAY ELEMENTS.
			IF (MD2MAXSPEED(MD2MOTORL) * MD2UNITS(MD2MOTORL)) >= MD2VELOCITY(MD2I) THEN EXIT FOR
		NEXT MD2I
		'INTERPOLATE BETWEEN ELEMENTS.
		MD2WAY = ((MD2MAXSPEED(MD2MOTORL) * MD2UNITS(MD2MOTORL)) - MD2VELOCITY(MD2I - 1)) / (MD2VELOCITY(MD2I) - MD2VELOCITY(MD2I - 1))
		MD2MAXDELAY = INT((((MD2DELAY(MD2I) - MD2DELAY(MD2I - 1)) * MD2WAY) + MD2DELAY(MD2I - 1)) + .5)

		'SET UP RAMPING PARAMETERS.
		IF (MD2SLOPE(MD2MOTORL) <= 0) OR (MD2MINDELAY <= MD2MAXDELAY) THEN

			'CONSTANT SPEED MODE----------------.

			'CONSTANT SPEED MODE.
			MD2MODE = 2

			'CURRENT SPEED.
			MD2DELAYCUR = MD2MAXDELAY

		ELSE

			'RAMPING MOVE-----------------.

			'ACCELERATION MODE.
			MD2MODE = -1

			'CURRENT STARTING SPEED.
			MD2DELAYCUR = MD2MINDELAY

			'SET STEP COUNT TO BEGIN DECELERATION.
			IF MD2STEPSL < ((MD2SLOPE(MD2MOTORL) * MD2UNITS(MD2MOTORL)) * 2) THEN
				'MOVE TOO SHORT TO REACH MAX SPEED.
				MD2DECEL = INT((MD2STEPSL / 2) + .5)
			ELSE
				'MOVE LONG ENOUGH TO REACH MAX SPEED.
				MD2DECEL = INT(MD2SLOPE(MD2MOTORL) * MD2UNITS(MD2MOTORL))
			END IF
		   
			'SET SLOPES.
			MD2DELAYSLOPE = INT(MD2SLOPE(MD2MOTORL) * MD2UNITS(MD2MOTORL))
			MD2DELAYSLOPE2 = MD2DELAYSLOPE + MD2DELAYSLOPE

			'SET AMOUNT OF CHANGE IN DELAY COUNT.
			MD2DELAYCHANGE = MD2MINDELAY - MD2MAXDELAY

			'DIFFERENT CALCULATIONS FOR SLOPE <=1 AND >1.
			IF MD2DELAYCHANGE > MD2DELAYSLOPE THEN
				'SLOPE > 1.
				'SET DELAY ADJUSTMENT THAT OCCURS EVERY STEP.
				MD2DELAYEACH = INT(MD2DELAYCHANGE / MD2DELAYSLOPE)
				'RESET DELAYCHANGE ACCORDING TO PREVIOUS CALC.
				MD2DELAYCHANGE = MD2DELAYCHANGE - (MD2DELAYSLOPE * MD2DELAYEACH)
			ELSE
				'SLOPE <= 1.
				'DON'T ADJUST EVERY STEP.
				MD2DELAYEACH = 0
			END IF
			   
			'PRECALCULATE THIS.
			MD2DELAYCHANGE2 = MD2DELAYCHANGE + MD2DELAYCHANGE
		   
			'PRESET ADJUSTMENT FLAG.
			MD2DELAYFLAG = MD2DELAYCHANGE2 - MD2DELAYSLOPE
		   
		END IF
		
	END IF


	'TURN OFF COM INTERRUPTS?
	IF MD2INTERRUPTS = 0 THEN
		MD2INTREG = INP(&H21)
		OUT &H21, MD2INTREG OR &H98
	END IF

	'INITIALIZE STATUS.
	MD2STATUS = "O"
  
	'MOVE LOOP.
	DO UNTIL MD2STEPSL = 0
		
		'MOVE LONG MOTOR.
	   
		'POINT TO THE NEXT STEP PATTERN.
		MD2PATPTR(MD2MOTORL) = (MD2PATPTR(MD2MOTORL) + MD2DIRL) AND &H7

		'GET STEP PATTERN AND MASK OFF UNNEEDED BITS.
		MD2PATS = MD2STPPAT(MD2PATPTR(MD2MOTORL)) AND MD2MASKL

		'MOVE SHORT MOTOR IF NEEDED.
		IF MD2LIERROR >= 0 THEN
		   
			'POINT TO THE NEXT STEP PATTERN.
			MD2PATPTR(MD2MOTORS) = (MD2PATPTR(MD2MOTORS) + MD2DIRS) AND &H7
	   
			'OR IN SHORT MOTOR PATTERN.
			MD2PATS = MD2PATS OR (MD2STPPAT(MD2PATPTR(MD2MOTORS)) AND MD2MASKS)

			'UPDATE ERROR FOR LINEAR INTERPOLATION.
			MD2LIERROR = MD2LIERROR - MD2STEPSL2
		   
		ELSE

			'DON'T MOVE SHORT.
			MD2PATS = MD2PATS OR (INP(MD2MTRADR) AND MD2MASKS)
	   
		END IF
		   
		'UPDATE ERROR FOR LINEAR INTERPOLATION.
		MD2LIERROR = MD2LIERROR + MD2STEPSS2
	   
		'OUTPUT THE STEP PATTERN TO MOVE THE MOTOR.
		OUT MD2MTRADR, MD2PATS

		'DELAY BETWEEN STEPS.
		FOR MD2DELAYTMP = 1 TO MD2DELAYCUR: NEXT MD2DELAYTMP
	  
		'DECREMENT STEP COUNT.
		MD2STEPSL = MD2STEPSL - 1
					   
		'CALCULATE NEW SPEED.
	   
		IF MD2MODE = 1 THEN   'DECELERATION.

			'BEGIN CONSTANT IF REACHED MIN SPEED.
			IF MD2DELAYCUR >= MD2MINDELAY THEN MD2MODE = 2
		   
			'DECREASE CURRENT SPEED.
			MD2DELAYCUR = MD2DELAYCUR + MD2DELAYEACH
			IF MD2DELAYFLAG >= 0 THEN
				MD2DELAYCUR = MD2DELAYCUR + 1
				MD2DELAYFLAG = MD2DELAYFLAG - MD2DELAYSLOPE2
			END IF
			MD2DELAYFLAG = MD2DELAYFLAG + MD2DELAYCHANGE2

		END IF
	  
		IF MD2MODE = 0 THEN   'CONSTANT THEN DECELERATION.
			'BEGIN DECELERATION BASED ON STEPS REMAINING.
			IF MD2STEPSL <= MD2DECEL THEN MD2MODE = 1
		END IF
	   
		IF MD2MODE = -1 THEN  'ACCELERATION.

			'INCREASE CURRENT SPEED.
			MD2DELAYCUR = MD2DELAYCUR - MD2DELAYEACH
			IF MD2DELAYFLAG >= 0 THEN
				MD2DELAYCUR = MD2DELAYCUR - 1
				MD2DELAYFLAG = MD2DELAYFLAG - MD2DELAYSLOPE2
			END IF
			MD2DELAYFLAG = MD2DELAYFLAG + MD2DELAYCHANGE2

			'BEGIN DECELERATION BASED ON STEPS REMAINING.
			IF MD2STEPSL <= MD2DECEL THEN MD2MODE = 1
		  
			'BEGIN CONSTANT SPEED IF REACHED MAX SPEED.
			IF MD2DELAYCUR <= MD2MAXDELAY THEN MD2MODE = 0
	  
		END IF
	   
	LOOP
   
	'TURN COM INTERRUPTS BACK ON?
	IF MD2INTERRUPTS = 0 THEN OUT &H21, MD2INTREG

	'POWER OFF MOTOR IF DESIRED.
	IF MD2HOLD = 0 THEN
		OUT MD2MTRADR, &HFF
		'DELAY TO ALLOW STEP TO TAKE.
		FOR MD2DELAYTMP = 1 TO MD2MINDELAY: NEXT MD2DELAYTMP
	END IF

	'KEYPRESS?
	INREG.AX = &HB00
	INTERRUPT &H21, INREG, OUTREG
	IF OUTREG.AX > &HB00 THEN
		INREG.AX = &HC00
		INTERRUPT &H21, INREG, OUTREG
		MD2STATUS = "K"
	END IF

END SUB

SUB MD2OFF

	'---------------------------------------------------------------------
	'NAME:      MD2OFF
	'DESC:      THE MD2OFF PROCEDURE RETURNS A PARALLEL PRINTER PORT
	'           REFERENCED BY THE MOTOR # TO ITS PREVIOUS STATE READY
	'           FOR USE WITH A PRINTER AND DISABLES THE MD-2.
	'USAGE:     USE AT THE END OF A MOTION CONTROL PROGRAM.
	'INPUTS:    MOTOR # DETERMINES PORT.
	'OUTPUTS:   NONE.
	'---------------------------------------------------------------------
   
	'LOCAL VARIABLES.
	DIM MD2TIMER AS SINGLE

	'CHECK FOR SETUP.
	IF MD2MTRADR12 <> &H3BC THEN MD2STATUS = "S": EXIT SUB

	'CHECK FOR VALID MOTOR #.
	MD2STATUS = "B"
	IF MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 3 THEN MD2STATUS = "O"
	IF MD2MOTOR = 4 OR MD2MOTOR = 5 OR MD2MOTOR = 6 THEN MD2STATUS = "O"
	IF MD2MOTOR = 12 OR MD2MOTOR = 34 OR MD2MOTOR = 56 THEN MD2STATUS = "O"
	IF MD2STATUS <> "O" THEN EXIT SUB

	'CHECK FOR PORT AVAILABILITY.
	MD2STATUS = "P"
	IF (MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 12) AND NOT MD2AVAILABLE12 THEN EXIT SUB
	IF (MD2MOTOR = 3 OR MD2MOTOR = 4 OR MD2MOTOR = 34) AND NOT MD2AVAILABLE34 THEN EXIT SUB
	IF (MD2MOTOR = 5 OR MD2MOTOR = 6 OR MD2MOTOR = 56) AND NOT MD2AVAILABLE56 THEN EXIT SUB
		 
	'-STROBE PIN HIGH, -ALF PIN HIGH, -INIT PIN LOW, -SELIN PIN LOW, IRQ OFF.
	IF MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 12 THEN
		OUT MD2MTRADR12 + 2, &H4
	END IF
	IF MD2MOTOR = 3 OR MD2MOTOR = 4 OR MD2MOTOR = 34 THEN
		OUT MD2MTRADR34 + 2, &H4
	END IF
	IF MD2MOTOR = 5 OR MD2MOTOR = 6 OR MD2MOTOR = 56 THEN
		OUT MD2MTRADR56 + 2, &H4
	END IF

	'DELAY FOR .2 SECONDS.
	MD2TIMER = TIMER: DO: LOOP UNTIL TIMER > MD2TIMER + .2
   
	'TURN -INIT PIN HIGH.
	IF MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 12 THEN
		OUT MD2MTRADR12 + 2, &HC
		MD2ENABLED12 = 0
	END IF
	IF MD2MOTOR = 3 OR MD2MOTOR = 4 OR MD2MOTOR = 34 THEN
		OUT MD2MTRADR34 + 2, &HC
		MD2ENABLED34 = 0
	END IF
	IF MD2MOTOR = 5 OR MD2MOTOR = 6 OR MD2MOTOR = 56 THEN
		OUT MD2MTRADR56 + 2, &HC
		MD2ENABLED56 = 0
	END IF
															
	MD2STATUS = "O"

END SUB

SUB MD2ON

	'---------------------------------------------------------------------
	'NAME:      MD2ON
	'DESC:      THE MD2ON PROCEDURE INITIALIZES A PARALLEL PRINTER PORT
	'           AND TURNS ON AN MD-2.
	'USAGE:     USE AT THE BEGINNING OF A MOTION CONTROL PROGRAM BUT
	'           AFTER THE MD2SETUP SUBROUTINE.
	'INPUTS:    MOTOR # DETERMINES PORT.
	'OUTPUTS:   NONE.
	'---------------------------------------------------------------------

	'CHECK FOR SETUP.
	IF MD2MTRADR12 <> &H3BC THEN MD2STATUS = "S": EXIT SUB

	'CHECK FOR VALID MOTOR #.
	MD2STATUS = "B"
	IF MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 3 THEN MD2STATUS = "O"
	IF MD2MOTOR = 4 OR MD2MOTOR = 5 OR MD2MOTOR = 6 THEN MD2STATUS = "O"
	IF MD2MOTOR = 12 OR MD2MOTOR = 34 OR MD2MOTOR = 56 THEN MD2STATUS = "O"
	IF MD2STATUS <> "O" THEN EXIT SUB

	'CHECK FOR PORT AVAILABILITY.
	MD2STATUS = "P"
	IF (MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 12) AND NOT MD2AVAILABLE12 THEN EXIT SUB
	IF (MD2MOTOR = 3 OR MD2MOTOR = 4 OR MD2MOTOR = 34) AND NOT MD2AVAILABLE34 THEN EXIT SUB
	IF (MD2MOTOR = 5 OR MD2MOTOR = 6 OR MD2MOTOR = 56) AND NOT MD2AVAILABLE56 THEN EXIT SUB

	'TURN OFF ALL MOTOR PHASES,
	'SET -STROBE LOW (MD-2 ENABLED),
	'    -ALF LOW (OUTPUT #1 OFF HIGH),
	'    -INIT LOW (STANDBY OFF),
	'    -SELIN LOW (OUTPUT #2 OF HIGH),
	'    IRQ DISABLED.
   
	IF MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 12 THEN
		OUT MD2MTRADR12, &HFF
		OUT MD2MTRADR12 + 2, &H5
		MD2ENABLED12 = -1
		MD2STATUS = "O"
	END IF
   
	IF MD2MOTOR = 3 OR MD2MOTOR = 4 OR MD2MOTOR = 34 THEN
		OUT MD2MTRADR34, &HFF
		OUT MD2MTRADR34 + 2, &H5
		MD2ENABLED34 = -1
		MD2STATUS = "O"
	END IF
   
	IF MD2MOTOR = 5 OR MD2MOTOR = 6 OR MD2MOTOR = 56 THEN
		OUT MD2MTRADR56, &HFF
		OUT MD2MTRADR56 + 2, &H5
		MD2ENABLED56 = -1
		MD2STATUS = "O"
	END IF

END SUB

SUB MD2OUTPUTS
 
	'---------------------------------------------------------------------
	'NAME:      MD2OUTPUTS
	'DESC:      THE MD2OUTPUTS PROCEDURE TURNS OUTPUT PINS ON AND OFF.
	'           THE INPUT/OUTPUT PORT ON THE BACK OF THE MD-2 SYSTEM
	'           HAS 2 OUTPUTS AND 3 INPUTS.  SOME MD-2 SYSTEMS DO NOT
	'           HAVE THIS I/O PORT.  CHECK YOUR MANUAL FOR MORE INFORMATION.
	'           WHEN TURNED ON, THE OUTPUT PIN WILL BE DRIVEN TO GROUND
	'           TO TURN ON SINKING LOADS SUCH AS RELAYS WHICH CAN CONTROL
	'           LARGER LOADS.  WHEN TURNED OFF, THEN OUTPUT PIN WILL
	'           RETURN TO A LOGIC 1 WHICH IS 5 VOLTS DC.
	'           MAKE SURE NOT TO EXCEED MAXIMUM CURRENT DRAW ON OUTPUT PINS.
	'           THE MD2OUTPUTCODE DETERMINES WHICH MD-2, PIN AND ON/OFF.
	'           USE THE FOLLOWING CHART TO DETERMINE WHICH CODE TO USE.
	'           NOTICE THE FIRST DIGIT OF THE CODE SELECTS WHICH MD-2 SYSTEM,
	'           THE SECOND DIGIT SELECTS THE PIN, AND THE THIRD SELECTS
	'           ON OR OFF (1 OR 0).
	'
	'           MD2OUTPUTCODE    MD-2 PORT    OUTPUT  ACTION
	'           -------------    ---------    ------  ------
	'               110             3BC         #1      OFF
	'               111             3BC         #1      ON
	'               120             3BC         #2      OFF
	'               121             3BC         #2      ON
	'
	'               210             378         #1      OFF
	'               211             378         #1      ON
	'               220             378         #2      OFF
	'               221             378         #2      ON
	'
	'               310             278         #1      OFF
	'               311             278         #1      ON
	'               320             278         #2      OFF
	'               321             278         #2      ON
	'
	'USAGE:     SET THE MD2OUTPUTCODE PARAMETER AND CALL.  USE TO CONTROL
	'           TOOLING SUCH AS DRILLS AND DISPENSERS.
	'INPUTS:    MD2OUTPUTCODE DETERMINES WHICH, MD-2, WHICH PIN, ON/OFF.
	'OUTPUTS:   NONE.
	'---------------------------------------------------------------------

	'PORT 3BC, OUTPUT #1, OFF
	IF MD2OUTPUTCODE = 110 THEN
		OUT MD2MTRADR12 + 2, INP(MD2MTRADR12 + 2) AND &HF7
	END IF

	'PORT 3BC, OUTPUT #1, ON
	IF MD2OUTPUTCODE = 111 THEN
		OUT MD2MTRADR12 + 2, INP(MD2MTRADR12 + 2) OR &H8
	END IF

	'PORT 3BC, OUTPUT #2, OFF
	IF MD2OUTPUTCODE = 120 THEN
		OUT MD2MTRADR12 + 2, INP(MD2MTRADR12 + 2) AND &HFD
	END IF

	'PORT 3BC, OUTPUT #2, ON
	IF MD2OUTPUTCODE = 121 THEN
		OUT MD2MTRADR12 + 2, INP(MD2MTRADR12 + 2) OR &H2
	END IF
  

	'PORT 378, OUTPUT #1, OFF
	IF MD2OUTPUTCODE = 210 THEN
		OUT MD2MTRADR34 + 2, INP(MD2MTRADR34 + 2) AND &HF7
	END IF

	'PORT 378, OUTPUT #1, ON
	IF MD2OUTPUTCODE = 211 THEN
		OUT MD2MTRADR34 + 2, INP(MD2MTRADR34 + 2) OR &H8
	END IF

	'PORT 378, OUTPUT #2, OFF
	IF MD2OUTPUTCODE = 220 THEN
		OUT MD2MTRADR34 + 2, INP(MD2MTRADR34 + 2) AND &HFD
	END IF

	'PORT 378, OUTPUT #2, ON
	IF MD2OUTPUTCODE = 221 THEN
		OUT MD2MTRADR34 + 2, INP(MD2MTRADR34 + 2) OR &H2
	END IF
  

	'PORT 278, OUTPUT #1, OFF
	IF MD2OUTPUTCODE = 310 THEN
		OUT MD2MTRADR56 + 2, INP(MD2MTRADR56 + 2) AND &HF7
	END IF

	'PORT 278, OUTPUT #1, ON
	IF MD2OUTPUTCODE = 311 THEN
		OUT MD2MTRADR56 + 2, INP(MD2MTRADR56 + 2) OR &H8
	END IF

	'PORT 278, OUTPUT #2, OFF
	IF MD2OUTPUTCODE = 320 THEN
		OUT MD2MTRADR56 + 2, INP(MD2MTRADR56 + 2) AND &HFD
	END IF

	'PORT 278, OUTPUT #2, ON
	IF MD2OUTPUTCODE = 321 THEN
		OUT MD2MTRADR56 + 2, INP(MD2MTRADR56 + 2) OR &H2
	END IF

	MD2STATUS = "O"

END SUB

SUB MD2PAREDIT

	'---------------------------------------------------------------------
	'NAME:      MD2PAREDIT
	'DESC:      THIS PROCEDURE IS USED TO EDIT (CHANGE) ALL MOTOR
	'           PARAMETERS USING THE SCREEN AND KEYBOARD.  THE OPERATOR
	'           MAY VIEW AND MODIFY ANY PARAMETER.
	'USAGE:     SIMPLY CALL ALLOWING THE OPERATOR TO EDIT PARAMETERS.
	'           THIS ROUTINE DISTROYS THE SCREEN REQUIRING REPAINTING.
	'INPUTS:    ALL MOTOR PARAMETERS.
	'OUTPUTS:   ALL MOTOR PARAMETERS.
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	DIM MD2COLUMN AS INTEGER        'COLUMN.
	DIM MD2ROW AS INTEGER           'ROW.
	DIM MD2MTR AS INTEGER           'MOTOR COUNTER.
	DIM MD2ENTRY AS STRING          'STRING ENTERED.
	DIM MD2VALUE AS SINGLE          'VALUE ENTERED.

MD2PAREDITLOOP:
  
	'SETUP THE SCREEN.
	SCREEN 0
	KEY OFF
	COLOR 7, 0
	CLS

	'BLUE BACK GROUND.
	COLOR 7, 1: LOCATE 3, 1: PRINT SPACE$(80 * 18);

	'DISPLAY HEADINGS.
	COLOR 4, 0: LOCATE 1, 2
	PRINT "Arrick Robotics            Edit MD-2 Motor Parameters";
	LOCATE 1, 70: PRINT DATE$;

	COLOR 0, 7: LOCATE 2, 1
	PRINT SPACE$(19); "Motor 1   Motor 2   Motor 3   ";
	PRINT "Motor 4   Motor 5   Motor 6    ";
	LOCATE 3, 1
	PRINT "(B)     Backlash "
	PRINT "(HD)  Home Direc "
	PRINT "(HO) Home Offset "
	PRINT "(LF)   Limit Fwd "
	PRINT "(LR)   Limit Rev "
	PRINT "(MI)   Min Speed "
	PRINT "(MA)   Max Speed "
	PRINT "(MN)  Motor Name "
	PRINT "(P)     Position "
	PRINT "(SL)       Slope "
	PRINT "(T)       Target "
	PRINT "(UN)   Unit Name "
	PRINT "(UV)  Unit Value ";
  
	'DISPLAY MOTOR PARAMETERS.
	COLOR 7, 1
	FOR MD2MTR = 1 TO 6
		LOCATE 3, 9 + (MD2MTR * 10): PRINT MD2BACKLASH(MD2MTR);
		LOCATE 4, 10 + (MD2MTR * 10)
		IF MD2HOMEDIR(MD2MTR) = 0 THEN PRINT "Reverse";  ELSE PRINT "Forward";
		LOCATE 5, 9 + (MD2MTR * 10): PRINT MD2HOMEOFFSET(MD2MTR);
		LOCATE 6, 9 + (MD2MTR * 10): PRINT MD2LIMITF(MD2MTR);
		LOCATE 7, 9 + (MD2MTR * 10): PRINT MD2LIMITR(MD2MTR);
		LOCATE 8, 9 + (MD2MTR * 10): PRINT MD2MINSPEED(MD2MTR);
		LOCATE 9, 9 + (MD2MTR * 10): PRINT MD2MAXSPEED(MD2MTR);
		LOCATE 10, 10 + (MD2MTR * 10): PRINT MD2MOTORNAME(MD2MTR);
		LOCATE 11, 9 + (MD2MTR * 10): PRINT MD2POSITION(MD2MTR);
		LOCATE 12, 9 + (MD2MTR * 10): PRINT MD2SLOPE(MD2MTR);
		LOCATE 13, 9 + (MD2MTR * 10): PRINT MD2TARGET(MD2MTR);
		LOCATE 14, 10 + (MD2MTR * 10): PRINT MD2UNITNAME(MD2MTR);
		LOCATE 15, 9 + (MD2MTR * 10): PRINT MD2UNITS(MD2MTR);
	NEXT MD2MTR

	'WHITE BARS & EDGE.
	LOCATE 16, 1: COLOR 0, 7: PRINT SPACE$(80);
	LOCATE 21, 1: COLOR 0, 7: PRINT SPACE$(80);
	COLOR 0, 7: FOR MD2ROW = 3 TO 20: LOCATE MD2ROW, 80: PRINT " "; : NEXT MD2ROW

	'OTHER PARAMETERS.
	LOCATE 17, 1: COLOR 0, 7: PRINT "(H)         Hold "; :   COLOR 7, 1
	IF MD2HOLD = 0 THEN PRINT " No";  ELSE PRINT " Yes";
	LOCATE 18, 1: COLOR 0, 7: PRINT "(I)   Interrupts "; : COLOR 7, 1
	IF MD2INTERRUPTS = 0 THEN PRINT " No";  ELSE PRINT " Yes";
	LOCATE 19, 1: COLOR 0, 7: PRINT "(MT)   Move Type "; : COLOR 7, 1
	IF MD2MOVETYPE = "A" THEN PRINT " Absolute";
	IF MD2MOVETYPE = "R" THEN PRINT " Relative";
	LOCATE 20, 1: COLOR 0, 7: PRINT "(ST)   Step Type "; :  COLOR 7, 1
	IF MD2STEPTYPE = "H" THEN PRINT " Half Step";
	IF MD2STEPTYPE = "D" THEN PRINT " Double Full Step";
	IF MD2STEPTYPE = "S" THEN PRINT " Single Full Step";
	LOCATE 17, 40: COLOR 0, 7: PRINT "(M)   Active Motor "; :   COLOR 7, 1
	PRINT MD2MOTOR;
	LOCATE 18, 40: COLOR 0, 7: PRINT "(F) Parameter File "; :   COLOR 7, 1
	PRINT " "; MD2PARFILE;
	LOCATE 19, 40: COLOR 0, 7: PRINT "(O)    Output Code "; :   COLOR 7, 1
	PRINT MD2OUTPUTCODE;
	LOCATE 20, 40: COLOR 0, 7: PRINT "            Status "; :   COLOR 7, 1
	PRINT " "; MD2STATUS;

	'OTHER MENU ITEMS.
	LOCATE 25, 1: COLOR 0, 7
	PRINT " (C) Circle     (G) Grid     (L) Load     (S) Save     (PP) Print     (Q) Quit  ";
  
	'GET MENU SELECTION AND ACT.
	LOCATE 23, 27: COLOR 15, 0
	INPUT "Enter Selection Code ", MD2ENTRY
	MD2ENTRY = UCASE$(MD2ENTRY)
	LOCATE 22, 1: PRINT SPACE$(80 * 3); : LOCATE 22, 1
 
	'LIST CALIBRATION ARRAY.
	IF MD2ENTRY = "CAL" THEN
		COLOR 7, 0
		CLS
		COLOR 4, 0: LOCATE 1, 2
		PRINT "Arrick Robotics           Velocity Calibration Array";
		LOCATE 1, 70: PRINT DATE$;
		COLOR 7, 0
		LOCATE 3, 1: PRINT " #   DELAY     VELOCITY";
		LOCATE 3, 40: PRINT " #    DELAY     VELOCITY";
		FOR MD2ROW = 1 TO 24
			IF MD2ROW > 12 THEN
				LOCATE MD2ROW - 9, 40
				PRINT MD2ROW; TAB(45); MD2DELAY(MD2ROW); TAB(55); MD2VELOCITY(MD2ROW);
			ELSE
				LOCATE MD2ROW + 3, 1
				PRINT MD2ROW; TAB(5); MD2DELAY(MD2ROW); TAB(15); MD2VELOCITY(MD2ROW);
			END IF
		NEXT MD2ROW
		LOCATE 22, 25
		INPUT "Press ENTER to continue  ", MD2ENTRY
		GOTO MD2PAREDITLOOP
	END IF


	'LOAD PARAMETERS.
	IF MD2ENTRY = "L" THEN
		COLOR 16, 7: LOCATE 25, 29: PRINT " (L) Load ";
		COLOR 7, 0: LOCATE 23, 1
		INPUT "Enter parameter file name to load ", MD2ENTRY
		IF MD2ENTRY <> "" THEN MD2PARFILE = MD2ENTRY
		MD2PARLOAD
		GOTO MD2PAREDITLOOP
	END IF
  
	'SAVE PARAMETERS.
	IF MD2ENTRY = "S" THEN
		COLOR 16, 7: LOCATE 25, 42: PRINT " (S) Save ";
		COLOR 7, 0: LOCATE 23, 1
		INPUT "Enter parameter file name to save ", MD2ENTRY
		IF MD2ENTRY <> "" THEN MD2PARFILE = MD2ENTRY
		MD2PARSAVE
		GOTO MD2PAREDITLOOP
	END IF
 
	'HOLD.
	IF MD2ENTRY = "H" THEN
		COLOR 16, 7: LOCATE 17, 1: PRINT "(H)         Hold ";
		COLOR 7, 0: LOCATE 23, 1
		INPUT "Hold motors after moves? (Y=Yes, N=No) ", MD2ENTRY
		IF LEFT$(UCASE$(MD2ENTRY), 1) = "Y" THEN MD2HOLD = -1
		IF LEFT$(UCASE$(MD2ENTRY), 1) = "N" THEN MD2HOLD = 0
		GOTO MD2PAREDITLOOP
	END IF
  
	'INTERRUPTS.
	IF MD2ENTRY = "I" THEN
		COLOR 16, 7: LOCATE 18, 1: PRINT "(I)   Interrupts ";
		COLOR 7, 0: LOCATE 23, 1
		INPUT "Leave interrupts on during moves? (Y=Yes, N=No) ", MD2ENTRY
		IF LEFT$(UCASE$(MD2ENTRY), 1) = "Y" THEN MD2INTERRUPTS = -1
		IF LEFT$(UCASE$(MD2ENTRY), 1) = "N" THEN MD2INTERRUPTS = 0
		GOTO MD2PAREDITLOOP
	END IF

	'MOVE TYPE.
	IF MD2ENTRY = "MT" THEN
		COLOR 16, 7: LOCATE 19, 1: PRINT "(MT)   Move Type ";
		COLOR 7, 0: LOCATE 23, 1
		INPUT "Enter move type (A=Absolute, R=Relative) ", MD2ENTRY
		IF LEFT$(UCASE$(MD2ENTRY), 1) = "A" THEN MD2MOVETYPE = "A"
		IF LEFT$(UCASE$(MD2ENTRY), 1) = "R" THEN MD2MOVETYPE = "R"
		GOTO MD2PAREDITLOOP
	END IF

	'STEP TYPE.
	IF MD2ENTRY = "ST" THEN
		COLOR 16, 7: LOCATE 20, 1: PRINT "(ST)   Step Type ";
		COLOR 7, 0: LOCATE 23, 1
		INPUT "Enter step type (H=Half Step, D=Double Full, S=Single Full) ", MD2ENTRY
		IF LEFT$(UCASE$(MD2ENTRY), 1) = "H" THEN MD2STEPTYPE = "H"
		IF LEFT$(UCASE$(MD2ENTRY), 1) = "D" THEN MD2STEPTYPE = "D"
		IF LEFT$(UCASE$(MD2ENTRY), 1) = "S" THEN MD2STEPTYPE = "S"
		GOTO MD2PAREDITLOOP
	END IF

	'ACTIVE MOTOR.
	IF MD2ENTRY = "M" THEN
		COLOR 16, 7: LOCATE 17, 40: PRINT "(M)   Active Motor ";
		COLOR 7, 0: LOCATE 23, 1
		INPUT "Enter active motor # (1-6,12,34,56) ", MD2ENTRY
		IF MD2ENTRY = "1" THEN MD2MOTOR = 1
		IF MD2ENTRY = "2" THEN MD2MOTOR = 2
		IF MD2ENTRY = "3" THEN MD2MOTOR = 3
		IF MD2ENTRY = "4" THEN MD2MOTOR = 4
		IF MD2ENTRY = "5" THEN MD2MOTOR = 5
		IF MD2ENTRY = "6" THEN MD2MOTOR = 6
		IF MD2ENTRY = "12" THEN MD2MOTOR = 12
		IF MD2ENTRY = "34" THEN MD2MOTOR = 34
		IF MD2ENTRY = "56" THEN MD2MOTOR = 56
		GOTO MD2PAREDITLOOP
	END IF

	'PARAMETER FILE.
	IF MD2ENTRY = "F" THEN
		COLOR 16, 7: LOCATE 18, 40: PRINT "(F) Parameter File ";
		COLOR 7, 0: LOCATE 23, 1
		INPUT "Enter parameter file name ", MD2PARFILE
		GOTO MD2PAREDITLOOP
	END IF

	'OUTPUT CODE.
	IF MD2ENTRY = "O" THEN
		COLOR 16, 7: LOCATE 19, 40: PRINT "(O)    Output Code ";
		COLOR 7, 0: LOCATE 23, 1
		INPUT "Enter output code for use with I/O port ", MD2OUTPUTCODE
		GOTO MD2PAREDITLOOP
	END IF
   
	'BACKLASH.
	IF MD2ENTRY = "B" THEN
		COLOR 16, 7: LOCATE 3, 1: PRINT "(B)     Backlash ";
		GOSUB GETMOTOR
		IF MD2MTR = 0 THEN GOTO MD2PAREDITLOOP
		INPUT "Enter backlash due to gear loosness or belt stretch in units ", MD2VALUE
		IF MD2MTR = 99 THEN
			MD2BACKLASH(1) = MD2VALUE
			MD2BACKLASH(2) = MD2VALUE
			MD2BACKLASH(3) = MD2VALUE
			MD2BACKLASH(4) = MD2VALUE
			MD2BACKLASH(5) = MD2VALUE
			MD2BACKLASH(6) = MD2VALUE
			GOTO MD2PAREDITLOOP
		END IF
		MD2BACKLASH(MD2MTR) = MD2VALUE
		GOTO MD2PAREDITLOOP
	END IF
  
	'HOME DIRECTION.
	IF MD2ENTRY = "HD" THEN
		COLOR 16, 7: LOCATE 4, 1: PRINT "(HD)  Home Direc ";
		GOSUB GETMOTOR
		IF MD2MTR = 0 THEN GOTO MD2PAREDITLOOP
		INPUT "Enter direction of home switch (R=reverse, F=forward) ", MD2ENTRY
		MD2ENTRY = UCASE$(MD2ENTRY)
		IF MD2ENTRY <> "R" AND MD2ENTRY <> "F" THEN GOTO MD2PAREDITLOOP
		IF MD2ENTRY = "R" THEN MD2VALUE = 0
		IF MD2ENTRY = "F" THEN MD2VALUE = -1
		IF MD2MTR = 99 THEN
			MD2HOMEDIR(1) = MD2VALUE
			MD2HOMEDIR(2) = MD2VALUE
			MD2HOMEDIR(3) = MD2VALUE
			MD2HOMEDIR(4) = MD2VALUE
			MD2HOMEDIR(5) = MD2VALUE
			MD2HOMEDIR(6) = MD2VALUE
			GOTO MD2PAREDITLOOP
		END IF
		MD2HOMEDIR(MD2MTR) = MD2VALUE
		GOTO MD2PAREDITLOOP
	END IF
  
	'HOME OFFSET.
	IF MD2ENTRY = "HO" THEN
		COLOR 16, 7: LOCATE 5, 1: PRINT "(HO) Home Offset ";
		GOSUB GETMOTOR
		IF MD2MTR = 0 THEN GOTO MD2PAREDITLOOP
		INPUT "Enter home offset distance in Units ", MD2VALUE
		IF MD2MTR = 99 THEN
			MD2HOMEOFFSET(1) = MD2VALUE
			MD2HOMEOFFSET(2) = MD2VALUE
			MD2HOMEOFFSET(3) = MD2VALUE
			MD2HOMEOFFSET(4) = MD2VALUE
			MD2HOMEOFFSET(5) = MD2VALUE
			MD2HOMEOFFSET(6) = MD2VALUE
			GOTO MD2PAREDITLOOP
		END IF
		MD2HOMEOFFSET(MD2MTR) = MD2VALUE
		GOTO MD2PAREDITLOOP
	END IF
 
	'LIMIT FORWARD.
	IF MD2ENTRY = "LF" THEN
		COLOR 16, 7: LOCATE 6, 1: PRINT "(LF)   Limit Fwd ";
		GOSUB GETMOTOR
		IF MD2MTR = 0 THEN GOTO MD2PAREDITLOOP
		INPUT "Enter forward limit in units ", MD2VALUE
		IF MD2MTR = 99 THEN
			MD2LIMITF(1) = MD2VALUE
			MD2LIMITF(2) = MD2VALUE
			MD2LIMITF(3) = MD2VALUE
			MD2LIMITF(4) = MD2VALUE
			MD2LIMITF(5) = MD2VALUE
			MD2LIMITF(6) = MD2VALUE
			GOTO MD2PAREDITLOOP
		END IF
		MD2LIMITF(MD2MTR) = MD2VALUE
		GOTO MD2PAREDITLOOP
	END IF
 
	'LIMIT REVERSE.
	IF MD2ENTRY = "LR" THEN
		COLOR 16, 7: LOCATE 7, 1: PRINT "(LR)   Limit Rev ";
		GOSUB GETMOTOR
		IF MD2MTR = 0 THEN GOTO MD2PAREDITLOOP
		INPUT "Enter reverse limit in units ", MD2VALUE
		IF MD2MTR = 99 THEN
			MD2LIMITR(1) = MD2VALUE
			MD2LIMITR(2) = MD2VALUE
			MD2LIMITR(3) = MD2VALUE
			MD2LIMITR(4) = MD2VALUE
			MD2LIMITR(5) = MD2VALUE
			MD2LIMITR(6) = MD2VALUE
			GOTO MD2PAREDITLOOP
		END IF
		MD2LIMITR(MD2MTR) = MD2VALUE
		GOTO MD2PAREDITLOOP
	END IF
 
	'MINIMUM SPEED.
	IF MD2ENTRY = "MI" THEN
		COLOR 16, 7: LOCATE 8, 1: PRINT "(MI)   Min Speed ";
		GOSUB GETMOTOR
		IF MD2MTR = 0 THEN GOTO MD2PAREDITLOOP
		INPUT "Enter minimum speed (at beginning of acceleration) in units per second ", MD2VALUE
		IF MD2MTR = 99 THEN
			MD2MINSPEED(1) = MD2VALUE
			MD2MINSPEED(2) = MD2VALUE
			MD2MINSPEED(3) = MD2VALUE
			MD2MINSPEED(4) = MD2VALUE
			MD2MINSPEED(5) = MD2VALUE
			MD2MINSPEED(6) = MD2VALUE
			GOTO MD2PAREDITLOOP
		END IF
		MD2MINSPEED(MD2MTR) = MD2VALUE
		GOTO MD2PAREDITLOOP
	END IF

	'MAXIMUM SPEED.
	IF MD2ENTRY = "MA" THEN
		COLOR 16, 7: LOCATE 9, 1: PRINT "(MA)   Max Speed ";
		GOSUB GETMOTOR
		IF MD2MTR = 0 THEN GOTO MD2PAREDITLOOP
		INPUT "Enter maximum speed (after acceleration) in units per second ", MD2VALUE
		IF MD2MTR = 99 THEN
			MD2MAXSPEED(1) = MD2VALUE
			MD2MAXSPEED(2) = MD2VALUE
			MD2MAXSPEED(3) = MD2VALUE
			MD2MAXSPEED(4) = MD2VALUE
			MD2MAXSPEED(5) = MD2VALUE
			MD2MAXSPEED(6) = MD2VALUE
			GOTO MD2PAREDITLOOP
		END IF
		MD2MAXSPEED(MD2MTR) = MD2VALUE
		GOTO MD2PAREDITLOOP
	END IF
   
	'MOTOR NAME.
	IF MD2ENTRY = "MN" THEN
		COLOR 16, 7: LOCATE 10, 1: PRINT "(MN)  Motor Name ";
		GOSUB GETMOTOR
		IF MD2MTR = 0 THEN GOTO MD2PAREDITLOOP
		INPUT "Enter motor name ", MD2ENTRY
		IF MD2MTR = 99 THEN
			MD2MOTORNAME(1) = MD2ENTRY
			MD2MOTORNAME(2) = MD2ENTRY
			MD2MOTORNAME(3) = MD2ENTRY
			MD2MOTORNAME(4) = MD2ENTRY
			MD2MOTORNAME(5) = MD2ENTRY
			MD2MOTORNAME(6) = MD2ENTRY
			GOTO MD2PAREDITLOOP
		END IF
		MD2MOTORNAME(MD2MTR) = MD2ENTRY
		GOTO MD2PAREDITLOOP
	END IF
 
	'POSITION.
	IF MD2ENTRY = "P" THEN
		COLOR 16, 7: LOCATE 11, 1: PRINT "(P)     Position ";
		GOSUB GETMOTOR
		IF MD2MTR = 0 THEN GOTO MD2PAREDITLOOP
		INPUT "Enter current motor position in units ", MD2VALUE
		IF MD2MTR = 99 THEN
			MD2POSITION(1) = MD2VALUE
			MD2POSITION(2) = MD2VALUE
			MD2POSITION(3) = MD2VALUE
			MD2POSITION(4) = MD2VALUE
			MD2POSITION(5) = MD2VALUE
			MD2POSITION(6) = MD2VALUE
			GOTO MD2PAREDITLOOP
		END IF
		MD2POSITION(MD2MTR) = MD2VALUE
		GOTO MD2PAREDITLOOP
	END IF
 
	'SLOPE.
	IF MD2ENTRY = "SL" THEN
		COLOR 16, 7: LOCATE 12, 1: PRINT "(S)        Slope ";
		GOSUB GETMOTOR
		IF MD2MTR = 0 THEN GOTO MD2PAREDITLOOP
		INPUT "Enter acceleration/deceleration slope in units ", MD2VALUE
		IF MD2MTR = 99 THEN
			MD2SLOPE(1) = MD2VALUE
			MD2SLOPE(2) = MD2VALUE
			MD2SLOPE(3) = MD2VALUE
			MD2SLOPE(4) = MD2VALUE
			MD2SLOPE(5) = MD2VALUE
			MD2SLOPE(6) = MD2VALUE
			GOTO MD2PAREDITLOOP
		END IF
		MD2SLOPE(MD2MTR) = MD2VALUE
		GOTO MD2PAREDITLOOP
	END IF
 
	'TARGET.
	IF MD2ENTRY = "T" THEN
		COLOR 16, 7: LOCATE 13, 1: PRINT "(T)       Target ";
		GOSUB GETMOTOR
		IF MD2MTR = 0 THEN GOTO MD2PAREDITLOOP
		INPUT "Enter target position or distance in units ", MD2VALUE
		IF MD2MTR = 99 THEN
			MD2TARGET(1) = MD2VALUE
			MD2TARGET(2) = MD2VALUE
			MD2TARGET(3) = MD2VALUE
			MD2TARGET(4) = MD2VALUE
			MD2TARGET(5) = MD2VALUE
			MD2TARGET(6) = MD2VALUE
			GOTO MD2PAREDITLOOP
		END IF
		MD2TARGET(MD2MTR) = MD2VALUE
		GOTO MD2PAREDITLOOP
	END IF

	'UNIT NAME.
	IF MD2ENTRY = "UN" THEN
		COLOR 16, 7: LOCATE 14, 1: PRINT "(UN)   Unit Name ";
		GOSUB GETMOTOR
		IF MD2MTR = 0 THEN GOTO MD2PAREDITLOOP
		INPUT "Enter unit name ", MD2ENTRY
		IF MD2MTR = 99 THEN
			MD2UNITNAME(1) = MD2ENTRY
			MD2UNITNAME(2) = MD2ENTRY
			MD2UNITNAME(3) = MD2ENTRY
			MD2UNITNAME(4) = MD2ENTRY
			MD2UNITNAME(5) = MD2ENTRY
			MD2UNITNAME(6) = MD2ENTRY
			GOTO MD2PAREDITLOOP
		END IF
		MD2UNITNAME(MD2MTR) = MD2ENTRY
		GOTO MD2PAREDITLOOP
	END IF

	'UNIT.
	IF MD2ENTRY = "UV" THEN
		COLOR 16, 7: LOCATE 15, 1: PRINT "(UV)  Unit Value ";
		GOSUB GETMOTOR
		IF MD2MTR = 0 THEN GOTO MD2PAREDITLOOP
		INPUT "Enter the # of motor steps in each unit ", MD2VALUE
		IF MD2MTR = 99 THEN
			MD2UNITS(1) = MD2VALUE
			MD2UNITS(2) = MD2VALUE
			MD2UNITS(3) = MD2VALUE
			MD2UNITS(4) = MD2VALUE
			MD2UNITS(5) = MD2VALUE
			MD2UNITS(6) = MD2VALUE
			GOTO MD2PAREDITLOOP
		END IF
		MD2UNITS(MD2MTR) = MD2VALUE
		GOTO MD2PAREDITLOOP
	END IF
  
	'LOAD PARAMETERS.
	IF MD2ENTRY = "LP" THEN
		INPUT "Enter motor parameter file name to load ", MD2PARFILE
		IF MD2PARFILE <> "" THEN MD2PARLOAD
		GOTO MD2PAREDITLOOP
	END IF
  
	'SAVE PARAMETERS.
	IF MD2ENTRY = "SP" THEN
		INPUT "Enter motor parameter file name to save ", MD2ENTRY
		IF MD2ENTRY <> "" THEN MD2PARFILE = MD2ENTRY
		MD2PARSAVE
		GOTO MD2PAREDITLOOP
	END IF

	'PRINT.
	IF MD2ENTRY = "PP" THEN
		COLOR 16, 7: LOCATE 25, 56: PRINT "(PP) Print";
		COLOR 7, 0: LOCATE 23, 1
		INPUT "Enter printer port 1=LPT1, 2=LPT2, 3=LPT3 ", MD2ENTRY
		IF MD2ENTRY = "" THEN GOTO MD2PAREDITLOOP
		IF MD2ENTRY = "1" THEN
			MD2MOTOR = 1: MD2OFF
			OPEN "LPT1" FOR OUTPUT AS #1
		END IF
		IF MD2ENTRY = "2" THEN
			MD2MOTOR = 3: MD2OFF
			OPEN "LPT2" FOR OUTPUT AS #1
		END IF
		IF MD2ENTRY = "3" THEN
			MD2MOTOR = 5: MD2OFF
			OPEN "LPT3" FOR OUTPUT AS #1
		END IF
		FOR MD2ROW = 2 TO 19
			FOR MD2COLUMN = 1 TO 80
				PRINT #1, CHR$(SCREEN(MD2ROW, MD2COLUMN));
			NEXT MD2COLUMN
			PRINT #1,
		NEXT MD2ROW
		PRINT #1, CHR$(12);
		CLOSE #1
		GOTO MD2PAREDITLOOP
	END IF
	   
	'EDIT GRID PARAMETERS.
	IF MD2ENTRY = "G" THEN

MD2GRIDLOOP:
	  
		'SETUP SCREEN.
		COLOR 7, 0: CLS

		'BLUE BACKGROUND.
		COLOR 7, 1: FOR MD2ROW = 3 TO 10: LOCATE MD2ROW, 20: PRINT SPACE$(40); : NEXT MD2ROW
	  
		'DISPLAY HEADING.
		COLOR 4, 0: LOCATE 1, 2
		PRINT "Arrick Robotics           Edit MD-2 Grid Parameters";
		LOCATE 1, 70: PRINT DATE$;

		'PARAMETER TITLES.
		COLOR 0, 7
		LOCATE 4, 20: PRINT "(BX)   Begin X ";
		LOCATE 5, 20: PRINT "(BY)   Begin Y ";
		LOCATE 6, 20: PRINT "(SX) Spacing X ";
		LOCATE 7, 20: PRINT "(SY) Spacing Y ";
		LOCATE 8, 20: PRINT "(TX)  Target X ";
		LOCATE 9, 20: PRINT "(TY)  Target Y ";
  
		'WHITE BARS & EDGE.
		LOCATE 3, 20: COLOR 0, 7: PRINT SPACE$(40);
		LOCATE 10, 20: COLOR 0, 7: PRINT SPACE$(40);
		COLOR 0, 7: FOR MD2ROW = 3 TO 10: LOCATE MD2ROW, 60: PRINT " "; : NEXT MD2ROW

		'PARAMETER VALUES.
		COLOR 7, 1
		LOCATE 4, 36: PRINT MD2GRIDBEGINX;
		LOCATE 5, 36: PRINT MD2GRIDBEGINY;
		LOCATE 6, 36: PRINT MD2GRIDSPACEX;
		LOCATE 7, 36: PRINT MD2GRIDSPACEY;
		LOCATE 8, 36: PRINT MD2GRIDTARGETX;
		LOCATE 9, 36: PRINT MD2GRIDTARGETY;
	  
		'GET MENU SELECTION AND ACT.
		LOCATE 16, 20: COLOR 15, 0
		INPUT "Enter Selection Code (Q to Quit) ", MD2ENTRY
		MD2ENTRY = UCASE$(MD2ENTRY)
  
		'QUIT.
		IF MD2ENTRY = "Q" OR MD2ENTRY = "" THEN GOTO MD2PAREDITLOOP
	  
		'BEGIN X.
		IF MD2ENTRY = "BX" THEN
			COLOR 16, 7: LOCATE 4, 20: PRINT "(BX)   Begin X ";
			COLOR 7, 0: LOCATE 18, 20
			INPUT "Enter X axis beginning position in units ", MD2GRIDBEGINX
			GOTO MD2GRIDLOOP
		END IF
	 
		'BEGIN Y.
		IF MD2ENTRY = "BY" THEN
			COLOR 16, 7: LOCATE 5, 20: PRINT "(BY)   Begin Y ";
			COLOR 7, 0: LOCATE 18, 20
			INPUT "Enter Y axis beginning position in units ", MD2GRIDBEGINY
			GOTO MD2GRIDLOOP
		END IF

		'SPACING X.
		IF MD2ENTRY = "SX" THEN
			COLOR 16, 7: LOCATE 6, 20: PRINT "(SX) Spacing X ";
			COLOR 7, 0: LOCATE 18, 20
			INPUT "Enter X axis spacing in units ", MD2GRIDSPACEX
			GOTO MD2GRIDLOOP
		END IF
	
		'SPACING Y.
		IF MD2ENTRY = "SY" THEN
			COLOR 16, 7: LOCATE 7, 20: PRINT "(SY) Spacing Y ";
			COLOR 7, 0: LOCATE 18, 20
			INPUT "Enter Y axis spacing in units ", MD2GRIDSPACEY
			GOTO MD2GRIDLOOP
		END IF

		'TARGET X.
		IF MD2ENTRY = "TX" THEN
			COLOR 16, 7: LOCATE 8, 20: PRINT "(TX)  Target X ";
			COLOR 7, 0: LOCATE 18, 20
			INPUT "Enter X axis target ", MD2GRIDTARGETX
			GOTO MD2GRIDLOOP
		END IF
   
		'TARGET Y.
		IF MD2ENTRY = "TY" THEN
			COLOR 16, 7: LOCATE 9, 20: PRINT "(TY)  Target Y ";
			COLOR 7, 0: LOCATE 18, 20
			INPUT "Enter Y axis target ", MD2GRIDTARGETY
			GOTO MD2GRIDLOOP
		END IF

	END IF

	'EDIT CIRCLE PARAMETERS.
	IF MD2ENTRY = "C" THEN

MD2CIRCLELOOP:
	 
		'SETUP SCREEN.
		COLOR 7, 0: CLS

		'BLUE BACKGROUND.
		COLOR 7, 1: FOR MD2ROW = 3 TO 11: LOCATE MD2ROW, 20: PRINT SPACE$(40); : NEXT MD2ROW
	 
		'DISPLAY HEADING.
		COLOR 4, 0: LOCATE 1, 2
		PRINT "Arrick Robotics            Edit MD-2 Circle Parameters";
		LOCATE 1, 70: PRINT DATE$;

		'PARAMETER TITLES.
		COLOR 0, 7
		LOCATE 4, 20: PRINT "(RX)    Radius X ";
		LOCATE 5, 20: PRINT "(RY)    Radius Y ";
		LOCATE 6, 20: PRINT "(CX)    Center X ";
		LOCATE 7, 20: PRINT "(CY)    Center Y ";
		LOCATE 8, 20: PRINT "(SA) Start Angle ";
		LOCATE 9, 20: PRINT "(AA)   Arc Angle ";
		LOCATE 10, 20: PRINT "(CA) Chord Angle ";
 
		'WHITE BARS & EDGE.
		LOCATE 3, 20: COLOR 0, 7: PRINT SPACE$(40);
		LOCATE 11, 20: COLOR 0, 7: PRINT SPACE$(40);
		COLOR 0, 7: FOR MD2ROW = 3 TO 11: LOCATE MD2ROW, 60: PRINT " "; : NEXT MD2ROW

		'PARAMETER VALUES.
		COLOR 7, 1
		LOCATE 4, 38: PRINT MD2CIRCLERADIUSX;
		LOCATE 5, 38: PRINT MD2CIRCLERADIUSY;
		LOCATE 6, 38: PRINT MD2CIRCLECENTERX;
		LOCATE 7, 38: PRINT MD2CIRCLECENTERY;
		LOCATE 8, 38: PRINT MD2CIRCLESTART;
		LOCATE 9, 38: PRINT MD2CIRCLEARC;
		LOCATE 10, 38: PRINT MD2CIRCLECHORD;
	 
		'GET MENU SELECTION AND ACT.
		LOCATE 16, 20: COLOR 15, 0
		INPUT "Enter Selection Code (Q to Quit) ", MD2ENTRY
		MD2ENTRY = UCASE$(MD2ENTRY)
 
		'QUIT.
		IF MD2ENTRY = "Q" OR MD2ENTRY = "" THEN GOTO MD2PAREDITLOOP
	 
		'RADIUS X.
		IF MD2ENTRY = "RX" THEN
			COLOR 16, 7: LOCATE 4, 20: PRINT "(RX)    Radius X";
			COLOR 7, 0: LOCATE 18, 20
			INPUT "Enter X axis radius in units ", MD2CIRCLERADIUSX
			GOTO MD2CIRCLELOOP
		END IF
	  
		'RADIUS X.
		IF MD2ENTRY = "RY" THEN
			COLOR 16, 7: LOCATE 5, 20: PRINT "(RY)    Radius Y";
			COLOR 7, 0: LOCATE 18, 20
			INPUT "Enter Y axis radius in units ", MD2CIRCLERADIUSY
			GOTO MD2CIRCLELOOP
		END IF
	  
		'CENTER X.
		IF MD2ENTRY = "CX" THEN
			COLOR 16, 7: LOCATE 6, 20: PRINT "(CX)    Center X";
			COLOR 7, 0: LOCATE 18, 20
			INPUT "Enter X axis center in units ", MD2CIRCLECENTERX
			GOTO MD2CIRCLELOOP
		END IF
	 
		'CENTER X.
		IF MD2ENTRY = "CY" THEN
			COLOR 16, 7: LOCATE 7, 20: PRINT "(CY)    Center Y";
			COLOR 7, 0: LOCATE 18, 20
			INPUT "Enter Y axis center in units ", MD2CIRCLECENTERY
			GOTO MD2CIRCLELOOP
		END IF
	  
		'START ANGLE.
		IF MD2ENTRY = "SA" THEN
			COLOR 16, 7: LOCATE 8, 20: PRINT "(SA) Start Angle";
			COLOR 7, 0: LOCATE 18, 20
			INPUT "Enter start angle in degrees ", MD2CIRCLESTART
			GOTO MD2CIRCLELOOP
		END IF
	  
		'ARC ANGLE.
		IF MD2ENTRY = "AA" THEN
			COLOR 16, 7: LOCATE 9, 20: PRINT "(AA)   Arc Angle";
			COLOR 7, 0: LOCATE 18, 20
			INPUT "Enter arc angle in degrees ", MD2CIRCLEARC
			GOTO MD2CIRCLELOOP
		END IF

		'CHORD ANGLE.
		IF MD2ENTRY = "CA" THEN
			COLOR 16, 7: LOCATE 10, 20: PRINT "(CA) Chord Angle";
			COLOR 7, 0: LOCATE 18, 20
			INPUT "Enter chord angle in degrees ", MD2CIRCLECHORD
			GOTO MD2CIRCLELOOP
		END IF

	END IF
			   
	'QUIT.
	IF MD2ENTRY = "Q" OR MD2ENTRY = "" THEN COLOR 7, 0: EXIT SUB

	GOTO MD2PAREDITLOOP

'GET MOTOR NUMBER.
GETMOTOR:
	COLOR 7, 0: LOCATE 23, 1
	INPUT "Enter motor number 1,2,3,4,5,6 or All (0 to quit) ", MD2ENTRY
	MD2MTR = 0
	COLOR 16, 7
	IF MD2ENTRY = "1" THEN MD2MTR = 1: LOCATE 2, 20: PRINT "Motor 1";
	IF MD2ENTRY = "2" THEN MD2MTR = 2: LOCATE 2, 30: PRINT "Motor 2";
	IF MD2ENTRY = "3" THEN MD2MTR = 3: LOCATE 2, 40: PRINT "Motor 3";
	IF MD2ENTRY = "4" THEN MD2MTR = 4: LOCATE 2, 50: PRINT "Motor 4";
	IF MD2ENTRY = "5" THEN MD2MTR = 5: LOCATE 2, 60: PRINT "Motor 5";
	IF MD2ENTRY = "6" THEN MD2MTR = 6: LOCATE 2, 70: PRINT "Motor 6";
	IF MD2ENTRY = "0" THEN MD2MTR = 0
	IF LEFT$(UCASE$(MD2ENTRY), 1) = "A" THEN
		MD2MTR = 99
		LOCATE 2, 20
		PRINT "Motor 1   Motor 2   Motor 3   ";
		PRINT "Motor 4   Motor 5   Motor 6    ";
	END IF
	LOCATE 23, 1: COLOR 7, 0: PRINT SPACE$(80); : LOCATE 23, 1
	RETURN

END SUB

SUB MD2PARLOAD

	'---------------------------------------------------------------------
	'NAME:      MD2PARLOAD
	'DESC:      THIS PROCEDURE IS USED TO LOAD ALL MOTOR PARAMETERS
	'           FROM A DISK FILE.  WHEN USED WITH THE MD2PARSAVE PROCEDURE,
	'           A PROGRAM CAN BE CREATED WHICH SAVES ITS STATUS BEFORE
	'           POWERING DOWN AND RESTORES WHEN COMMING BACK ON ELIMINATING
	'           THE NEED FOR THE USER TO RE-ENTER ALL MOTOR PARAMETERS.
	'USAGE:     SET THE DESIRED FILE NAME IN MD2PARFILE AND CALL.
	'INPUTS:    FILE NAME AND ALL MOTOR PARAMETERS.
	'OUTPUTS:   STATUS = O FOR OK OR E FOR ERROR.
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	DIM MD2MTR AS INTEGER      'MOTOR COUNTER.
	DIM MD2XS AS STRING        'DUMMY STRING VARIABLE.
	DIM MD2XI AS INTEGER       'DUMMY INTEGER VARIABLE.
	DIM MD2FILENUM AS INTEGER  'FREE FILE NUMBER.

	'DEFAULT RETURN STATUS TO OK.
	MD2STATUS = "O"
  
	'LOAD PARAMETERS.
	ON ERROR GOTO MD2ERROR
	MD2FILENUM = FREEFILE
	OPEN MD2PARFILE FOR INPUT AS MD2FILENUM
   
	'MOTOR PARAMETERS.
	INPUT #MD2FILENUM, MD2XS     'FILE NAME
	INPUT #MD2FILENUM, MD2HOLD
	INPUT #MD2FILENUM, MD2INTERRUPTS
	INPUT #MD2FILENUM, MD2MOTOR
	INPUT #MD2FILENUM, MD2MOVETYPE
	INPUT #MD2FILENUM, MD2STEPTYPE

	'CHECK FOR ERRORS INDICATING CORRUPT PARAMETER FILE.
	IF MD2HOLD <> 0 AND MD2HOLD <> -1 THEN MD2STATUS = "F"
	IF MD2MOVETYPE <> "A" AND MD2MOVETYPE <> "R" THEN MD2STATUS = "F"
   
	'MOTOR PARAMETER ARRAYS.
	FOR MD2MTR = 1 TO 6
		IF MD2STATUS <> "O" THEN EXIT FOR
		INPUT #MD2FILENUM, MD2XI
		INPUT #MD2FILENUM, MD2BACKLASH(MD2MTR)
		INPUT #MD2FILENUM, MD2HOMEDIR(MD2MTR)
		INPUT #MD2FILENUM, MD2HOMEOFFSET(MD2MTR)
		INPUT #MD2FILENUM, MD2LIMITF(MD2MTR)
		INPUT #MD2FILENUM, MD2LIMITR(MD2MTR)
		INPUT #MD2FILENUM, MD2MAXSPEED(MD2MTR)
		IF MD2MAXSPEED(MD2MTR) < 0 THEN MD2STATUS = "F"
		INPUT #MD2FILENUM, MD2MINSPEED(MD2MTR)
		IF MD2MINSPEED(MD2MTR) < 0 THEN MD2STATUS = "F"
		INPUT #MD2FILENUM, MD2MOTORNAME(MD2MTR)
		INPUT #MD2FILENUM, MD2POSITION(MD2MTR)
		INPUT #MD2FILENUM, MD2SLOPE(MD2MTR)
		IF MD2SLOPE(MD2MTR) < 0 THEN MD2STATUS = "F"
		INPUT #MD2FILENUM, MD2TARGET(MD2MTR)
		INPUT #MD2FILENUM, MD2UNITS(MD2MTR)
		IF MD2UNITS(MD2MTR) < 0 THEN MD2STATUS = "F"
		INPUT #MD2FILENUM, MD2UNITNAME(MD2MTR)
		INPUT #MD2FILENUM, MD2PATPTR(MD2MTR)
		INPUT #MD2FILENUM, MD2LASTDIR(MD2MTR)
	NEXT MD2MTR
   
	'GRID PARAMETERS.
	INPUT #MD2FILENUM, MD2GRIDBEGINX
	INPUT #MD2FILENUM, MD2GRIDBEGINY
	INPUT #MD2FILENUM, MD2GRIDSPACEX
	INPUT #MD2FILENUM, MD2GRIDSPACEY
	INPUT #MD2FILENUM, MD2GRIDTARGETX
	INPUT #MD2FILENUM, MD2GRIDTARGETY

	'CIRCLE PARAMETERS.
	INPUT #MD2FILENUM, MD2CIRCLERADIUSX
	INPUT #MD2FILENUM, MD2CIRCLERADIUSY
	INPUT #MD2FILENUM, MD2CIRCLECENTERX
	INPUT #MD2FILENUM, MD2CIRCLECENTERY
	INPUT #MD2FILENUM, MD2CIRCLESTART
	INPUT #MD2FILENUM, MD2CIRCLEARC
	INPUT #MD2FILENUM, MD2CIRCLECHORD
   
	'FINISH UP.
	CLOSE MD2FILENUM
	ON ERROR GOTO 0

END SUB

SUB MD2PARSAVE

	'---------------------------------------------------------------------
	'NAME:      MD2PARSAVE
	'DESC:      THIS PROCEDURE IS USED TO SAVE ALL MOTOR PARAMETERS
	'           TO A DISK FILE.  WHEN USED WITH THE MD2PARLOAD PROCEDURE,
	'           A PROGRAM CAN BE CREATED WHICH SAVES ITS STATUS BEFORE
	'           POWERING DOWN AND RESTORES WHEN COMMING BACK ON ELIMINATING
	'           THE NEED FOR THE USER TO RE-ENTER ALL MOTOR PARAMETERS.
	'USAGE:     SET THE DESIRED FILE NAME IN MD2PARFILE AND CALL.
	'INPUTS:    FILE NAME AND ALL MOTOR PARAMETERS.
	'OUTPUTS:   STATUS = O FOR OK OR E FOR ERROR.
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	DIM MD2MTR AS INTEGER           'MOTOR COUNTER.
	DIM MD2FILENUM AS INTEGER       'FREE FILE NUMBER.
 
	'DEFAULT RETURN STATUS TO OK.
	MD2STATUS = "O"
 
	'SAVE PARAMETERS.
	ON ERROR GOTO MD2ERROR
	MD2FILENUM = FREEFILE
	OPEN MD2PARFILE FOR OUTPUT AS MD2FILENUM
   
	'MOTOR PARAMETERS.
	WRITE #MD2FILENUM, MD2PARFILE
	WRITE #MD2FILENUM, MD2HOLD, MD2INTERRUPTS, MD2MOTOR, MD2MOVETYPE, MD2STEPTYPE
   
	'MOTOR PARAMETER ARRAYS.
	FOR MD2MTR = 1 TO 6
		IF MD2STATUS = "F" THEN EXIT FOR
		PRINT #MD2FILENUM, MD2MTR; ",";
		PRINT #MD2FILENUM, MD2BACKLASH(MD2MTR); ",";
		PRINT #MD2FILENUM, MD2HOMEDIR(MD2MTR); ",";
		PRINT #MD2FILENUM, MD2HOMEOFFSET(MD2MTR); ",";
		PRINT #MD2FILENUM, MD2LIMITF(MD2MTR); ",";
		PRINT #MD2FILENUM, MD2LIMITR(MD2MTR); ",";
		PRINT #MD2FILENUM, MD2MAXSPEED(MD2MTR); ",";
		PRINT #MD2FILENUM, MD2MINSPEED(MD2MTR); ",";
		PRINT #MD2FILENUM, CHR$(34); MD2MOTORNAME(MD2MTR); CHR$(34); ",";
		PRINT #MD2FILENUM, MD2POSITION(MD2MTR); ",";
		PRINT #MD2FILENUM, MD2SLOPE(MD2MTR); ",";
		PRINT #MD2FILENUM, MD2TARGET(MD2MTR); ",";
		PRINT #MD2FILENUM, MD2UNITS(MD2MTR); ",";
		PRINT #MD2FILENUM, CHR$(34); MD2UNITNAME(MD2MTR); CHR$(34); ",";
		PRINT #MD2FILENUM, MD2PATPTR(MD2MTR); ",";
		PRINT #MD2FILENUM, MD2LASTDIR(MD2MTR)
	NEXT MD2MTR
  
	'GRID PARAMETERS.
	PRINT #MD2FILENUM, MD2GRIDBEGINX; ",";
	PRINT #MD2FILENUM, MD2GRIDBEGINY; ",";
	PRINT #MD2FILENUM, MD2GRIDSPACEX; ",";
	PRINT #MD2FILENUM, MD2GRIDSPACEY; ",";
	PRINT #MD2FILENUM, MD2GRIDTARGETX; ",";
	PRINT #MD2FILENUM, MD2GRIDTARGETY

	'CIRCLE PARAMETERS.
	PRINT #MD2FILENUM, MD2CIRCLERADIUSX; ",";
	PRINT #MD2FILENUM, MD2CIRCLERADIUSY; ",";
	PRINT #MD2FILENUM, MD2CIRCLECENTERX; ",";
	PRINT #MD2FILENUM, MD2CIRCLECENTERY; ",";
	PRINT #MD2FILENUM, MD2CIRCLESTART; ",";
	PRINT #MD2FILENUM, MD2CIRCLEARC; ",";
	PRINT #MD2FILENUM, MD2CIRCLECHORD

	'FINISH UP.
	CLOSE MD2FILENUM
	ON ERROR GOTO 0
   
END SUB

SUB MD2SEQLOAD

	'---------------------------------------------------------------------
	'NAME:      MD2SEQLOAD
	'DESC:      THIS PROCEDURE IS USED TO LOAD A MOTION SEQUENCE
	'           FROM A DISK FILE AND STORES IN MD2SEQUENCE STRING.
	'           THIS SEQUENCE FILE CAN BE EXECUTED USING THE MD2SEQRUN
	'           SUBROUTINE.
	'USAGE:     SET THE DESIRED FILE NAME IN MD2SEQFILE AND CALL.
	'INPUTS:    FILE NAME AND ALL MOTOR PARAMETERS.
	'OUTPUTS:   STATUS = O FOR OK OR E FOR ERROR.
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	DIM MD2FILENUM AS INTEGER       'FREE FILE NUMBER.
 
	'INITIALIZE VARIABLES.
	MD2STATUS = "O"                 'DEFAULT RETURN STATUS TO OK.
	MD2SEQUENCE = ""                'CLEAR COMMANDS.
 
	'LOAD SEQUENCE FILE.
	ON ERROR GOTO MD2ERROR
	MD2FILENUM = FREEFILE
	OPEN MD2SEQFILE FOR INPUT AS MD2FILENUM
	MD2SEQUENCE = INPUT$(LOF(MD2FILENUM), MD2FILENUM)

	'FINISH UP.
	CLOSE MD2FILENUM
	ON ERROR GOTO 0

END SUB

SUB MD2SEQRUN

	'---------------------------------------------------------------------
	'NAME:      MD2SEQRUN
	'DESC:      THIS PROCEDURE WILL EXECUTE A SEQUENCE OF COMMANDS
	'           WHICH ARE STORED IN THE MD2SEQUENCE VARIABLE.
	'           THE LENGTH OF THE SEQUENCE IS LIMITED TO 32,766.
	'           COMMANDS ARE LIKE BASIC LANGUAGE COMMANDS SUCH AS
	'           MD2POSITION(2)=1000 OR SIMPLY MD2MOVE.  ALLOWS
	'           MOTOR PARAMETERS TO BE SET AND MD-2 SUBROUTINES
	'           TO BE RUN SUCH AS MD2MOVE, MD2HOME, MD2PARLOAD, ETC.
	'           EACH INDIVIDUAL COMMAND IS EXTRACTED FROM THE MD2SEQUENCE
	'           VARIABLE AND PASSED TO MD2SEQRUNDO FOR EXECUTION.
	'           EXECUTION BEGINS WITH MD2POINTER.
	'USAGE:     LOAD MD2SEQUENCE, SET MD2POINTER, THEN CALL MD2SEQRUN.
	'INPUTS:    MD2SEQUENCE,MD2POINTER AND ALL MOTOR PARAMETERS.
	'OUTPUTS:   ALL MOTOR PARAMETERS, MD2STATUS, MD2POINTER.
	'           DISTROYS MD2COMMAND.
	'           WHEN DONE, MD2POINTER POINTS TO THE BEGINNING OF THE
	'           COMMAND AFTER THE LAST ONE EXECUTED.
	'           IF AN ERROR OCCURED (MD2STATUS<>"O") THEN MD2COMMAND
	'           CONTAINS THE COMMAND WITH THE ERROR.
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	DIM MD2CHAR AS STRING           'CURRENT CHARACTER.
	DIM MD2PTR AS INTEGER           'COMMAND POINTER.
	DIM MD2CMD AS STRING            'WORKING COMMAND.
	DIM MD2TIMETEMP AS SINGLE       'TIMER STORAGE.
	DIM MD2NUMBER AS STRING         'VALID NUMBER CHECK.
	DIM MD2OPENP AS INTEGER         'OPEN PAREN LOCATION.
	DIM MD2CLOSEDP AS INTEGER       'CLOSED PAREN LOCATION.
	DIM MD2EQUAL AS INTEGER         'EQUAL SIGN LOCATION.
	DIM MD2VALUE AS STRING          'VALID VALUE.
	DIM MD2SUBSCRIPT AS STRING      'VALID SUBSCRIPT.
	DIM MD2VALIDCHAR AS INTEGER     'VALID CHARACTER COUNTER.
	
	'INITIALIZE.
	MD2COMMAND = ""                 'CLEAR COMMAND.
	MD2STATUS = "B"                 'BAD COMMAND.

	'CHECK FOR EMPTY SEQUENCE AND INVALID POINTER.
	IF MD2SEQUENCE = "" THEN EXIT SUB
	IF MD2POINTER = 0 THEN MD2POINTER = 1 
	IF MD2POINTER > LEN(MD2SEQUENCE) THEN EXIT SUB

'EXTRACT COMMANDS AND SEND TO MD2SEQRUNDO.
MD2SEQRUNEXTRACT:

	'GET THE CURRENT CHARACTER.
	MD2CHAR = MID$(MD2SEQUENCE, MD2POINTER, 1)

	'POINT TO NEXT CHARACTER.
	MD2POINTER = MD2POINTER + 1

	'IF COMMENT THEN IGNORE REST OF LINE.
	'POINTER = NEXT CHARACTER WHEN DONE.
	IF MD2CHAR = "'" THEN
		DO
			'END OF SEQUENCE?
			IF MD2POINTER > LEN(MD2SEQUENCE) THEN EXIT DO
		   
			'GET CHARACTER.
			MD2CHAR = MID$(MD2SEQUENCE, MD2POINTER, 1)
		   
			'POINT TO NEXT CHARACTER.
			MD2POINTER = MD2POINTER + 1
		   
			'DONE WITH COMMENT IF CR, LF, FF.
			IF MD2CHAR = CHR$(10) THEN EXIT DO
			IF MD2CHAR = CHR$(12) THEN EXIT DO
			IF MD2CHAR = CHR$(13) THEN EXIT DO
		LOOP
	END IF

	'END-OF-COMMAND CHARACTER?
	IF MD2CHAR = CHR$(10) THEN GOTO MD2SEQRUNSEND
	IF MD2CHAR = CHR$(12) THEN GOTO MD2SEQRUNSEND
	IF MD2CHAR = CHR$(13) THEN GOTO MD2SEQRUNSEND
	IF MD2CHAR = ":" THEN GOTO MD2SEQRUNSEND
   
	'ADD CHARACTER.
	MD2COMMAND = MD2COMMAND + MD2CHAR
   
	'END OF SEQUENCE?
	IF MD2POINTER > LEN(MD2SEQUENCE) THEN GOTO MD2SEQRUNSEND
   
	GOTO MD2SEQRUNEXTRACT


'DO THE COMMAND, CHECK FOR ERRORS, END AND SETUP FOR NEXT ONE.
'
MD2SEQRUNSEND:

	'DO THE COMMAND.
	GOSUB MD2SEQRUNDO

	'CHECK FOR ERRORS.
	IF MD2STATUS <> "O" THEN EXIT SUB

	'END?
	IF MD2CMD = "END" THEN EXIT SUB

	'END OF SEQUENCE?
	IF MD2POINTER > LEN(MD2SEQUENCE) THEN EXIT SUB

	'SETUP FOR NEXT COMMAND.
	MD2COMMAND = ""
	GOTO MD2SEQRUNEXTRACT

'PARS EQUATION----------------------------------------------------
'RETURN MD2SUBSCRIPT INSIDE (), IF NOT AVAILABLE OR NOT 1-6 THEN = "".
'RETURN MD2VALUE AFTER =, IF NOT AVAIABLE OR VALID THEN = "".

MD2SEQRUNPARS:
	'INITIALIZE VARIABLES.
	MD2VALUE = ""
	MD2SUBSCRIPT = ""

	'GET CHARACTERS AFTER =. CHECK FOR VALIDITY AND EMPTYNESS.
	MD2EQUAL = INSTR(MD2CMD, "=")
	IF MD2EQUAL <> 0 THEN
		MD2NUMBER = RIGHT$(MD2CMD, LEN(MD2CMD) - MD2EQUAL)
		GOSUB MD2SEQRUNVALID
		IF MD2NUMBER <> "" THEN MD2VALUE = MD2NUMBER
	END IF

	'GET CHARACTERS INSIDE ().

	'IF NO ( THEN DONE.
	MD2OPENP = INSTR(MD2CMD, "(")
	IF MD2OPENP = 0 THEN RETURN

	'IF NO ) THEN ERROR. IF ) BEFORE ( THEN ERROR, IF EMPTY THEN ERROR.
	MD2CLOSEDP = INSTR(MD2CMD, ")")
	IF MD2CLOSEDP = 0 THEN RETURN
	IF MD2CLOSEDP < MD2OPENP THEN RETURN
	IF MD2CLOSEDP = MD2OPENP + 1 THEN RETURN

	'EXTRACT THE SUBSCRIPT.
	MD2NUMBER = MID$(MD2CMD, MD2OPENP + 1, (MD2CLOSEDP - MD2OPENP) - 1)

	'CHECK FOR VALID NUMBER.
	GOSUB MD2SEQRUNVALID
	IF MD2NUMBER = "" THEN RETURN

	'IF NOT AN INTEGER THEN ERROR.
	IF VAL(MD2NUMBER) <> INT(VAL(MD2NUMBER)) THEN RETURN

	'IF NOT 1-6 THEN ERROR.
	IF VAL(MD2NUMBER) < 1 OR VAL(MD2NUMBER) > 6 THEN RETURN

	MD2SUBSCRIPT = MD2NUMBER
	RETURN


'VALID NUMBER?----------------------------------------------
'RETURN MD2NUMBER,  IF NOT VALID THEN MD2NUMBER=""
'
MD2SEQRUNVALID:
	IF MD2NUMBER = "" THEN RETURN
	FOR MD2VALIDCHAR = 1 TO LEN(MD2NUMBER)
		'GREATER THAN 57 ISN'T NUMBERS.
		IF ASC(MID$(MD2NUMBER, MD2VALIDCHAR, 1)) > 57 THEN
			MD2NUMBER = ""
			EXIT FOR
		END IF
		'LESS THAN 45 ISN'T NUMBERS BUT LEAVES - AND .
		IF ASC(MID$(MD2NUMBER, MD2VALIDCHAR, 1)) < 45 THEN
			MD2NUMBER = ""
			EXIT FOR
		END IF
		'IS IT A / ?
		IF ASC(MID$(MD2NUMBER, MD2VALIDCHAR, 1)) = 47 THEN
			MD2NUMBER = ""
			EXIT FOR
		END IF
	NEXT MD2VALIDCHAR
	RETURN
		  

'-----------------------------------------------------------------------
'CLEAN COMMAND, AND EXECUTE.
'
MD2SEQRUNDO:

	'DEFAULT TO BAD PARAMETER.
	MD2STATUS = "B"

	'CHECK FOR EMPTY COMMAND.
	IF MD2COMMAND = "" THEN MD2STATUS = "O": RETURN
  
	'REMOVE WHITE SPACES AND DOUBLE QUOTES.
	MD2CMD = ""
	FOR MD2PTR = 1 TO LEN(MD2COMMAND)
	  
		'GET CURRENT CHARACTER AND CONVERT TO UPPER CASE.
		MD2CHAR = UCASE$(MID$(MD2COMMAND, MD2PTR, 1))

		'IGNORE CONTROL CHARS, QUOTES, GRAPHICS.
		IF ASC(MD2CHAR) > 34 AND ASC(MD2CHAR) < 128 THEN
			MD2CMD = MD2CMD + MD2CHAR
		END IF

	NEXT MD2PTR
  
	'EXTRACT NUMBER AFTER = AND SUBSCRIPT IF AVAILABLE.
	GOSUB MD2SEQRUNPARS
	
	'SUBROUTINES ---------------------------------------------------
   
	IF MD2CMD = "" THEN MD2STATUS = "O": RETURN
	IF MD2CMD = "MD2MOVE" THEN MD2MOVE: RETURN
	IF MD2CMD = "MD2HOME" THEN MD2HOME: RETURN
	IF MD2CMD = "MD2ON" THEN MD2ON: RETURN
	IF MD2CMD = "MD2OFF" THEN MD2OFF: RETURN
	IF MD2CMD = "MD2SETUP" THEN MD2SETUP: RETURN
	IF MD2CMD = "MD2PARLOAD" THEN MD2PARLOAD: RETURN
	IF MD2CMD = "MD2PARSAVE" THEN MD2PARSAVE: RETURN
	IF MD2CMD = "MD2OUTPUTS" THEN MD2OUTPUTS: RETURN
	IF MD2CMD = "MD2CIRCLE" THEN MD2CIRCLE: RETURN
	IF MD2CMD = "MD2GRID" THEN MD2GRID: RETURN
	IF MD2CMD = "MD2STANDBYON" THEN MD2STANDBYON: RETURN
	IF MD2CMD = "MD2STANDBYOFF" THEN MD2STANDBYOFF: RETURN

	'COMMANDS -------------------------------------------------------

	'END.
	IF MD2CMD = "END" THEN
		MD2STATUS = "O": RETURN
	END IF

	'SLEEP.
	IF LEFT$(MD2CMD, 5) = "SLEEP" THEN
		IF VAL(RIGHT$(MD2CMD, LEN(MD2CMD) - 5)) = 0 THEN
			'Wait for Keypress (Sleep 0).
			DO
				INREG.AX = &HB00: INTERRUPT &H21, INREG, OUTREG
				IF OUTREG.AX > &HB00 THEN
					INREG.AX = &HC00: INTERRUPT &H21, INREG, OUTREG
					EXIT DO
				END IF
			LOOP
		ELSE
			'Wait second(s).
			MD2TIMETEMP = TIMER: DO: LOOP UNTIL TIMER > MD2TIMETEMP + VAL(RIGHT$(MD2CMD, LEN(MD2CMD) - 5))
		END IF
		MD2STATUS = "O": RETURN
	END IF
   
	'BEEP.
	IF LEFT$(MD2CMD, 4) = "BEEP" THEN
		BEEP
		MD2STATUS = "O": RETURN
	END IF
   
	'SET NON-ARRAY PARAMETERS ----------------------------------------
  
	'MOTOR.
	IF LEFT$(MD2CMD, 9) = "MD2MOTOR=" THEN
		IF MD2VALUE = "" THEN RETURN
		MD2MOTOR = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
  
	'INTERRUPTS.
	IF LEFT$(MD2CMD, 14) = "MD2INTERRUPTS=" THEN
		IF MD2VALUE = "" THEN RETURN
		MD2INTERRUPTS = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
  
	'HOLD.
	IF LEFT$(MD2CMD, 8) = "MD2HOLD=" THEN
		IF MD2VALUE = "" THEN RETURN
		MD2HOLD = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
  
	'MOVE TYPE.
	IF LEFT$(MD2CMD, 12) = "MD2MOVETYPE=" THEN
		MD2MOVETYPE = RIGHT$(MD2CMD, LEN(MD2CMD) - 12)
		MD2STATUS = "O"
		RETURN
	END IF
  
	'STEP TYPE.
	IF LEFT$(MD2CMD, 12) = "MD2STEPTYPE=" THEN
		MD2STEPTYPE = RIGHT$(MD2CMD, LEN(MD2CMD) - 12)
		MD2STATUS = "O"
		RETURN
	END IF
 
	'OUTPUT CODE.
	IF LEFT$(MD2CMD, 14) = "MD2OUTPUTCODE=" THEN
		IF MD2VALUE = "" THEN RETURN
		MD2OUTPUTCODE = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
  
	'MD2PARFILE.
	IF LEFT$(MD2CMD, 11) = "MD2PARFILE=" THEN
		MD2PARFILE = RIGHT$(MD2CMD, LEN(MD2CMD) - 11)
		MD2STATUS = "O"
		RETURN
	END IF
   
	'MD2CALFILE.
	IF LEFT$(MD2CMD, 11) = "MD2CALFILE=" THEN
		MD2CALFILE = RIGHT$(MD2CMD, LEN(MD2CMD) - 11)
		MD2STATUS = "O"
		RETURN
	END IF
   
	'MD2CIRCLERADIUSX.
	IF LEFT$(MD2CMD, 17) = "MD2CIRCLERADIUSX=" THEN
		IF MD2VALUE = "" THEN RETURN
		MD2CIRCLERADIUSX = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
			 
	'MD2CIRCLERADIUSY.
	IF LEFT$(MD2CMD, 17) = "MD2CIRCLERADIUSY=" THEN
		IF MD2VALUE = "" THEN RETURN
		MD2CIRCLERADIUSY = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
			
	'MD2CIRCLECENTERX.
	IF LEFT$(MD2CMD, 17) = "MD2CIRCLECENTERX=" THEN
		IF MD2VALUE = "" THEN RETURN
		MD2CIRCLECENTERX = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
			
	'MD2CIRCLECENTERY.
	IF LEFT$(MD2CMD, 17) = "MD2CIRCLECENTERY=" THEN
		IF MD2VALUE = "" THEN RETURN
		MD2CIRCLECENTERY = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
   
	'MD2CIRCLESTART.
	IF LEFT$(MD2CMD, 15) = "MD2CIRCLESTART=" THEN
		IF MD2VALUE = "" THEN RETURN
		MD2CIRCLESTART = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
		   
	'MD2CIRCLEARC.
	IF LEFT$(MD2CMD, 13) = "MD2CIRCLEARC=" THEN
		IF MD2VALUE = "" THEN RETURN
		MD2CIRCLEARC = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
		  
	'MD2CIRCLECHORD.
	IF LEFT$(MD2CMD, 15) = "MD2CIRCLECHORD=" THEN
		IF MD2VALUE = "" THEN RETURN
		MD2CIRCLECHORD = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
  
	'MD2GRIDBEGINX.
	IF LEFT$(MD2CMD, 14) = "MD2GRIDBEGINX=" THEN
		IF MD2VALUE = "" THEN RETURN
		MD2GRIDBEGINX = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
 
	'MD2GRIDBEGINY.
	IF LEFT$(MD2CMD, 14) = "MD2GRIDBEGINY=" THEN
		IF MD2VALUE = "" THEN RETURN
		MD2GRIDBEGINY = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
 
	'MD2GRIDSPACEX.
	IF LEFT$(MD2CMD, 14) = "MD2GRIDSPACEX=" THEN
		IF MD2VALUE = "" THEN RETURN
		MD2GRIDSPACEX = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF

	'MD2GRIDSPACEY.
	IF LEFT$(MD2CMD, 14) = "MD2GRIDSPACEY=" THEN
		IF MD2VALUE = "" THEN RETURN
		MD2GRIDSPACEY = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF

	'MD2GRIDTARGETX.
	IF LEFT$(MD2CMD, 15) = "MD2GRIDTARGETX=" THEN
		IF MD2VALUE = "" THEN RETURN
		MD2GRIDTARGETX = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF

	'MD2GRIDTARGETY.
	IF LEFT$(MD2CMD, 15) = "MD2GRIDTARGETY=" THEN
		IF MD2VALUE = "" THEN RETURN
		MD2GRIDTARGETY = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
		  
	'SET ARRAY PARAMETERS -----------------------------------------------

	'BACKLASH.
	IF LEFT$(MD2CMD, 12) = "MD2BACKLASH(" THEN
		IF MD2VALUE = "" OR MD2SUBSCRIPT = "" THEN RETURN
		MD2BACKLASH(VAL(MD2SUBSCRIPT)) = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
  
	'HOMEDIR.
	IF LEFT$(MD2CMD, 11) = "MD2HOMEDIR(" THEN
		IF MD2VALUE = "" OR MD2SUBSCRIPT = "" THEN RETURN
		MD2HOMEDIR(VAL(MD2SUBSCRIPT)) = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
  
	'HOMEOFFSET.
	IF LEFT$(MD2CMD, 14) = "MD2HOMEOFFSET(" THEN
		IF MD2VALUE = "" OR MD2SUBSCRIPT = "" THEN RETURN
		MD2HOMEOFFSET(VAL(MD2SUBSCRIPT)) = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
  
	'LIMITF.
	IF LEFT$(MD2CMD, 10) = "MD2LIMITF(" THEN
		IF MD2VALUE = "" OR MD2SUBSCRIPT = "" THEN RETURN
		MD2LIMITF(VAL(MD2SUBSCRIPT)) = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
  
	'LIMITR.
	IF LEFT$(MD2CMD, 10) = "MD2LIMITR(" THEN
		IF MD2VALUE = "" OR MD2SUBSCRIPT = "" THEN RETURN
		MD2LIMITR(VAL(MD2SUBSCRIPT)) = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
  
	'MAXSPEED.
	IF LEFT$(MD2CMD, 12) = "MD2MAXSPEED(" THEN
		IF MD2VALUE = "" OR MD2SUBSCRIPT = "" THEN RETURN
		MD2MAXSPEED(VAL(MD2SUBSCRIPT)) = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
  
	'MINSPEED.
	IF LEFT$(MD2CMD, 12) = "MD2MINSPEED(" THEN
		IF MD2VALUE = "" OR MD2SUBSCRIPT = "" THEN RETURN
		MD2MINSPEED(VAL(MD2SUBSCRIPT)) = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
  
	'MOTORNAME.
	IF LEFT$(MD2CMD, 13) = "MD2MOTORNAME(" THEN
		IF MD2SUBSCRIPT = "" THEN RETURN
		MD2MOTORNAME(VAL(MD2SUBSCRIPT)) = RIGHT$(MD2CMD, LEN(MD2CMD) - 16)
		MD2STATUS = "O": RETURN
	END IF
  
	'POSITION.
	IF LEFT$(MD2CMD, 12) = "MD2POSITION(" THEN
		IF MD2VALUE = "" OR MD2SUBSCRIPT = "" THEN RETURN
		MD2POSITION(VAL(MD2SUBSCRIPT)) = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
  
	'SLOPE.
	IF LEFT$(MD2CMD, 9) = "MD2SLOPE(" THEN
		IF MD2VALUE = "" OR MD2SUBSCRIPT = "" THEN RETURN
		MD2SLOPE(VAL(MD2SUBSCRIPT)) = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
  
	'TARGET.
	IF LEFT$(MD2CMD, 10) = "MD2TARGET(" THEN
		IF MD2VALUE = "" OR MD2SUBSCRIPT = "" THEN RETURN
		MD2TARGET(VAL(MD2SUBSCRIPT)) = VAL(MD2VALUE)
		MD2STATUS = "O"
		RETURN
	END IF
  
	'UNITS.
	IF LEFT$(MD2CMD, 9) = "MD2UNITS(" THEN
		IF MD2VALUE = "" OR MD2SUBSCRIPT = "" THEN RETURN
		MD2UNITS(VAL(MD2SUBSCRIPT)) = INT(VAL(MD2VALUE))
		MD2STATUS = "O"
		RETURN
	END IF
 
	'UNITNAME.
	IF LEFT$(MD2CMD, 12) = "MD2UNITNAME(" THEN
		IF MD2SUBSCRIPT = "" THEN RETURN
		MD2UNITNAME(VAL(MD2SUBSCRIPT)) = RIGHT$(MD2CMD, LEN(MD2CMD) - 15)
		MD2STATUS = "O"
		RETURN
	END IF

	'MUST BE A BAD COMMAND ---------------------------------------------
   
	RETURN
  
END SUB

SUB MD2SEQSAVE

	'---------------------------------------------------------------------
	'NAME:      MD2SEQSAVE
	'DESC:      THIS PROCEDURE IS USED TO SAVE A MOTION SEQUENCE
	'           TO A DISK FILE FROM MD2SEQUENCE STRING.
	'USAGE:     SET THE DESIRED FILE NAME IN MD2SEQFILE AND CALL.
	'INPUTS:    FILE NAME, MD2SEQUENCE.
	'OUTPUTS:   STATUS = O FOR OK OR E FOR ERROR.
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	DIM MD2FILENUM AS INTEGER      'FREE FILE NUMBER.

	'DEFAULT RETURN STATUS TO OK.
	MD2STATUS = "O"

	'SAVE PARAMETERS.
	ON ERROR GOTO MD2ERROR
	MD2FILENUM = FREEFILE
	OPEN MD2SEQFILE FOR OUTPUT AS MD2FILENUM
	PRINT #MD2FILENUM, MD2SEQUENCE;

	'FINISH UP.
	CLOSE MD2FILENUM
	ON ERROR GOTO 0
   
END SUB

SUB MD2SETUP

	'---------------------------------------------------------------------
	'NAME:      MD2SETUP
	'DESC:      THE MD2SETUP PROCEDURE SETS DEFAULT MD-2 SYSTEM
	'           PARAMETERS SUCH AS MOTOR SPEED, CURRENT POSITION, ETC.
	'           AVAILABLE PARALLEL PRINTER PORTS ARE LOCATED AND
	'           VARIABLES ARE SET ACCORDINGLY.
	'USAGE:     USE AT THE BEGINNING OF A MOTION CONTROL PROGRAM.
	'           MUST BE USED -BEFORE- ANY OTHER MOTION CONTROL SUBROUTINE!
	'INPUTS:    NONE.
	'OUTPUTS:   DEFAULT MOTOR PARAMETERS, PORT AVAILABLE VARIABLES.
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	DIM MD2ELEMENT AS INTEGER           'ARRAY ELEMENT COUNTER.
	DIM MD2FILENUM AS INTEGER           'FREE FILE NUMBER.
   
	'SETUP PORT ADDRESSES.
	MD2MTRADR12 = &H3BC
	MD2MTRADR34 = &H378
	MD2MTRADR56 = &H278

	'SETUP MOTOR PARAMETER DEFAULTS.
	FOR MD2MOTOR = 1 TO 6
		MD2BACKLASH(MD2MOTOR) = 0       'BACKLASH COMPENSATION.
		MD2HOMEOFFSET(MD2MOTOR) = 0     'HOME OFFSET.
		MD2HOMEDIR(MD2MOTOR) = 0        'HOME DIRECTION.
		MD2LIMITF(MD2MOTOR) = 999999    'FORWARD LIMIT.
		MD2LIMITR(MD2MOTOR) = -999999   'REVERSE LIMIT.
		MD2LASTDIR(MD2MOTOR) = 1        'LAST DIRECTION - FORWARD.
		MD2POSITION(MD2MOTOR) = 0       'POSITIONS = 0.
		MD2PATPTR(MD2MOTOR) = 0         'STEP PATTERN POINTERS.
		MD2TARGET(MD2MOTOR) = 400       'TARGET POSITION,STEPS.
		MD2MINSPEED(MD2MOTOR) = 100     'MINIMUM SPEED.
		MD2MAXSPEED(MD2MOTOR) = 500     'MAXIMUM SPEED.
		MD2MOTORNAME(MD2MOTOR) = "#" + STR$(MD2MOTOR)    'NAME OF MOTOR.
		MD2SLOPE(MD2MOTOR) = 100        'RAMPING SLOPE IN STEPS.
		MD2UNITS(MD2MOTOR) = 1          '1 UNIT (STEPS) PER STEP.
		MD2UNITNAME(MD2MOTOR) = "Steps"
	NEXT MD2MOTOR
	MD2MOTOR = 1                'MOTOR NUMBER.
	MD2MOVETYPE = "R"           'RELATIVE MOVES.
	MD2STEPTYPE = "H"           'HALF STEP MODE.
	MD2STATUS = "O"             'STATUS=OK.
	MD2HOLD = 0                 'DONT HOLD MOTORS.
	MD2INTERRUPTS = 0           'INTERRUPTS OFF DURING MOVES.
	MD2PARFILE = "DEFAULTS.PAR" 'DEFAULT MOTOR PARAMETERS.
	MD2SEQFILE = ""             'SEQUENCE FILE.
	IF MD2CALFILE = "" THEN MD2CALFILE = "MD2.CAL"  'CALIBRATION FILE.
	MD2CALMODE = 0              'CALIBRATION MODE-OFF.

	'GRID DEFAULT PARAMETERS.
	MD2GRIDBEGINX = 0           'COLUMN BEGIN.
	MD2GRIDBEGINY = 0           'ROW BEGIN.
	MD2GRIDSPACEX = 1           'COLUMN SPACING.
	MD2GRIDSPACEY = 1           'ROW SPACING.
	MD2GRIDTARGETX = 0          'COLUMN TARGET.
	MD2GRIDTARGETY = 0          'ROW TARGET.
   
	'CIRCLE DEFAULT PARAMETERS.
	MD2CIRCLERADIUSX = 1        'RADIUS X.
	MD2CIRCLERADIUSY = 1        'RADIUS Y.
	MD2CIRCLECENTERX = 1        'CENTER X.
	MD2CIRCLECENTERY = 1        'CENTER Y.
	MD2CIRCLESTART = 0          'START ANGLE.
	MD2CIRCLEARC = 360          'ARC ANGLE (360=CIRCLE).
	MD2CIRCLECHORD = 1          'CHORD ANGLE.
   
	'INSURE ALL MD-2 SYSTEMS ARE OFF.
	OUT MD2MTRADR12 + 2, &HC
	OUT MD2MTRADR34 + 2, &HC
	OUT MD2MTRADR56 + 2, &HC

	'FIND PARALLEL PRINTER PORTS AND SET VARIABLES.
	OUT MD2MTRADR12, &HAA          'SEND DATA TO PORTS.
	OUT MD2MTRADR34, &HAA          'IF READ BACK THEN AVAILABLE.
	OUT MD2MTRADR56, &HAA
	IF INP(MD2MTRADR12) = &HAA THEN MD2AVAILABLE12 = -1 ELSE MD2AVAILABLE12 = 0
	IF INP(MD2MTRADR34) = &HAA THEN MD2AVAILABLE34 = -1 ELSE MD2AVAILABLE34 = 0
	IF INP(MD2MTRADR56) = &HAA THEN MD2AVAILABLE56 = -1 ELSE MD2AVAILABLE56 = 0
	OUT MD2MTRADR12, &HFF          'TURN OFF ALL PHASES.
	OUT MD2MTRADR34, &HFF
	OUT MD2MTRADR56, &HFF

	'LOAD SPEED CALIBRATION PARAMETERS.
	MD2STATUS = "O"
	ON ERROR GOTO MD2ERROR
	MD2FILENUM = FREEFILE
	OPEN MD2CALFILE FOR INPUT AS MD2FILENUM
	FOR MD2ELEMENT = 1 TO 24
		IF MD2STATUS <> "O" THEN EXIT FOR
		INPUT #MD2FILENUM, MD2DELAY(MD2ELEMENT), MD2VELOCITY(MD2ELEMENT)
		IF MD2DELAY(MD2ELEMENT) <= 0 THEN MD2STATUS = "F"
		IF MD2VELOCITY(MD2ELEMENT) <= 0 THEN MD2STATUS = "F"
	NEXT MD2ELEMENT
	'CLEAR CALIBRATION ARRAYS IF FILE ERROR.
	IF MD2STATUS <> "O" THEN
		FOR MD2ELEMENT = 1 TO 16
			MD2DELAY(MD2ELEMENT) = 0: MD2VELOCITY(MD2ELEMENT) = 0
		NEXT MD2ELEMENT
	END IF
	CLOSE MD2FILENUM
	ON ERROR GOTO 0
	'SET MD2CALIBRATED FLAG.
	IF MD2VELOCITY(1) = 0 THEN
		MD2CALIBRATED = 0
		MD2STATUS = "C"
	ELSE
		MD2CALIBRATED = -1
	END IF
		 
END SUB

SUB MD2STANDBYOFF
  
	'---------------------------------------------------------------------
	'NAME:      MD2STANDBYOFF
	'DESC:      THE MD2STANDBYOFF PROCEDURE RETURNS AN MD-2 TO FULL
	'           CURRENT MODE.
	'           THE MD-2 IS SELECTED BY THE MD2MOTOR PARAMETER.
	'           BOTH MOTORS ON THE MD-2 IS AFFECTED
	'           SOME MD-2 SYSTEMS DO NOT HAVE THIS FEATURE, CHECK WITH
	'           YOUR MANUAL FOR MORE INFORMATION.
	'USAGE:     USE TO RETURN AN MD-2 TO FULL CURRENT MODE BEFORE MOVING
	'           THEM FOR MAXIMUM PERFORMANCE.
	'INPUTS:    MOTOR # DETERMINES MD-2.
	'OUTPUTS:   NONE.
	'---------------------------------------------------------------------

	'CHECK FOR SETUP.
	IF MD2MTRADR12 <> &H3BC THEN MD2STATUS = "S": EXIT SUB

	'CHECK FOR VALID MOTOR #.
	MD2STATUS = "B"
	IF MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 3 THEN MD2STATUS = "O"
	IF MD2MOTOR = 4 OR MD2MOTOR = 5 OR MD2MOTOR = 6 THEN MD2STATUS = "O"
	IF MD2MOTOR = 12 OR MD2MOTOR = 34 OR MD2MOTOR = 56 THEN MD2STATUS = "O"
	IF MD2STATUS <> "O" THEN EXIT SUB

	'CHECK FOR PORT AVAILABILITY.
	MD2STATUS = "P"
	IF (MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 12) AND NOT MD2AVAILABLE12 THEN EXIT SUB
	IF (MD2MOTOR = 3 OR MD2MOTOR = 4 OR MD2MOTOR = 34) AND NOT MD2AVAILABLE34 THEN EXIT SUB
	IF (MD2MOTOR = 5 OR MD2MOTOR = 6 OR MD2MOTOR = 56) AND NOT MD2AVAILABLE56 THEN EXIT SUB

	'CHECK FOR ENABLED MD-2.
	MD2STATUS = "E"
	IF (MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 12) AND NOT MD2ENABLED12 THEN EXIT SUB
	IF (MD2MOTOR = 3 OR MD2MOTOR = 4 OR MD2MOTOR = 34) AND NOT MD2ENABLED34 THEN EXIT SUB
	IF (MD2MOTOR = 5 OR MD2MOTOR = 6 OR MD2MOTOR = 56) AND NOT MD2ENABLED56 THEN EXIT SUB
	
	'TURN STANDBY MODE OFF.
	IF MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 12 THEN
		OUT MD2MTRADR12 + 2, INP(MD2MTRADR12 + 2) OR &H4
	END IF
	IF MD2MOTOR = 3 OR MD2MOTOR = 4 OR MD2MOTOR = 34 THEN
		OUT MD2MTRADR34 + 2, INP(MD2MTRADR34 + 2) OR &H4
	END IF
	IF MD2MOTOR = 5 OR MD2MOTOR = 6 OR MD2MOTOR = 56 THEN
		OUT MD2MTRADR56 + 2, INP(MD2MTRADR56 + 2) OR &H4
	END IF
												  
	MD2STATUS = "O"

END SUB

SUB MD2STANDBYON
   
	'---------------------------------------------------------------------
	'NAME:      MD2STANDBYON
	'DESC:      THE MD2STANDBYON PROCEDURE PUTS AN MD-2 INTO STANDBY MODE
	'           WHICH REDUCES THE MOTOR CURRENT BY 50%.  THIS RESULTS
	'           IN LOWER HEAT DISSIPATION DURING STANDSTILL WITHOUT
	'           LOSSING ALL HOLDING TORQUE.
	'           THE MD-2 SYSTEM IS SELECTED BY THE MD2MOTOR PARAMETER.
	'           BOTH MOTORS ON AN MD-2 ARE AFFECTED.
	'           MAKE SURE THAT MOTORS ARE ENERGIZED BY FIRST MOVING WITH
	'           MD2HOLD=-1 OR STANDBY WILL NOT WORK.
	'           SOME MD-2 SYSTEMS DO NOT HAVE THIS FEATURE, CHECK WITH
	'           YOUR MANUAL FOR MORE INFORMATION.
	'USAGE:     USE AFTER A MOTION TO MAINTAIN SOME HOLDING TORQUE WHILE
	'           REDUCING HEAT.
	'INPUTS:    MOTOR # DETERMINES MD-2.
	'OUTPUTS:   NONE
	'---------------------------------------------------------------------

	'CHECK FOR SETUP.
	IF MD2MTRADR12 <> &H3BC THEN MD2STATUS = "S": EXIT SUB

	'CHECK FOR VALID MOTOR #.
	MD2STATUS = "B"
	IF MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 3 THEN MD2STATUS = "O"
	IF MD2MOTOR = 4 OR MD2MOTOR = 5 OR MD2MOTOR = 6 THEN MD2STATUS = "O"
	IF MD2MOTOR = 12 OR MD2MOTOR = 34 OR MD2MOTOR = 56 THEN MD2STATUS = "O"
	IF MD2STATUS <> "O" THEN EXIT SUB

	'CHECK FOR PORT AVAILABILITY.
	MD2STATUS = "P"
	IF (MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 12) AND NOT MD2AVAILABLE12 THEN EXIT SUB
	IF (MD2MOTOR = 3 OR MD2MOTOR = 4 OR MD2MOTOR = 34) AND NOT MD2AVAILABLE34 THEN EXIT SUB
	IF (MD2MOTOR = 5 OR MD2MOTOR = 6 OR MD2MOTOR = 56) AND NOT MD2AVAILABLE56 THEN EXIT SUB

	'CHECK FOR ENABLED MD-2.
	MD2STATUS = "E"
	IF (MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 12) AND NOT MD2ENABLED12 THEN EXIT SUB
	IF (MD2MOTOR = 3 OR MD2MOTOR = 4 OR MD2MOTOR = 34) AND NOT MD2ENABLED34 THEN EXIT SUB
	IF (MD2MOTOR = 5 OR MD2MOTOR = 6 OR MD2MOTOR = 56) AND NOT MD2ENABLED56 THEN EXIT SUB

	'TURN STANDBY MODE ON.
	IF MD2MOTOR = 1 OR MD2MOTOR = 2 OR MD2MOTOR = 12 THEN
		OUT MD2MTRADR12 + 2, INP(MD2MTRADR12 + 2) AND &HFB
	END IF
	IF MD2MOTOR = 3 OR MD2MOTOR = 4 OR MD2MOTOR = 34 THEN
		OUT MD2MTRADR34 + 2, INP(MD2MTRADR34 + 2) AND &HFB
	END IF
	IF MD2MOTOR = 5 OR MD2MOTOR = 6 OR MD2MOTOR = 56 THEN
		OUT MD2MTRADR56 + 2, INP(MD2MTRADR56 + 2) AND &HFB
	END IF

	MD2STATUS = "O"

END SUB

