/*---------------------------------------------------------------------
NAME:      MD2QC2S.C
DESC:      MD-2 LEVEL 2 C MOTION CONTROL SUBROUTINES
USAGE:     USE WHEN CREATING COMPLEX MOTION CONTROL PROGRAMS.
		   USE LEVEL 1 LIBRARY WHEN CREATING SIMPLE PROGRAMS.

		   FEATURES:
			   DUAL MOTOR MOVES.
			   RELATIVE/ABSOLUTE MOVES.
			   ADJUSTABLE ACCELERATION/DECELERATION.
			   2-AXIS LINEAR INTERPOLATION.
			   2-AXIS CIRCULAR INTERPOLATION.
			   ARC AND ELLIPSE MOVES.
			   2-AXIS GRID MOVES.
			   USER DEFINED UNITS.
			   BACKLASH COMPENSATION.
			   SOFT LIMITS.
			   SPEED GIVEN IN UNITS PER SECOND.
			   INPUT/OUTPUT PORT CONTROL.

		   ALL MOTOR PARAMETERS ARE PASSED TO SUBROUTINES VIA
		   GLOBAL VARIABLES.  ALL VARIABLES AND SUBS BEGIN WITH
		   MD2 TO AVOID CONFLICTS.
		   SEE .TXT FILE FOR COMPLETE DESCRIPTION.
BY:        COPYRIGHT (C) 1994 ARRICK ROBOTICS, ROGER ARRICK.
DATE:      7/12/94
EDIT:      211

PUBLIC SUBROUTINES ---------------------------------------------------

	NAME           DESCRIPTION
	----           -----------
	MD2SETUP       Used at the beginning of a program to initialize
				   motor parameters to their default values and checks
				   for parallel port availability. Loads calibration file.
				   Use this SUB before any other.
	MD2CALIBRATE   Calibrates motor speeds with computer speeds and saves
				   to disk.  Use this SUB once on your cpu before moving.
	MD2ON          Turns on an MD-2 system. Use before any moves.
	MD2OFF         Turns off an MD-2 system. Use at end of program.
	MD2HOME        Moves a motor to the Home position by watching the
				   home switch.  Current position is set to zero.
	MD2MOVE        Moves a motor to the desired target position using
				   the various motor parameters.  Also moves two motors
				   together providing linear interpolation.
	MD2CIRCLE      Performs 2-axis circular, arc and ellipise moves.
	MD2GRID        Moves 2 motors using grid coordinates.
	MD2OUTPUTS     Turns on and off output pins to control tools.
	MD2INPUTS      Reads condition of input pins and home switches.
	MD2STANDBYON   Puts MD-2 into standby mode which reduces motor current
				   and heat while maintaining some holding torque.
	MD2STANDBYOFF  Turns off standby mode giving full current to motors.
	MD2PARLOAD     Loads motor parameters from a disk file.
	MD2PARSAVE     Saves motor parameters to a disk file.


PUBLIC MOTOR PARAMETERS AND VARIABLES ----------------------------------

	(M) indicates motor # 1,2,3,4,5 or 6.

	NAME            TYPE    DESCRIPTION
	----            ----    -----------

	MD2BACKLASH(M)  DOUBLE  Backlash compensation in units.  Adds this
							amount to every move that changes direction.
							Uses MD2LASTDIR(M).
							Value is in Units.
	MD2ENABLED12    INTEGER Port 3BC for motors 1 & 2 has been enabled
							by the MD2ON sub. 0=not enabled, -1=available.
	MD2ENABLED34    INTEGER Port 3BC for motors 3 & 4 has been enabled
							by the MD2ON sub. 0=not enabled, -1=available.
	MD2ENABLED56    INTEGER Port 3BC for motors 5 & 6 has been enabled
							by the MD2ON sub. 0=not enabled, -1=available.
	MD2HOLD         INTEGER -1=Leaves the motor energized after a move
							to cause the motor to hold its position.
							0 causes the motor to turn off after a move
							which conserves power and reduces heat.
	MD2HOMEOFFSET(M)DOUBLE  Distance from home switch to call home.
							Value is in units.
	MD2HOMEDIR(M)   INTEGER Direction to home switch, 0=rev, -1=fwd.
							Reverses meaning of forward and reverse
							target directions.  All directions are reversed.
	MD2INTERRUPTS   INTEGER 0=turns COM port interrupts off during moves
							which prevents mouse & modem interference.
							This can be important under WINDOWS.
							-1 leaves interrupts enabled.
	MD2LIMITF(M)    DOUBLE  Forward soft limit in units.  Prevents moves
							beyond this value in the forward direction.
							Value is in Units.
	MD2LIMITR(M)    DOUBLE  Reverse soft limit in units.  Prevents moves
							beyond this value in the reverse direction
							Value is in Units.
	MD2MOTOR        INTEGER The selected motor to act upon. 1,2,3,4,5 or 6.
							Set to 12, 34, 56 for dual motor moves.
	MD2MOTORNAME(M) STRING  User definable motor name.
	MD2MAXSPEED(M)  DOUBLE  Motor fastest speed after accelerating.
							In Units per Second.
	MD2MINSPEED(M)  DOUBLE  Motor speed at the beginning and end of ramping.
							In Units per Second.
	MD2MOVETYPE     CHAR    Tells the MD2MOVE sub what type of move.
							R=relative, A=absolute. Affects all motors.
							With relative moves, the MD2TARGET parameter
							tells how many steps to move from the current
							position. Negative numbers are reverse, positive
							numbers are forward.
							With absolute moves, the MD2TARGET parameter
							tells what position to move to.  The number of
							steps and direction are determined by the
							current position. Negative numbers are reverse
							from home and positive numbers are forward.
							negative then reverse, positive then forward.
	MD2AVAILABLE12  INTEGER Port 3BC availability for motors 1 & 2 set by
							the MD2SETUP sub. 0=not available, -1=available.
	MD2AVAILABLE34  INTEGER Port 378 availability for motors 3 & 4 set by
							the MD2SETUP sub. 0=not available, -1=available.
	MD2AVAILABLE56  INTEGER Port 278 availability for motors 5 & 6 set by
							the MD2SETUP sub. 0=not available, -1=available.
	MD2POSITION(M)  DOUBLE  Current motor position for each motor (M=motor #)
							relative to home. Negative = reverse from home
							and positive = forward from home.
							Value is in units.
	MD2SLOPE(M)     DOUBLE  The slope of the ramp (acceleration/deceleration)
							in number Units.  A 2 would mean that the
							motor would accelerate from the MD2MINSPEED to
							the MD2MAXSPEED in 2 units.
	MD2STATUS       CHAR    Completion status.
							O=motion completed OK,
							K=a keypress stopped the motion,
							B=bad parameter,
							L=soft limit was exceeded.
							E=MD-2 not enabled.
							P=Port not available.
							F=file error.
	MD2STEPTYPE     CHAR    H=Half step, D=Double phase full step,
							S=Single phase full step.
							Affects all motors.
	MD2TARGET(M)    DOUBLE  The number of units to move in relative moves,
							the position to move to in absolute moves.
							Value is in units which is converted to steps.
	MD2UNITS(M)     LONG    Converts user definable units to steps
							for MD2MOVE subroutine.
							This is number of steps in each unit.
	MD2UNITNAME(M)  STRING  This is the name of the units such as
							steps, inches, degrees, etc.

PUBLIC INPUT/OUTPUT PORT PARAMETERS----------------------------------

	MD2OUTPUTCODE   INTEGER Tells the MD2OUTPUTS sub which MD-2, which
							output pin and on/off action.
	MD2INPUT(CODE)  INTEGER Contains status of input pins and home switches
							after using MD2INPUTS sub.

PUBLIC PARAMETER FILE VARIABLES---------------------------

	MD2PARFILE      STRING  Parameter file name.

PUBLIC GRID MOVE PARAMETERS-------------------------------------------

	MD2GRIDBEGINX   DOUBLE  Grid X start position in units.
	MD2GRIDBEGINY   DOUBLE  Grid Y start position in units.
	MD2GRIDSPACEX   DOUBLE  Grid X spacing in units.
	MD2GRIDSPACEY   DOUBLE  Grid Y spacing in units.
	MD2GRIDTARGETX  INTEGER Grid X target.
	MD2GRIDTARGETY  INTEGER Grid Y target.

PUBLIC CIRCLE/ARC MOVE PARAMETERS------------------------------------

	MD2CIRCLERADIUSX   DOUBLE  Circle X radius in units.
	MD2CIRCLERADIUSY   DOUBLE  Circle Y radius in units.
	MD2CIRCLECENTERX   DOUBLE  Circle X center in units.
	MD2CIRCLECENTERY   DOUBLE  Circle Y center in units.
	MD2CIRCLESTART     INTEGER Circle start angle in degrees.
	MD2CIRCLEARC       INTEGER Circle arc angle in degrees. -=CW,+=CCW.
	MD2CIRCLECHORD     INTEGER Circle chord angle in degrees.

PUBLIC MOTOR SPEED CALIBRATION PARAMETERS--------------------------------

	MD2VELOCITY(24)    DOUBLE  Fastest possible motor speed in steps/sec.
	MD2DELAY(24)       LONG    Slowest possible motor speed in steps/sec.
	MD2CALFILE         STRING  Calibration file name.
	MD2CALIBRATED      INTEGER Calibrated status, 0=false, -1=true.

------------------------------------------------------------------------*/

/* Include files */
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <ctype.h>          /* tolower, toupper */
#include <conio.h>          /* outp, inp */
#include <stdlib.h>         /* abs */
#include <time.h>           /* clock */

/* remove macro definitions of inp & outp so they can be
   used as conditionals in while loop */
#undef inp
#undef outp

/* Functions prototypes */
void MD2Setup(void);        /* Initialize MD-2 system */
void MD2Calibrate(void);    /* Calibrate motor speeds */
void MD2On(void);           /* Turn ON MD-2 system */
void MD2Off(void);          /* Turn OFF MD-2 system */
void MD2StandbyOn(void);    /* Turn ON standby mode */
void MD2StandbyOff(void);   /* Turn OFF standby mode */
void MD2Home(void);         /* Move 1 motor to the home switch */
void MD2Move(void);         /* Move 1 or 2 motors */
void MD2Circle(void);       /* Move 2 motors in a circle, arc, ellipse */
void MD2Grid(void);         /* Move 2 motors to grid coordinates */
void MD2Outputs(void);      /* Control output signals */
void MD2Inputs(void);       /* Read input signals and home switches */
void MD2ParLoad(void);      /* Read parameters from a disk file */
void MD2ParSave(void);      /* Save parameters to a disk file */
static void MD2GetNumber(void);    /* Move file to next number field */
static void MD2GetString(void);    /* Return next string field */

/* Variables not needed by programmer */
static int MD2MtrAdr12;     /* Port for motors 1 & 2 */
static int MD2MtrAdr34;     /* Port for motors 3 & 4 */
static int MD2MtrAdr56;     /* Port for motors 5 & 6 */
static int MD2StpPat[8];    /* Step pattern array */
static int MD2PatPtr[7];    /* Step pattern array pointers */
static int MD2LastDir[7];   /* Last direction -1=rev, 1=fwd */
static int MD2CalMode;      /* Calibration mode 0=off,-1=on */
static char MD2String[30];  /* String */
static FILE *MD2File;       /* Calibration or parameter file */

/* Global motor parameters needed by programmer */
int MD2Available12;         /* Port 3bc for motors 1,2 is available */
int MD2Available34;         /* Port 3bc for motors 3,4 is available */
int MD2Available56;         /* Port 3bc for motors 5,6 is available */
double MD2Backlash[7];      /* Backlash compensation in units */
char MD2CalFile[13];        /* Calibration file name */
int MD2Calibrated;          /* Calibrated -1=true, 0=false */
double MD2CircleRadiusX;    /* Circle X radius in units */
double MD2CircleRadiusY;    /* Circle Y radius in units */
double MD2CircleCenterX;    /* Circle X center in units */
double MD2CircleCenterY;    /* Circle Y center in units */
int MD2CircleStart;         /* Circle starting angle in degrees */
int MD2CircleArc;           /* Circle arc angle in degrees */
int MD2CircleChord;         /* Circle chord angle in degrees */
long int MD2Delay[25];      /* Delay array for speed control */
int MD2Enabled12;           /* Port 3bc for motors 1,2 turned on */
int MD2Enabled34;           /* Port 378 for motors 3,4 turned on */
int MD2Enabled56;           /* Port 278 for motors 5,6 turned on */
double MD2GridBeginX;       /* Grid X start position in units */
double MD2GridBeginY;       /* Grid Y start position in units */
double MD2GridSpaceX;       /* Grid X spacing in units */
double MD2GridSpaceY;       /* Grid Y spacing in units */
int MD2GridTargetX;         /* Grid X target */
int MD2GridTargetY;         /* Grid Y target */
int MD2Hold;                /* Hold motors after moves. -1=yes,0=no */
double MD2HomeOffset[7];    /* Distance to move away from home switch */
int MD2HomeDir[7];          /* Direction of the home switch. 0=rev,-1=fwd */
int MD2Input[36];           /* Status of input signals and home switches */
int MD2Interrupts;          /* Leave COM interrupts on during moves? 0=no */
double MD2LimitF[7];        /* Forward soft limit in units */
double MD2LimitR[7];        /* Reverse soft limit in units */
int MD2Motor;               /* Selected motor or pair 1-6, 12,34,56 */
char MD2MotorName[7][9];    /* Motor names */
double MD2MaxSpeed[7];      /* Speed after acceleration in units/sec */
double MD2MinSpeed[7];      /* Start and stop speed in units/sec */
char MD2MoveType;           /* R=relative, A=absolute */
int MD2OutputCode;          /* Port,bit and status for MD2Outputs function */
char MD2ParFile[13];        /* Parameter file name */
double MD2Position[7];      /* Current motor position */
double MD2Slope[7];         /* Accel/Decel slope in units */
char MD2Status;             /* Completion status O=ok, B=bad parameter,
							   F=file error, L=limit exceeded, K=kepress stop
							   E=port not enabled, A=port not available. */
char MD2StepType;           /* H=half, D=double full, S=single full */
double MD2Target[7];        /* # of units to move or target position */
long int MD2Units[7];       /* # of steps in each unit */
char MD2UnitName[7][9];     /* Name of user-definable unit */
double MD2Velocity[25];     /* Velocity array for speed control */


void MD2ParLoad(void)
{

	/*---------------------------------------------------------------------
	'NAME:      MD2PARLOAD
	'DESC:      THIS PROCEDURE IS USED TO LOAD ALL MOTOR PARAMETERS
	'           FROM A DISK FILE.  WHEN USED WITH THE MD2PARSAVE PROCEDURE,
	'           A PROGRAM CAN BE CREATED WHICH SAVES ITS STATUS BEFORE
	'           POWERING DOWN AND RESTORES WHEN COMMING BACK ON ELIMINATING
	'           THE NEED FOR THE USER TO RE-ENTER ALL MOTOR PARAMETERS.
	'USAGE:     SET THE DESIRED FILE NAME IN MD2PARFILE AND CALL.
	'INPUTS:    FILE NAME AND ALL MOTOR PARAMETERS.
	'OUTPUTS:   STATUS = O FOR OK OR E FOR ERROR.
	'---------------------------------------------------------------------*/

	/* Local variables */
	int MD2Mtr;         /* Motor counter */
	char MD2xs[30];     /* Dummy string */
	int MD2xi;          /* Dummy int */

	/* Default return status to OK */
	MD2Status = 'O';

	/* Load parameters */

	/* Open file - if error then done else read contents */
	if ((MD2File = fopen( MD2ParFile, "r" )) == NULL) {
		MD2Status = 'F';
		return;
	}

	/* Motor parameters */
	MD2GetString(); strcpy(MD2xs, MD2String);
	MD2GetNumber(); fscanf(MD2File, "%i", &MD2Hold);
	MD2GetNumber(); fscanf(MD2File, "%i", &MD2Interrupts);
	MD2GetNumber(); fscanf(MD2File, "%i", &MD2Motor);
	MD2GetString(); MD2MoveType = MD2String[0];
	MD2GetString(); MD2StepType = MD2String[0];

	/* Check for corrupt parameter file */
	if ((MD2Hold != 0) && (MD2Hold != -1)) MD2Status = 'F';
	if ((MD2MoveType != 'A') && (MD2MoveType != 'R')) MD2Status = 'F';

	/* Motor parameter arrays */
	for (MD2Mtr=1; MD2Mtr<7; MD2Mtr++) {
		if (MD2Status != 'O') break;
		MD2GetNumber(); fscanf(MD2File, "%i", &MD2xi);
		MD2GetNumber(); fscanf(MD2File, "%lf", &MD2Backlash[MD2Mtr]);
		MD2GetNumber(); fscanf(MD2File, "%i", &MD2HomeDir[MD2Mtr]);
		MD2GetNumber(); fscanf(MD2File, "%lf", &MD2HomeOffset[MD2Mtr]);
		MD2GetNumber(); fscanf(MD2File, "%lf", &MD2LimitF[MD2Mtr]);
		MD2GetNumber(); fscanf(MD2File, "%lf", &MD2LimitR[MD2Mtr]);
		MD2GetNumber(); fscanf(MD2File, "%lf", &MD2MaxSpeed[MD2Mtr]);
		if (MD2MaxSpeed[MD2Mtr] < 0) MD2Status = 'F';
		MD2GetNumber(); fscanf(MD2File, "%lf", &MD2MinSpeed[MD2Mtr]);
		if (MD2MinSpeed[MD2Mtr] < 0) MD2Status = 'F';
		MD2GetString(); strcpy(MD2MotorName[MD2Mtr], MD2String);
		MD2GetNumber(); fscanf(MD2File, "%lf", &MD2Position[MD2Mtr]);
		MD2GetNumber(); fscanf(MD2File, "%lf", &MD2Slope[MD2Mtr]);
		if (MD2Slope[MD2Mtr] < 0) MD2Status = 'F';
		MD2GetNumber(); fscanf(MD2File, "%lf", &MD2Target[MD2Mtr]);
		MD2GetNumber(); fscanf(MD2File, "%li", &MD2Units[MD2Mtr]);
		if (MD2Units[MD2Mtr] < 0) MD2Status = 'F';
		MD2GetString(); strcpy(MD2UnitName[MD2Mtr], MD2String);
		MD2GetNumber(); fscanf(MD2File, "%i", &MD2PatPtr[MD2Mtr]);
		MD2GetNumber(); fscanf(MD2File, "%i", &MD2LastDir[MD2Mtr]);
	}

	/* Grid parameters */
	MD2GetNumber(); fscanf(MD2File, "%lf", &MD2GridBeginX);
	MD2GetNumber(); fscanf(MD2File, "%lf", &MD2GridBeginY);
	MD2GetNumber(); fscanf(MD2File, "%lf", &MD2GridSpaceX);
	MD2GetNumber(); fscanf(MD2File, "%lf", &MD2GridSpaceY);
	MD2GetNumber(); fscanf(MD2File, "%i", &MD2GridTargetX);
	MD2GetNumber(); fscanf(MD2File, "%i", &MD2GridTargetY);

	/* Circle parameters */
	MD2GetNumber(); fscanf(MD2File, "%lf", &MD2CircleRadiusX);
	MD2GetNumber(); fscanf(MD2File, "%lf", &MD2CircleRadiusY);
	MD2GetNumber(); fscanf(MD2File, "%lf", &MD2CircleCenterX);
	MD2GetNumber(); fscanf(MD2File, "%lf", &MD2CircleCenterY);
	MD2GetNumber(); fscanf(MD2File, "%i", &MD2CircleStart);
	MD2GetNumber(); fscanf(MD2File, "%i", &MD2CircleArc);
	MD2GetNumber(); fscanf(MD2File, "%i", &MD2CircleChord);

	/* Finish up */
	if (fclose(MD2File) == EOF) MD2Status = 'F';

}


static void MD2GetNumber(void)
{
	/* Move file pointer to beginning of number */

	/* Local variables */
	char MD2Char;

	while(-1) {
		/* Read the character */
		MD2Char = (char)fgetc(MD2File);
		if ((MD2Char > 47) && (MD2Char < 58)) break;
		/* . */
		if (MD2Char == 46) break;
		/* - */
		if (MD2Char == 45) break;
	}

	/* Move back one character */
	fseek(MD2File,-1,SEEK_CUR);

}


static void MD2GetString(void)
{
	/* Set MD2String to string within "" from MD2File */

	/* Local variables */
	char MD2Char;           /* Current character */
	int MD2StringP;         /* String pointer */

	/* Look for " */
	do {
		MD2Char = (char)fgetc(MD2File);
	} while(MD2Char != 34);

	/* Get characters until " */
	MD2StringP = 0;
	while(-1) {
		MD2String[MD2StringP] = (char)fgetc(MD2File);
		if (MD2String[MD2StringP] == 34) break;
		MD2StringP++;
	}

	/* End of string */
	MD2String[MD2StringP] = '\0';
}


void MD2Setup(void)
{

	/*---------------------------------------------------------------------
	'name:       MD2Setup
	'desc:       The MD2Setup procedure sets default md-2 system
	'            parameters such as motor speed, current position, etc.
	'            available parallel printer ports are located and
	'            variables are set accordingly.
	'usage:      Use at the beginning of a motion control program.
	'            must be used -before- any other motion control subroutine!
	'inputs:     None.
	'outputs:    Default motor parameters, port available variables.
	'---------------------------------------------------------------------*/

	/* Local variables */
	int MD2Element;                    /* Array element counter */

	/* Setup port addresses */
	MD2MtrAdr12 = 0x3bc;
	MD2MtrAdr34 = 0x378;
	MD2MtrAdr56 = 0x278;

	/* Set motor parameter defaults */
	for (MD2Motor = 1; MD2Motor < 7; MD2Motor++ ) {
	   MD2Backlash[MD2Motor] = 0;      /* Backlash compensation */
	   MD2HomeOffset[MD2Motor] = 0;    /* Hhome offset */
	   MD2HomeDir[MD2Motor] = 0;       /* Home direction */
	   MD2LimitF[MD2Motor] = 999999;   /* Forward limit */
	   MD2LimitR[MD2Motor] = -999999;  /* Reverse limit */
	   MD2LastDir[MD2Motor] = 1;       /* Last direction = forward */
	   MD2Position[MD2Motor] = 0;      /* Current position */
	   MD2PatPtr[MD2Motor] = 0;        /* Pattern pointer */
	   MD2Target[MD2Motor] = 400;      /* Target position or distance */
	   MD2MinSpeed[MD2Motor] = 100;    /* Minimum speed */
	   MD2MaxSpeed[MD2Motor] = 500;    /* Maximum speed */
	   /* name of motor */
	   sprintf(MD2MotorName[MD2Motor],"# %i",MD2Motor);
	   MD2Slope[MD2Motor] = 100;       /* Ramping slope */
	   MD2Units[MD2Motor] = 1;         /* 1 unit per step */
	   /* name of unit */
	   strcpy(MD2UnitName[MD2Motor],"steps");
	}

	MD2Motor = 1;                       /* Motor number */
	MD2MoveType = 'R';                  /* Relative moves */
	MD2StepType = 'H';                  /* Half step mode */
	MD2Status = 'O';                    /* Status = ok */
	MD2Hold = 0;                        /* Dont holde motors */
	MD2Interrupts = 0;                  /* Interrupts off during moves */
	strcpy(MD2ParFile,"DEFAULTS.PAR");  /* Motor parameter file name */

	MD2CalMode = 0;                     /* Calibration mode off */
	if (isalnum(MD2CalFile[0]) == 0)
		strcpy(MD2CalFile,"MD2.CAL");   /* Calibration file */

	/* Grid default parameters */
	MD2GridBeginX = 0;                  /* Column begin */
	MD2GridBeginY = 0;                  /* Row begin */
	MD2GridSpaceX = 1;                  /* Column spacing */
	MD2GridSpaceY = 1;                  /* Row spacing */
	MD2GridTargetX = 0;                 /* Column target */
	MD2GridTargetY = 0;                 /* Row target */

	/* Circle default parameters */
	MD2CircleRadiusX = 1;               /* radius x */
	MD2CircleRadiusY = 1;               /* radius y */
	MD2CircleCenterX = 1;               /* center x */
	MD2CircleCenterY = 1;               /* center y */
	MD2CircleStart = 0;                 /* start angle */
	MD2CircleArc = 360;                 /* arc angle */
	MD2CircleChord = 1;                 /* chord angle */

	/* Insure all md-2 systems are off */
	outp(MD2MtrAdr12 + 2, 0x0c);
	outp(MD2MtrAdr34 + 2, 0x0c);
	outp(MD2MtrAdr56 + 2, 0x0c);

	/* Find parallel printer ports */
	outp(MD2MtrAdr12, 0xaa);
	outp(MD2MtrAdr34, 0xaa);
	outp(MD2MtrAdr56, 0xaa);
	if (inp(MD2MtrAdr12) == 0xaa) MD2Available12 = -1; else MD2Available12 = 0;
	if (inp(MD2MtrAdr34) == 0xaa) MD2Available34 = -1; else MD2Available34 = 0;
	if (inp(MD2MtrAdr56) == 0xaa) MD2Available56 = -1; else MD2Available56 = 0;
	outp(MD2MtrAdr12, 0xff);
	outp(MD2MtrAdr34, 0xff);
	outp(MD2MtrAdr56, 0xff);

	/* Load speed calibration parameters */
	MD2Status = 'O';

	/* Open the file */
	if ((MD2File = fopen(MD2CalFile,"r")) == NULL) {
		MD2Status = 'F';
		return;
	}

	/* Read each element */
	for(MD2Element = 1; MD2Element < 25;  MD2Element++) {

		/* Quit reading if error occurs */
		if (MD2Status != 'O') break;

		/* Go to next field (search for a number) */
		MD2GetNumber();
		fscanf(MD2File, "%li", &MD2Delay[MD2Element]);

		/* Go to next field (search for a number) */
		MD2GetNumber();
		fscanf(MD2File, "%lf", &MD2Velocity[MD2Element]);

		/* Test for bad numbers */
		if (MD2Delay[MD2Element] < 0) MD2Status = 'F';
		if (MD2Velocity[MD2Element] < 0) MD2Status = 'F';
	}

	/* Clear calibration arrays if file error */
	if (MD2Status != 'O') {
		for (MD2Element = 1; MD2Element < 25; MD2Element++) {
			MD2Delay[MD2Element] = 0;
			MD2Velocity[MD2Element] = 0;
		}
	}

	/* Close file */
	if (fclose(MD2File) == EOF) {
		MD2Status = 'F';
		return;
	}

	/* Set MD2Calibrated flag */
	if (MD2Velocity[1] == 0) {
		MD2Calibrated = 0;
		MD2Status = 'F';
	}
	else {
		MD2Calibrated = -1;
	}

}


void MD2Inputs(void)
{

	/*---------------------------------------------------------------------
	'NAME:      MD2INPUTS
	'DESC:      THE MD2INPUTS PROCEDURE READS ALL INPUTS FROM ALL MD-2
	'           SYSTEMS INCLUDING HOME SWITCHES AND GENERAL PURPOSE INPUTS
	'           AND SETS VARIABLES ACCORDINGLY.  VARIABLES FOR PORTS THAT
	'           ARE NOT CONNECTED TO MD-2 SYSTEMS WILL CONTAIN USELESS
	'           INFORMATION.  EACH MD-2 SYSTEM HAS 2 HOME SWITCHES (ONE
	'           ASSOCIATED WITH EACH MOTOR) WHICH ARE INPUTS AND SOME
	'           MD-2 SYSTEMS HAVE 3 GENERAL PURPOSE INPUTS WHICH ARE
	'           USER DEFINED.  CHECK YOUR MANUAL TO SEE IF YOU HAVE
	'           GENERAL PURPOSE INPUTS.
	'USAGE:     USE TO READ THE CONDITION OF HOME SWITCHES, SENSORS
	'           AND OTHER INPUTS.
	'INPUTS:    NONE.
	'OUTPUTS:   INPUT CODES  -1=ACTIVE(LOGIC 0), 0=NOT ACTIVE(LOGIC 1).
	'
	'           VARIABLES  MD-2 PORT  INPUT           CONDITION
	'           ---------  ---------  -----           ---------
	'           MD2Input(11)  3BC     HOME SWITCH #1  -1=SWITCH ACTIVE (LOW).
	'           MD2Input(12)  3BC     HOME SWITCH #2  -1=SWITCH ACTIVE (LOW).
	'           MD2Input(13)  3BC     GEN PURP IN #1  -1=INPUT ACTIVE (LOW).
	'           MD2Input(14)  3BC     GEN PURP IN #2  -1=INPUT ACTIVE (LOW).
	'           MD2Input(15)  3BC     GEN PURP IN #3  -1=INPUT ACTIVE (LOW).
	'
	'           MD2Input(21)  378     HOME SWITCH #1  -1=SWITCH ACTIVE (LOW).
	'           MD2Input(22)  378     HOME SWITCH #2  -1=SWITCH ACTIVE (LOW).
	'           MD2Input(23)  378     GEN PURP IN #1  -1=INPUT ACTIVE (LOW).
	'           MD2Input(24)  378     GEN PURP IN #2  -1=INPUT ACTIVE (LOW).
	'           MD2Input(25)  378     GEN PURP IN #3  -1=INPUT ACTIVE (LOW).
	'
	'           MD2Input(31)  278     HOME SWITCH #1  -1=SWITCH ACTIVE (LOW).
	'           MD2Input(32)  278     HOME SWITCH #2  -1=SWITCH ACTIVE (LOW).
	'           MD2Input(33)  278     GEN PURP IN #1  -1=INPUT ACTIVE (LOW).
	'           MD2Input(34)  278     GEN PURP IN #2  -1=INPUT ACTIVE (LOW).
	'           MD2Input(35)  278     GEN PURP IN #3  -1=INPUT ACTIVE (LOW).
	'---------------------------------------------------------------------*/

	/* Clear all variables */
	MD2Input[11] = 0;
	MD2Input[12] = 0;
	MD2Input[13] = 0;
	MD2Input[14] = 0;
	MD2Input[15] = 0;
	MD2Input[21] = 0;
	MD2Input[22] = 0;
	MD2Input[23] = 0;
	MD2Input[24] = 0;
	MD2Input[25] = 0;
	MD2Input[31] = 0;
	MD2Input[32] = 0;
	MD2Input[33] = 0;
	MD2Input[34] = 0;
	MD2Input[35] = 0;

	/* Read inputs and set variables if inputs are active (logic 0) */

	/* MD-2 at 3bch */
	if ((inp(MD2MtrAdr12 + 1) & 0x20) == 0) MD2Input[11] = -1;
	if ((inp(MD2MtrAdr12 + 1) & 0x10) == 0) MD2Input[12] = -1;
	if ((inp(MD2MtrAdr12 + 1) & 0x80) != 0) MD2Input[13] = -1;
	if ((inp(MD2MtrAdr12 + 1) & 0x40) != 0) MD2Input[14] = -1;
	if ((inp(MD2MtrAdr12 + 1) & 0x08) != 0) MD2Input[15] = -1;

	/* MD-2 at 378h */
	if ((inp(MD2MtrAdr34 + 1) & 0x20) == 0) MD2Input[21] = -1;
	if ((inp(MD2MtrAdr34 + 1) & 0x10) == 0) MD2Input[22] = -1;
	if ((inp(MD2MtrAdr34 + 1) & 0x80) != 0) MD2Input[23] = -1;
	if ((inp(MD2MtrAdr34 + 1) & 0x40) != 0) MD2Input[24] = -1;
	if ((inp(MD2MtrAdr34 + 1) & 0x08) != 0) MD2Input[25] = -1;

	/* MD-2 at 278h */
	if ((inp(MD2MtrAdr56 + 1) & 0x20) == 0) MD2Input[31] = -1;
	if ((inp(MD2MtrAdr56 + 1) & 0x10) == 0) MD2Input[32] = -1;
	if ((inp(MD2MtrAdr56 + 1) & 0x80) != 0) MD2Input[33] = -1;
	if ((inp(MD2MtrAdr56 + 1) & 0x40) != 0) MD2Input[34] = -1;
	if ((inp(MD2MtrAdr56 + 1) & 0x08) != 0) MD2Input[35] = -1;

}


void MD2Outputs(void)
{

	/*---------------------------------------------------------------------
	'NAME:      MD2OUTPUTS
	'DESC:      THE MD2OUTPUTS PROCEDURE TURNS OUTPUT PINS ON AND OFF.
	'           THE INPUT/OUTPUT PORT ON THE BACK OF THE MD-2 SYSTEM
	'           HAS 2 OUTPUTS AND 3 INPUTS.  SOME MD-2 SYSTEMS DO NOT
	'           HAVE THIS I/O PORT.  CHECK YOUR MANUAL FOR MORE INFORMATION.
	'           WHEN TURNED ON, THE OUTPUT PIN WILL BE DRIVEN TO GROUND
	'           TO TURN ON SINKING LOADS SUCH AS RELAYS WHICH CAN CONTROL
	'           LARGER LOADS.  WHEN TURNED OFF, THEN OUTPUT PIN WILL
	'           RETURN TO A LOGIC 1 WHICH IS 5 VOLTS DC.
	'           MAKE SURE NOT TO EXCEED MAXIMUM CURRENT DRAW ON OUTPUT PINS.
	'           THE MD2OUTPUTCODE DETERMINES WHICH MD-2, PIN AND ON/OFF.
	'           USE THE FOLLOWING CHART TO DETERMINE WHICH CODE TO USE.
	'           NOTICE THE FIRST DIGIT OF THE CODE SELECTS WHICH MD-2 SYSTEM,
	'           THE SECOND DIGIT SELECTS THE PIN, AND THE THIRD SELECTS
	'           ON OR OFF (1 OR 0).
	'
	'           MD2OUTPUTCODE    MD-2 PORT    OUTPUT  ACTION
	'           -------------    ---------    ------  ------
	'               110             3BC         #1      OFF
	'               111             3BC         #1      ON
	'               120             3BC         #2      OFF
	'               121             3BC         #2      ON
	'
	'               210             378         #1      OFF
	'               211             378         #1      ON
	'               220             378         #2      OFF
	'               221             378         #2      ON
	'
	'               310             278         #1      OFF
	'               311             278         #1      ON
	'               320             278         #2      OFF
	'               321             278         #2      ON
	'
	'USAGE:     SET THE MD2OUTPUTCODE PARAMETER AND CALL.  USE TO CONTROL
	'           TOOLING SUCH AS DRILLS AND DISPENSERS.
	'INPUTS:    MD2OUTPUTCODE DETERMINES WHICH, MD-2, WHICH PIN, ON/OFF.
	'OUTPUTS:   NONE.
	'---------------------------------------------------------------------*/

	/* Port 3bc, output #1, off */
	if (MD2OutputCode == 110)
		outp((MD2MtrAdr12 + 2), (inp(MD2MtrAdr12 + 2) & 0xf7));

	/* Port 3bc, output #1, on */
	if (MD2OutputCode == 111)
		outp((MD2MtrAdr12 + 2), (inp(MD2MtrAdr12 + 2) | 0x08));

	/* Port 3bc, output #2, off */
	if (MD2OutputCode == 120)
		outp((MD2MtrAdr12 + 2), (inp(MD2MtrAdr12 + 2) & 0xfd));

	/* Port 3bc, output #2, on */
	if (MD2OutputCode == 121)
		outp((MD2MtrAdr12 + 2), (inp(MD2MtrAdr12 + 2) | 0x02));


	/* Port 378, output #1, off */
	if (MD2OutputCode == 210)
		outp((MD2MtrAdr34 + 2), (inp(MD2MtrAdr34 + 2) & 0xf7));

	/* Port 378, output #1, on */
	if (MD2OutputCode == 211)
		outp((MD2MtrAdr34 + 2), (inp(MD2MtrAdr34 + 2) | 0x08));

	/* Port 378, output #2, off */
	if (MD2OutputCode == 220)
		outp((MD2MtrAdr34 + 2), (inp(MD2MtrAdr34 + 2) & 0xfd));

	/* Port 378, output #2, on */
	if (MD2OutputCode == 221)
		outp((MD2MtrAdr34 + 2), (inp(MD2MtrAdr34 + 2) | 0x02));


	/* Port 278, output #1, off */
	if (MD2OutputCode == 310)
		outp((MD2MtrAdr56 + 2), (inp(MD2MtrAdr56 + 2) & 0xf7));

	/* Port 278, output #1, on */
	if (MD2OutputCode == 311)
		outp((MD2MtrAdr56 + 2), (inp(MD2MtrAdr56 + 2) | 0x08));

	/* Port 278, output #2, off */
	if (MD2OutputCode == 320)
		outp((MD2MtrAdr56 + 2), (inp(MD2MtrAdr56 + 2) & 0xfd));

	/* Port 278, output #2, on */
	if (MD2OutputCode == 321)
		outp((MD2MtrAdr56 + 2), (inp(MD2MtrAdr56 + 2) | 0x02));

	MD2Status = 'O';

}


void MD2On(void)
{

	/*---------------------------------------------------------------------
	'NAME:      MD2ON
	'DESC:      THE MD2ON PROCEDURE INITIALIZES A PARALLEL PRINTER PORT
	'           AND TURNS ON AN MD-2.
	'USAGE:     USE AT THE BEGINNING OF A MOTION CONTROL PROGRAM BUT
	'           AFTER THE MD2SETUP SUBROUTINE.
	'INPUTS:    MOTOR # DETERMINES PORT.
	'OUTPUTS:   NONE.
	'---------------------------------------------------------------------*/

	/* Check for setup */
	if (MD2MtrAdr12 != 0x3bc) {
		MD2Status = 'S';
		return;
	}

	/* Check for valid motor # */
	MD2Status = 'B';
	if ((MD2Motor == 1)  || (MD2Motor == 2)  || (MD2Motor == 3))  MD2Status='O';
	if ((MD2Motor == 4)  || (MD2Motor == 5)  || (MD2Motor == 6))  MD2Status='O';
	if ((MD2Motor == 12) || (MD2Motor == 34) || (MD2Motor == 56)) MD2Status='O';
	if (MD2Status != 'O') return;

	/* Check for port availability */
	MD2Status = 'P';
	if ((MD2Motor == 1) || (MD2Motor == 2) || (MD2Motor == 12))
		if (MD2Available12 == 0) return;
	if ((MD2Motor == 3) || (MD2Motor == 4) || (MD2Motor == 34))
		if (MD2Available34 == 0) return;
	if ((MD2Motor == 5) || (MD2Motor == 6) || (MD2Motor == 56))
		if (MD2Available56 == 0) return;

	/*
	'TURN OFF ALL MOTOR PHASES,
	'SET -STROBE LOW (MD-2 ENABLED),
	'    -ALF LOW (OUTPUT #1 OFF HIGH),
	'    -INIT LOW (STANDBY OFF),
	'    -SELIN LOW (OUTPUT #2 OF HIGH),
	'    IRQ DISABLED. */

	if ((MD2Motor == 1) || (MD2Motor == 2) || (MD2Motor == 12)) {
		outp(MD2MtrAdr12, 0xff);
		outp((MD2MtrAdr12 + 2), 0x05);
		MD2Enabled12 = -1;
		MD2Status = 'O';
	}

	if ((MD2Motor == 3) || (MD2Motor == 4) || (MD2Motor == 34)) {
		outp(MD2MtrAdr34, 0xff);
		outp((MD2MtrAdr34 + 2), 0x05);
		MD2Enabled34 = -1;
		MD2Status = 'O';
	}

	if ((MD2Motor == 5) || (MD2Motor == 6) || (MD2Motor == 56)) {
		outp(MD2MtrAdr56, 0xff);
		outp((MD2MtrAdr56 + 2), 0x05);
		MD2Enabled56 = -1;
		MD2Status = 'O';
	}

}


void MD2Off(void)
{

	/*---------------------------------------------------------------------
	'NAME:      MD2OFF
	'DESC:      THE MD2OFF PROCEDURE RETURNS A PARALLEL PRINTER PORT
	'           REFERENCED BY THE MOTOR # TO ITS PREVIOUS STATE READY
	'           FOR USE WITH A PRINTER AND DISABLES THE MD-2.
	'USAGE:     USE AT THE END OF A MOTION CONTROL PROGRAM.
	'INPUTS:    MOTOR # DETERMINES PORT.
	'OUTPUTS:   NONE.
	'---------------------------------------------------------------------*/

	/* Local variables */
	double MD2Time;

	/* Check for setup */
	if (MD2MtrAdr12 != 0x3bc) {
		MD2Status = 'S';
		return;
	}

	/* Check for valid motor # */
	MD2Status = 'B';
	if ((MD2Motor == 1)  || (MD2Motor == 2)  || (MD2Motor == 3))  MD2Status='O';
	if ((MD2Motor == 4)  || (MD2Motor == 5)  || (MD2Motor == 6))  MD2Status='O';
	if ((MD2Motor == 12) || (MD2Motor == 34) || (MD2Motor == 56)) MD2Status='O';
	if (MD2Status != 'O') return;

	/* Check for port availability */
	MD2Status = 'P';
	if ((MD2Motor == 1) || (MD2Motor == 2) || (MD2Motor == 12))
		if (MD2Available12 == 0) return;
	if ((MD2Motor == 3) || (MD2Motor == 4) || (MD2Motor == 34))
		if (MD2Available34 == 0) return;
	if ((MD2Motor == 5) || (MD2Motor == 6) || (MD2Motor == 56))
		if (MD2Available56 == 0) return;

	/* -STROBE PIN HIGH, -ALF PIN HIGH, -INIT PIN LOW,
	   -SELIN PIN LOW, IRQ OFF */
	if ((MD2Motor == 1) || (MD2Motor == 2) || (MD2Motor == 12))
		outp((MD2MtrAdr12 + 2), 0x04);
	if ((MD2Motor == 3) || (MD2Motor == 4) || (MD2Motor == 34))
		outp((MD2MtrAdr34 + 2), 0x04);
	if ((MD2Motor == 5) || (MD2Motor == 6) || (MD2Motor == 56))
		outp((MD2MtrAdr56 + 2), 0x04);

	/* Delay 1 second */
	MD2Time = (double) clock() / CLOCKS_PER_SEC;
	while ( (MD2Time + 1) > ( (double) clock() / CLOCKS_PER_SEC ) ) ;

	/* Turn -init pin high */
	if ((MD2Motor == 1) || (MD2Motor == 2) || (MD2Motor == 12)) {
		outp((MD2MtrAdr12 + 2), 0x0c);
		MD2Enabled12 = 0;
	}
	if ((MD2Motor == 3) || (MD2Motor == 4) || (MD2Motor == 34)) {
		outp((MD2MtrAdr34 + 2), 0x0c);
		MD2Enabled34 = 0;
	}
	if ((MD2Motor == 5) || (MD2Motor == 6) || (MD2Motor == 56)) {
		outp((MD2MtrAdr56 + 2), 0x0c);
		MD2Enabled56 = 0;
	}

	MD2Status = 'O';

}


void MD2StandbyOn(void)
{

	/*---------------------------------------------------------------------
	'NAME:      MD2STANDBYON
	'DESC:      THE MD2STANDBYON PROCEDURE PUTS AN MD-2 INTO STANDBY MODE
	'           WHICH REDUCES THE MOTOR CURRENT BY 50%.  THIS RESULTS
	'           IN LOWER HEAT DISSIPATION DURING STANDSTILL WITHOUT
	'           LOSSING ALL HOLDING TORQUE.
	'           THE MD-2 SYSTEM IS SELECTED BY THE MD2MOTOR PARAMETER.
	'           BOTH MOTORS ON AN MD-2 ARE AFFECTED.
	'           MAKE SURE THAT MOTORS ARE ENERGIZED BY FIRST MOVING WITH
	'           MD2HOLD=-1 OR STANDBY WILL NOT WORK.
	'           SOME MD-2 SYSTEMS DO NOT HAVE THIS FEATURE, CHECK WITH
	'           YOUR MANUAL FOR MORE INFORMATION.
	'USAGE:     USE AFTER A MOTION TO MAINTAIN SOME HOLDING TORQUE WHILE
	'           REDUCING HEAT.
	'INPUTS:    MOTOR # DETERMINES MD-2.
	'OUTPUTS:   NONE
	'---------------------------------------------------------------------*/

	/* Check for setup */
	if (MD2MtrAdr12 != 0x3bc) {
		MD2Status = 'S';
		return;
	}

	/* Check for valid motor # */
	MD2Status = 'B';
	if ((MD2Motor == 1)  || (MD2Motor == 2)  || (MD2Motor == 3))  MD2Status='O';
	if ((MD2Motor == 4)  || (MD2Motor == 5)  || (MD2Motor == 6))  MD2Status='O';
	if ((MD2Motor == 12) || (MD2Motor == 34) || (MD2Motor == 56)) MD2Status='O';
	if (MD2Status != 'O') return;

	/* Check for port availability */
	MD2Status = 'P';
	if ((MD2Motor == 1) || (MD2Motor == 2) || (MD2Motor == 12))
		if (MD2Available12 == 0) return;
	if ((MD2Motor == 3) || (MD2Motor == 4) || (MD2Motor == 34))
		if (MD2Available34 == 0) return;
	if ((MD2Motor == 5) || (MD2Motor == 6) || (MD2Motor == 56))
		if (MD2Available56 == 0) return;

	/* Check for enabled MD-2. */
	MD2Status = 'E';
	if ((MD2Motor == 1) || (MD2Motor == 2) || (MD2Motor == 12))
		if (MD2Enabled12 == 0) return;
	if ((MD2Motor == 3) || (MD2Motor == 4) || (MD2Motor == 34))
		if (MD2Enabled34 == 0) return;
	if ((MD2Motor == 5) || (MD2Motor == 6) || (MD2Motor == 56))
		if (MD2Enabled56 == 0) return;

	/* Turn standby mode on */
	if ((MD2Motor == 1) || (MD2Motor == 2) || (MD2Motor == 12))
		outp((MD2MtrAdr12 + 2), (inp(MD2MtrAdr12 + 2) & 0xfb));
	if ((MD2Motor == 3) || (MD2Motor == 4) || (MD2Motor == 34))
		outp((MD2MtrAdr34 + 2), (inp(MD2MtrAdr34 + 2) & 0xfb));
	if ((MD2Motor == 5) || (MD2Motor == 6) || (MD2Motor == 56))
		outp((MD2MtrAdr56 + 2), (inp(MD2MtrAdr56 + 2) & 0xfb));

	MD2Status = 'O';

}


void MD2StandbyOff(void)
{

	/*---------------------------------------------------------------------
	'NAME:      MD2STANDBYOFF
	'DESC:      THE MD2STANDBYOFF PROCEDURE RETURNS AN MD-2 TO FULL
	'           CURRENT MODE.
	'           THE MD-2 IS SELECTED BY THE MD2MOTOR PARAMETER.
	'           BOTH MOTORS ON THE MD-2 IS AFFECTED
	'           SOME MD-2 SYSTEMS DO NOT HAVE THIS FEATURE, CHECK WITH
	'           YOUR MANUAL FOR MORE INFORMATION.
	'USAGE:     USE TO RETURN AN MD-2 TO FULL CURRENT MODE BEFORE MOVING
	'           THEM FOR MAXIMUM PERFORMANCE.
	'INPUTS:    MOTOR # DETERMINES MD-2.
	'OUTPUTS:   NONE.
	'---------------------------------------------------------------------*/

	/* Check for setup */
	if (MD2MtrAdr12 != 0x3bc) {
		MD2Status = 'S';
		return;
	}

	/* Check for valid motor # */
	MD2Status = 'B';
	if ((MD2Motor == 1)  || (MD2Motor == 2)  || (MD2Motor == 3))  MD2Status='O';
	if ((MD2Motor == 4)  || (MD2Motor == 5)  || (MD2Motor == 6))  MD2Status='O';
	if ((MD2Motor == 12) || (MD2Motor == 34) || (MD2Motor == 56)) MD2Status='O';
	if (MD2Status != 'O') return;

	/* Check for port availability */
	MD2Status = 'P';
	if ((MD2Motor == 1) || (MD2Motor == 2) || (MD2Motor == 12))
		if (MD2Available12 == 0) return;
	if ((MD2Motor == 3) || (MD2Motor == 4) || (MD2Motor == 34))
		if (MD2Available34 == 0) return;
	if ((MD2Motor == 5) || (MD2Motor == 6) || (MD2Motor == 56))
		if (MD2Available56 == 0) return;

	/* Check for enabled MD-2. */
	MD2Status = 'E';
	if ((MD2Motor == 1) || (MD2Motor == 2) || (MD2Motor == 12))
		if (MD2Enabled12 == 0) return;
	if ((MD2Motor == 3) || (MD2Motor == 4) || (MD2Motor == 34))
		if (MD2Enabled34 == 0) return;
	if ((MD2Motor == 5) || (MD2Motor == 6) || (MD2Motor == 56))
		if (MD2Enabled56 == 0) return;

	/* Turn standby mode off */
	if ((MD2Motor == 1) || (MD2Motor == 2) || (MD2Motor == 12))
		outp((MD2MtrAdr12 + 2), (inp(MD2MtrAdr12 + 2) || 0x04));
	if ((MD2Motor == 3) || (MD2Motor == 4) || (MD2Motor == 34))
		outp((MD2MtrAdr34 + 2), (inp(MD2MtrAdr34 + 2) || 0x04));
	if ((MD2Motor == 5) || (MD2Motor == 6) || (MD2Motor == 56))
		outp((MD2MtrAdr56 + 2), (inp(MD2MtrAdr56 + 2) || 0x04));

	MD2Status = 'O';

}


void MD2ParSave(void)
{

	/* -------------------------------------------------------------------
	'Name:       MD2ParSave
	'Desc:       This procedure is used to save all motor parameters
	'            to a disk file.  when used with the MD2ParLoad procedure,
	'            a program can be created which saves its status before
	'            powering down and restores when comming back on eliminating
	'            the need for the user to re-enter all motor parameters.
	'Usage:      Set the desired file name in MD2ParFile and call.
	'Inputs:     File name and all motor parameters.
	'Outputs:    Status = o for ok or e for error.
	'------------------------------------------------------------------*/

	/* Local variables */
	int MD2Mtr;             /* Motor counter */

	/* Default return status to ok */
	MD2Status = 'O';

	/* Open file - if error then done else continue */
	if ((MD2File = fopen( MD2ParFile, "w" )) == NULL) {
		MD2Status = 'F';
		return;
	}

	/* Motor parameters */
	fprintf (MD2File, "\"%s\"\n", MD2ParFile);
	fprintf (MD2File, "%i, %i, %i, \"%c\", \"%c\"\n", MD2Hold, MD2Interrupts, MD2Motor, MD2MoveType, MD2StepType);

	/* Motor parameter arrays */
	for (MD2Mtr = 1; MD2Mtr <= 6; MD2Mtr++) {
		if (MD2Status == 'F') break;
		fprintf (MD2File, "%i,", MD2Mtr);
		fprintf (MD2File, " %lf,", MD2Backlash[MD2Mtr]);
		fprintf (MD2File, " %i,", MD2HomeDir[MD2Mtr]);
		fprintf (MD2File, " %lf,", MD2HomeOffset[MD2Mtr]);
		fprintf (MD2File, " %lf,", MD2LimitF[MD2Mtr]);
		fprintf (MD2File, " %lf,", MD2LimitR[MD2Mtr]);
		fprintf (MD2File, " %lf,", MD2MaxSpeed[MD2Mtr]);
		fprintf (MD2File, " %lf,", MD2MinSpeed[MD2Mtr]);
		fprintf (MD2File, " \"%s\",", MD2MotorName[MD2Mtr]);
		fprintf (MD2File, " %lf,", MD2Position[MD2Mtr]);
		fprintf (MD2File, " %lf,", MD2Slope[MD2Mtr]);
		fprintf (MD2File, " %lf,", MD2Target[MD2Mtr]);
		fprintf (MD2File, " %li,", MD2Units[MD2Mtr]);
		fprintf (MD2File, " \"%s\",", MD2UnitName[MD2Mtr]);
		fprintf (MD2File, " %i,", MD2PatPtr[MD2Mtr]);
		fprintf (MD2File, " %i\n", MD2LastDir[MD2Mtr]);
	}

	/* Grid parameters */
	fprintf (MD2File, "%lf,", MD2GridBeginX);
	fprintf (MD2File, " %lf,", MD2GridBeginY);
	fprintf (MD2File, " %lf,", MD2GridSpaceX);
	fprintf (MD2File, " %lf,", MD2GridSpaceY);
	fprintf (MD2File, " %i,", MD2GridTargetX);
	fprintf (MD2File, " %i\n", MD2GridTargetY);

	/* Circle parameters */
	fprintf (MD2File, "%lf,", MD2CircleRadiusX);
	fprintf (MD2File, " %lf,", MD2CircleRadiusY);
	fprintf (MD2File, " %lf,", MD2CircleCenterX);
	fprintf (MD2File, " %lf,", MD2CircleCenterY);
	fprintf (MD2File, " %i,", MD2CircleStart);
	fprintf (MD2File, " %i,", MD2CircleArc);
	fprintf (MD2File, " %i\n", MD2CircleChord);

	/* Finish up */
	if (fclose(MD2File) == EOF) MD2Status = 'F';
}


void MD2Calibrate(void)
{

    /*---------------------------------------------------------------------
	'Name:      MD2Calibrate
	'Desc:      This procedure calibrates motor speeds on this particular
	'           computer.  the calibration information is stored in the
	'           arrays MD2Velocity(24) and MD2Delay(24) then saved to disk
	'           in the file name stored in MD2CalFile.  calibration
	'           information is read by the MD2Setup sub at the beginning
	'           of each motion program and is used when moving motors.
	'           this is transparent to the user and programmer.
	'           without calibration, motor speeds would vary depending on
	'           computer speed and other factors.
	'           it is only necessary to calibrate on a particular computer
	'           once.  MD2Setup must be used first.
	'           MD2Velocity(1)=fastest possible speed in steps per second.
	'           MD2Velocity(24)=slowest possible speed in steps per second
	'Usage:     Simply call and wait several minuites.
	'Inputs:    MD2CalFile = calibration file name usually "MD2.cal"
	'Outputs:   File containing calibration parameters,arrays,MD2Status.
	'---------------------------------------------------------------------*/

	/* Local variables */
	int MD2Element;         /* Element counter */
	double MD2STime;        /* Beginning time */
	double MD2ETime;        /* Ending time */
	long int MD2LDelay;     /* Delay */
	int MD2TmpMotor;        /* Temp variable storage */
	char MD2TmpMoveType;
	double MD2TmpSlope;
	double MD2TmpBacklash;
	long int MD2TmpUnits;
	double MD2TmpLimitF;
	double MD2TmpLimitR;
	double MD2TmpTarget;
	double MD2TmpPosition;
                      
	/* Check for setup */
	if (MD2MtrAdr12 != 0x3bc) {
		MD2Status = 'S';
		return;
	}

	/* Save parameters */
	MD2TmpMotor = MD2Motor;
	MD2TmpMoveType = MD2MoveType;
	MD2TmpSlope = MD2Slope[1];
	MD2TmpBacklash = MD2Backlash[1];
	MD2TmpUnits = MD2Units[1];
	MD2TmpLimitF = MD2LimitF[1];
	MD2TmpLimitR = MD2LimitR[1];
	MD2TmpTarget = MD2Target[1];
	MD2TmpPosition = MD2Position[1];
   
	/* Initialize motor parameters and variables */
	MD2Motor = 1;
	MD2MoveType = 'R';
	MD2Slope[1] = 0;
	MD2Backlash[1] = 0;
	MD2Units[1] = 1;
	MD2LimitF[1] = 2000000000.0;
	MD2LimitR[1] = -2000000000.0;
	MD2LDelay = 1L;
	MD2CalMode = -1;
	MD2Status = 'O';
	MD2Off();

	/* For each delay (1,2,4,8,etc), find velocity and place in array */
	for( MD2Element = 1; MD2Element < 25; MD2Element++) {

		/* Initialize variables */
        MD2Target[1] = 1;
        MD2MaxSpeed[1] = MD2LDelay;
		MD2Delay[MD2Element] = MD2LDelay;

MD2Caloop:

		/* Time the move to determine speed */

		/* Wait until edge of timer tick */
		MD2STime = (double) clock() / CLOCKS_PER_SEC;
		while ( (MD2STime) == ( (double) clock() / CLOCKS_PER_SEC ) ) ;

		/* Get starting time */
		MD2STime = (double) clock() / CLOCKS_PER_SEC;

		/* Move */
		MD2Move();

		/* Get ending time */
		MD2ETime = (double) clock() / CLOCKS_PER_SEC;

		/* If move error then bail out */
		if ( MD2Status != 'O') break;

		/* Try again if not 10 seconds or more of test */
		if ((MD2ETime - MD2STime) <= 10.0) {
            MD2Target[1] = MD2Target[1] + MD2Target[1];
			goto MD2Caloop;
		}

		/* Put velocity in array */
		MD2Velocity[MD2Element] = MD2Target[1] / (MD2ETime - MD2STime);
       
		/* Quit if last measurement was slower than .01sps. */
		if ( MD2Velocity[MD2Element] <= 0.01 ) {
			/* Fill remaining elements with last entry. */
			do  {
                MD2Delay[MD2Element] = MD2LDelay;
                MD2Velocity[MD2Element] = MD2Target[1] / (MD2ETime - MD2STime);
                MD2Element = MD2Element + 1;
			}
			while ( MD2Element < 25 );
            break;
		}

		/* Do next delay */
        MD2LDelay = MD2LDelay + MD2LDelay;

	}   /*  next MD2Element */
   
    MD2CalMode = 0;
  
	if ( MD2Status == 'O') {
		/* Save calibration parameters if no failure */
		MD2File = fopen (MD2CalFile, "w");
		if (!MD2File) MD2Status = 'F';
		for (MD2Element = 1; MD2Element < 25; MD2Element++) {
			if (MD2Status == 'F') break;
			fprintf (MD2File, "%li, %lf\n",
                   MD2Delay[MD2Element],
                   MD2Velocity[MD2Element]);
		}
		if (fclose(MD2File) == EOF) MD2Status = 'F';
		MD2Calibrated = -1;
	}
	else {
		MD2Calibrated = 0;
	}

	/* Restore parameters */
	MD2MoveType = MD2TmpMoveType;
	MD2Motor = MD2TmpMotor;
	MD2Slope[1] = MD2TmpSlope;
	MD2Backlash[1] = MD2TmpBacklash;
	MD2Units[1] = MD2TmpUnits;
	MD2LimitF[1] = MD2TmpLimitF;
	MD2LimitR[1] = MD2TmpLimitR;
	MD2Target[1] = MD2TmpTarget;
	MD2Position[1] = MD2TmpPosition;
}


void MD2Circle(void)
{

	/* --------------------------------------------------------------------
	'Name:      MD2Circle
	'Desc:      The MD2Circle will move 2 motors which are attached to an
	'           xy positioning table in a circular, arc or ellipse pattern.
	'           the 2 motors must be 1 & 2, 3 & 4 or 5 & 6.
	'           the user can set the chord angle in degrees which determines
	'           the resolution of the circle.  a chord angle of 45 will
	'           produce an octagon.  a chord angle of 1 degree will produce
	'           a circle with 360 sides.
	'           various arc angles can also be given which will produce
	'           parts of a circle. an arc angle of 90 will produce an arc
	'           of 90 degrees.  360 will produce a complete circle.
	'           x & y radius values are provided to produce an ellipse.
	'           if both x & y radius values are the same, a circle will
	'           result.
	'           positive arc angles will produce counter-clockwise rotation,
	'           negative values clockwise.
	'           moves are performed at the minimum speed with no ramping.
	'           to make a circle or arc at an absolute position, set the
	'           CenterX,y parameters to the absolute center positions and
	'           set MD2MoveType to "A".
	'           to make a circle or arc relative to the current position,
	'           set CenterX,y to offsets from current motor positions and
	'           set MD2MoveType to "R".
	'           ramping is disabled.  motors will move at minimum speed.
	'
	'               parameter            description
	'           ----------------    -----------------------
	'           MD2CircleRadiusx    x-axis radius in units.
	'           MD2CircleRadiusY    y-axis radius in units.
	'           MD2CircleCenterX    x-axis center in units, rel or abs.
	'           MD2CircleCenterY    y-axis center in units, rel or abs.
	'           MD2CircleStart      starting angle in degrees.
	'           MD2CircleArc        arc angle in degrees. 360=circle.
	'                               +=ccw, -=cw rotation direction.
	'           MD2CircleChord      chord angle in degrees.
	'           MD2Motor            12, 34 or 45.
	'           MD2MoveType         "A"=absolute, "R"=relative.
	'
	'Usage:     Set circle parameters and motor parameters then call.
	'Inputs:    All motor parameters, circle parameters.
	'Outputs:   MD2Status, MD2Target(m), MD2Position(m)
	'---------------------------------------------------------------------*/

	/* Local variables */
	int MD2EndAngle;        /* End angle */
	int MD2Angle;           /* Current angle */
	double MD2AngleR;       /* Current angle in radians */
	int MD2Motor1;          /* 1st motor # */
	int MD2Motor2;          /* 2nd motor # */
	char MD2TmpMoveType;    /* Save movetype */
	double MD2TmpSlope1;    /* Save slope 1 */
	double MD2TmpSlope2;    /* Save slope 2 */
	double MD2TmpSpeed1;    /* Save max speed 1 */
	double MD2TmpSpeed2;    /* Save max speed 2 */
	int MD2TmpHold;         /* Save hold */
	double MD2CntrX;        /* Center x */
	double MD2CntrY;        /* Center y */

	/* Check for bad parameters */
	MD2Status = 'B';
	if ((MD2Motor != 12) && (MD2Motor != 34) && (MD2Motor != 56)) return;
	if ((MD2CircleArc == 0) || (abs(MD2CircleArc) > 360)) return;
	if ((MD2CircleChord <= 0) || (MD2CircleChord > 180)) return;
	if ((MD2CircleStart > 360) || (MD2CircleStart < 0)) return;
	MD2MoveType = (char)toupper(MD2MoveType);
	if ((MD2MoveType != 'A') && (MD2MoveType != 'R')) return;
	MD2Status = 'O';

	/* Initialize variables. */
	if (MD2Motor == 12) { MD2Motor1 = 1; MD2Motor2 = 2;}
	if (MD2Motor == 34) { MD2Motor1 = 3; MD2Motor2 = 4;}
	if (MD2Motor == 56) { MD2Motor1 = 5; MD2Motor2 = 6;}
   
	/* Save parameters */
	MD2TmpMoveType = MD2MoveType;
	MD2TmpSlope1 = MD2Slope[MD2Motor1];
	MD2TmpSlope2 = MD2Slope[MD2Motor2];
	MD2TmpSpeed1 = MD2MaxSpeed[MD2Motor1];
	MD2TmpSpeed2 = MD2MaxSpeed[MD2Motor2];
	MD2TmpHold = MD2Hold;

	/* Setup motor parameters */
	if (MD2MoveType == 'R') {
		/* Relative center positions */
		MD2CntrX = MD2Position[MD2Motor1] + MD2CircleCenterX;
		MD2CntrY = MD2Position[MD2Motor2] + MD2CircleCenterY;
	}
	else {
		/* Absolute center positions */
		MD2CntrX = MD2CircleCenterX;
		MD2CntrY = MD2CircleCenterY;
	}
	MD2MoveType = 'A';
   
	MD2Slope[MD2Motor1] = 0;
	MD2Slope[MD2Motor2] = 0;
	MD2MaxSpeed[MD2Motor1] = MD2MinSpeed[MD2Motor1];
	MD2MaxSpeed[MD2Motor2] = MD2MinSpeed[MD2Motor2];
	MD2Hold = -1;
   
	/* Set angle and end angle */
	MD2EndAngle = MD2CircleStart + MD2CircleArc;
	MD2Angle = MD2CircleStart;
   
	/* Plot based on direction */
	if (MD2CircleArc > 0) {

		/* ccw plot loop */

		do {

			/* Convert to radians */
			MD2AngleR = (double)MD2Angle * (3.1416 / 180.0);

			/* Calculate x & y and move */
			MD2Target[MD2Motor1] = (MD2CircleRadiusX * cos(MD2AngleR)) + MD2CntrX;
			MD2Target[MD2Motor2] = (MD2CircleRadiusY * sin(MD2AngleR)) + MD2CntrY;

			/* Move the motors */
			MD2Move();

			if (MD2Status != 'O') break;
			MD2Angle = MD2Angle + MD2CircleChord;

		} while(MD2Angle < MD2EndAngle);
	}
	else {

		/* cw plot loop */

		do {

			/* Convert to radians */
			MD2AngleR = (double)MD2Angle * (3.1416 / 180.0);

			/* Calculate x & y and move */
			MD2Target[MD2Motor1] = (MD2CircleRadiusX * cos(MD2AngleR)) + MD2CntrX;
			MD2Target[MD2Motor2] = (MD2CircleRadiusY * sin(MD2AngleR)) + MD2CntrY;

			/* Move the motors */
			MD2Move();

			if (MD2Status != 'O') break;
			MD2Angle = MD2Angle - MD2CircleChord;

		} while(MD2Angle > MD2EndAngle);
	}
   
	/* Plot last angle unless previous error */
	if (MD2Status == 'O') {

		/* Last point */
		MD2Angle = MD2EndAngle;

		/* Convert to radians */
		MD2AngleR = (double)MD2Angle * (3.1416 / 180.0);

		/* Calculate x & y and move */
		MD2Target[MD2Motor1] = (MD2CircleRadiusX * cos(MD2AngleR)) + MD2CntrX;
		MD2Target[MD2Motor2] = (MD2CircleRadiusY * sin(MD2AngleR)) + MD2CntrY;

		/* Move the motors */
		MD2Move();
	}
   
	/* Restore parameters */
	MD2MoveType = MD2TmpMoveType;
	MD2Slope[MD2Motor1] = MD2TmpSlope1;
	MD2Slope[MD2Motor2] = MD2TmpSlope2;
	MD2MaxSpeed[MD2Motor1] = MD2TmpSpeed1;
	MD2MaxSpeed[MD2Motor2] = MD2TmpSpeed2;
	MD2Hold = MD2TmpHold;
  
	/* Power off motor if desired */
	if (MD2Hold == 0) {
		if (MD2Motor == 12) outp(MD2MtrAdr12, 0xff);
		if (MD2Motor == 34) outp(MD2MtrAdr34, 0xff);
		if (MD2Motor == 56) outp(MD2MtrAdr56, 0xff);
	}

}


void MD2Grid(void)
{

	/*---------------------------------------------------------------------
	'Name:      MD2Grid
	'Desc:      The MD2Grid procedure is used to move motors to a location
	'           of a grid.  a grid is an x y matrix of locations like
	'           a checker board.  this allows a programmer to move to
	'           a row & column of a grid instead of moving by passing
	'           MD2Grid works with absolute target positions only.
	'           the specific target location in steps or inches, etc.
	'           the following parameters define the grid.
	'
	'               parameter       description
	'               ---------       -----------
	'               MD2GridBeginX   x begin absolute position in units
	'               MD2GridBeginY   y begin absolute position in units
	'               MD2GridSpaceX   x spacing in units.
	'               MD2GridSpaceY   y spacing in units.
	'               MD2GridTargetX  x target, absolute.
	'               MD2GridTargetY  y target, absolute.
	'
	'Usage:     Set grid parameters to define the grid, set MD2Motor
	'           to a valid motor pair (12,34 or 56), set motor parameters
	'           then call.
	'Inputs:    All grid and motor parameters.
	'Outputs:   MD2Status,MD2Target(m),MD2Position(m).
	'---------------------------------------------------------------------*/

	/* Local variables */
	int MD2x;           /* x motor # */
	int MD2y;           /* y motor # */
	char MD2mt;         /* movetype storage */

	/* Check for bad parameters */
	MD2Status = 'B';
	if ((MD2Motor != 12) && (MD2Motor != 34) && (MD2Motor != 56)) return;
	MD2Status = 'O';

	/* Set motor # */
	if (MD2Motor == 12) { MD2x = 1; MD2y = 2;}
	if (MD2Motor == 34) { MD2x = 3; MD2y = 4;}
	if (MD2Motor == 56) { MD2x = 5; MD2y = 6;}

	/* Set target positions */
	MD2Target[MD2x] = MD2GridBeginX + ((MD2GridTargetX - 1) * MD2GridSpaceX);
	MD2Target[MD2y] = MD2GridBeginY + ((MD2GridTargetY - 1) * MD2GridSpaceY);

	/* Grid move */
	MD2mt = MD2MoveType;    /* Save movetype */
	MD2MoveType = 'A';      /* Absolute moves */
	MD2Move();              /* Move to the grid coordinates */
	MD2MoveType = MD2mt;    /* Restore movetype */

}


void MD2Home(void)
{

	/*---------------------------------------------------------------------
	'name:      MD2Home
	'desc:      The MD2Home procedure is used to move the stepper motor
	'           to a known home position.  all other moves are relative
	'           to this home (zero) position.  the selected motor is
	'           moved reverse until the switch is activated, then forward
	'           until deactivated, then moved forward according to the
	'           MD2HomeOffset(m) parameter. the current position is then
	'           set to zero - this is the home position.  the MD2MinSpeed
	'           parameter is used for the motor speed - no ramping is done.
	'           only 1 motor (1-6) can be home'd at a time.
	'usage:     Set the desired motor #, motor parameters and call.
	'inputs:    Motor # and motor parameters.
	'outputs:   Current position set to zero, MD2Status.
	'---------------------------------------------------------------------*/

	/* Local variables */
	long int MD2LDelay;         /* Delay counter between steps */
	int MD2Dir;                 /* Direction. -1=rev, 1=fwd */
	int MD2MtrAdr;              /* Motor port address */
	int MD2Pats;                /* Calculated step pattern */
	int MD2PMask;               /* Selected step pattern mask */
	int MD2OMask;               /* Other motor pattern mask */
	int MD2SMask;               /* Switch mask */
	int MD2IntReg;              /* Interrupt register */
	long int MD2I;              /* Temporary */
	double MD2Way;              /* Temp for velocity interolation */
	long int MD2MinDelay;       /* Calculated delay counter */

	/* Check for setup */
	if (MD2MtrAdr12 != 0x3bc) {
		MD2Status = 'S';
		return;
	}

	/* Check for calibration */
	if (MD2Calibrated == 0) {
		MD2Status = 'C';
		return;
	}

	/* Check for port availability */
	MD2Status = 'P';
	if ((MD2Motor == 1) || (MD2Motor == 2) || (MD2Motor == 12))
		if (MD2Available12 == 0) return;
	if ((MD2Motor == 3) || (MD2Motor == 4) || (MD2Motor == 34))
		if (MD2Available34 == 0) return;
	if ((MD2Motor == 5) || (MD2Motor == 6) || (MD2Motor == 56))
		if (MD2Available56 == 0) return;

	/* Check for enabled MD-2. */
	MD2Status = 'E';
	if ((MD2Motor == 1) || (MD2Motor == 2) || (MD2Motor == 12))
		if (MD2Enabled12 == 0) return;
	if ((MD2Motor == 3) || (MD2Motor == 4) || (MD2Motor == 34))
		if (MD2Enabled34 == 0) return;
	if ((MD2Motor == 5) || (MD2Motor == 6) || (MD2Motor == 56))
		if (MD2Enabled56 == 0) return;

	/* Check for valid motor # */
	MD2Status = 'B';
	if ((MD2Motor == 1) || (MD2Motor == 2) || (MD2Motor == 3)) MD2Status = 'O';
	if ((MD2Motor == 4) || (MD2Motor == 5) || (MD2Motor == 6)) MD2Status = 'O';
	if (MD2Status != 'O') return;

	/* Check for valid step type */
	MD2StepType = (char)toupper(MD2StepType);
	if ((MD2StepType != 'H') && (MD2StepType != 'D') && (MD2StepType != 'S')) {
		MD2Status = 'B';
		return;
	}

	/* Initialize status */
	MD2Status = 'O';

	/* Set step patterns */
	if (MD2StepType == 'H') {   /* Half step mode */
		MD2StpPat[0] = 0x66; MD2StpPat[1] = 0x77;
		MD2StpPat[2] = 0x33; MD2StpPat[3] = 0xbb;
		MD2StpPat[4] = 0x99; MD2StpPat[5] = 0xdd;
		MD2StpPat[6] = 0xcc; MD2StpPat[7] = 0xee;
	}
	if (MD2StepType == 'S') {   /* Single phase full step mode */
		MD2StpPat[0] = 0x77; MD2StpPat[1] = 0xbb;
		MD2StpPat[2] = 0xdd; MD2StpPat[3] = 0xee;
		MD2StpPat[4] = 0x77; MD2StpPat[5] = 0xbb;
		MD2StpPat[6] = 0xdd; MD2StpPat[7] = 0xee;
	}
	if (MD2StepType == 'D') {   /* Double phase full step mode */
		MD2StpPat[0] = 0x66; MD2StpPat[1] = 0x33;
		MD2StpPat[2] = 0x99; MD2StpPat[3] = 0xcc;
		MD2StpPat[4] = 0x66; MD2StpPat[5] = 0x33;
		MD2StpPat[6] = 0x99; MD2StpPat[7] = 0xcc;
	}

	/* Set up address */
	if ((MD2Motor == 1) || (MD2Motor == 2)) MD2MtrAdr = MD2MtrAdr12;
	if ((MD2Motor == 3) || (MD2Motor == 4)) MD2MtrAdr = MD2MtrAdr34;
	if ((MD2Motor == 5) || (MD2Motor == 6)) MD2MtrAdr = MD2MtrAdr56;

	/* Set up pattern mask, other motor's mask and switch mask */
	if ((MD2Motor == 1) || (MD2Motor == 3) || (MD2Motor == 5)) {
		MD2PMask = 0xf;
		MD2OMask = 0xf0;
		MD2SMask = 0x20;
	}
	else {
		MD2PMask = 0xf0;
		MD2OMask = 0xf;
		MD2SMask = 0x10;
	}

	/* Convert minspeed velocity to delay counts */
	for (MD2I = 2; MD2I < 24; MD2I++)    /* Search for correct elements */
		if ((MD2MinSpeed[MD2Motor] * MD2Units[MD2Motor]) >= MD2Velocity[MD2I]) break;

	/* Interpolate */
	MD2Way = ((MD2MinSpeed[MD2Motor] * MD2Units[MD2Motor]) - MD2Velocity[MD2I - 1]) / (MD2Velocity[MD2I] - MD2Velocity[MD2I - 1]);
	MD2MinDelay = (long int)((((MD2Delay[MD2I] - MD2Delay[MD2I - 1]) * MD2Way) + MD2Delay[MD2I - 1]) + .5);

	/* Select direction to home*/
	if (MD2HomeDir[MD2Motor] == 0) MD2Dir = -1; else MD2Dir = 1;
   
	/* Turn off com interrupts */
	if (MD2Interrupts == 0) {
		MD2IntReg = inp(0x21);
		outp(0x21, MD2IntReg | 0x98);
	}

	/*---------------------------------------------*/
	/*  Move motor reverse until switch activated  */
	/*---------------------------------------------*/

	/* Loop until switch is activated */
	while ((inp(MD2MtrAdr + 1) & MD2SMask) != 0) {

		/* Quit moving if key pressed */
		if (kbhit()) {
			getch();
			MD2Status = 'K';
			break;
		}

		/* Point to the next step pattern */
		MD2PatPtr[MD2Motor] = (MD2PatPtr[MD2Motor] + MD2Dir) & 0x7;

		/* Get step pattern and mask off unneeded bits */
		MD2Pats = MD2StpPat[MD2PatPtr[MD2Motor]] & MD2PMask;

		/* Don't disturb other motors bits */
		MD2Pats = MD2Pats | (inp(MD2MtrAdr) & MD2OMask);

		/* Output the step pattern to move the motor */
		outp(MD2MtrAdr, MD2Pats);

		/* Delay between steps */
		for (MD2LDelay = 1; MD2LDelay <= MD2MinDelay; MD2LDelay++) ;

	}

	/*----------------------------------------------*/
	/* Move motor forward until switch deactivated  */
	/*----------------------------------------------*/

	/* Loop until switch is deactivated.*/
	while ((inp(MD2MtrAdr + 1) & MD2SMask) == 0) {

		/* Quit moving if key pressed */
		if (kbhit()) {
			getch();
			MD2Status = 'K';
			break;
		}
		/* Point to the next step pattern */
		MD2PatPtr[MD2Motor] = (MD2PatPtr[MD2Motor] - MD2Dir) & 0x7;

		/* Get step pattern and mask off unneeded bits */
		MD2Pats = MD2StpPat[MD2PatPtr[MD2Motor]] & MD2PMask;

		/* Don't disturb other motors bits */
		MD2Pats = MD2Pats | (inp(MD2MtrAdr) & MD2OMask);
        
		/* Output the step pattern to move the motor */
		outp(MD2MtrAdr, MD2Pats);

		/* Delay between steps */
		for (MD2LDelay = 1; MD2LDelay <= MD2MinDelay; MD2LDelay++) ;

	}
   
	/*------------------------------------------------*/
	/* Over step home position using MD2HomeOffset(m) */
	/*------------------------------------------------*/

	/* Step until count depleted */
	MD2I = (long int)(MD2HomeOffset[MD2Motor] * MD2Units[MD2Motor]);
	while (MD2I != 0) {
        
		/* Quit moving if key pressed */
		if (kbhit()) {
			getch();
			MD2Status = 'K';
			break;
		}

		/* Point to the next step pattern */
		MD2PatPtr[MD2Motor] = (MD2PatPtr[MD2Motor] - MD2Dir) & 0x7;
       
		/* Get step pattern & mask off unneeded bits */
		MD2Pats = MD2StpPat[MD2PatPtr[MD2Motor]] & MD2PMask;

		/* Don't disturb other motors bits */
		MD2Pats = MD2Pats | (inp(MD2MtrAdr) & MD2OMask);

		/* Output the step pattern to move the motor */
		outp(MD2MtrAdr, MD2Pats);

		/* Delay between steps */
		for (MD2LDelay = 1; MD2LDelay <= MD2MinDelay; MD2LDelay++) ;

		/* Update step count */
		MD2I = MD2I - 1L;

	}

	/* Turn com interrupts back on */
	if (MD2Interrupts == 0) outp(0x21, MD2IntReg);

	/* Set last direction according to HomeDir */
	if (MD2HomeDir[MD2Motor] == 0)
		MD2LastDir[MD2Motor] = 1;   /* HomeDir=rev, set lastdir to forward */
	else
		MD2LastDir[MD2Motor] = -1;  /* HomeDir=fwd, set lastdir to reverse */

	/* Set position to 0 */
	MD2Position[MD2Motor] = 0;

	/* Power off motor if desired */
	if (MD2Hold == 0) outp(MD2MtrAdr, 0xff);

}


void MD2Move(void)
{

	/*---------------------------------------------------------------------
	'name:      MD2Move
	'desc:      This procedure is used to move motors.
	'           set MD2Motor to 1,2,3,4,5 or 6 for single motor moves or
	'           to 12, 34 or 56 for dual motor moves.
	'           when moving 2 motors, both motors start and stop at the
	'           same time resulting in straight line motion also known
	'           as linear interpolation.
	'           speed and ramping of the motor with the longest
	'           travel will be used.
	'           soft limit values will prevent overtravel.
	'           the home switch is ignored.
	'           a keypress will stop motion.
	'           if slope=0 then the motor will move at MD2MaxSpeed
	'usage:     Set motor #, motor parameters and call.
	'inputs:    Motor # (1-6,12,34 or 56) and motor parameters.
	'outputs:   Current position, MD2Status.
	'---------------------------------------------------------------------*/

	/* Local variables */
	int MD2MtrAdr;              /* Motor port address */
	int MD2Pats;                /* Calculated step pattern */
	int MD2IntReg;              /* Interrupt register */
	int MD2DirSwapL;            /* Long motor swap dir. -1=yes, 1=no */
	int MD2DirSwapS;            /* Long motor swap dir. -1=yes, 1=no */

	/* Ramping variables */
	long int MD2DelayTmp;       /* Temporary delay loop counter */
	long int MD2DelayCur;       /* Current speed delay count */
	long int MD2DelayFlag;      /* Flag indicating delay adjustment */
	long int MD2DelayEach;      /* Delay adjustment each step */
	long int MD2DelayChange;    /* Amount of change in delay */
	long int MD2DelayChange2;   /* Delaychange * 2 */
	long int MD2Decel;          /* Step count to begin deceleration mode */
	long int MD2DelaySlope;     /* Slope */
	long int MD2DelaySlope2;    /* Slope * 2 */
	int MD2Mode;                /* Speed mode. 1=decel,0=constant,-1=accel */
	long int MD2MinDelay;       /* Minimum speed delay count */
	long int MD2MaxDelay;       /* Maximum speed delay count */
	double MD2Way;               /* Temp for velocity interolation */
	int MD2I;                   /* Array element counter */

	/* Linear interpolation variables */
	long int MD2LiError;        /* Error value for linear interpolation */
	int MD2MotorL;              /* Motor # traveling the longest */
	int MD2MotorS;              /* Motor # traveling the shortest */
	int MD2DirL;                /* Long motor direction. -1=rev,1=fwd */
	int MD2DirS;                /* Short motor direction */
	int MD2MaskL;               /* Long motor pattern mask */
	int MD2MaskS;               /* Short motor pattern mask */
	long int MD2StepsL;         /* # of steps for long motor */
	long int MD2StepsS;         /* # of steps for short motor */
	long int MD2StepsL2;        /* MD2StepsL * 2 */
	long int MD2StepsS2;        /* MD2StepsS * 2 */
	long int MD2SwapL;          /* Swap temporary variable */

	/* Check for setup */
	if (MD2MtrAdr12 != 0x3bc) {
		MD2Status = 'S';
		return;
	}

	/* Ignore some checks if (in calibration mode */
	if (MD2CalMode == 0) {

		/* Check for calibration */
		if (! MD2Calibrated) {
			MD2Status = 'C';
			return;
		}

		/* Check for port availability */
		MD2Status = 'P';
		if ((MD2Motor == 1) || (MD2Motor == 2) || (MD2Motor == 12))
			if (MD2Available12 == 0) return;
		if ((MD2Motor == 3) || (MD2Motor == 4) || (MD2Motor == 34))
			if (MD2Available34 == 0) return;
		if ((MD2Motor == 5) || (MD2Motor == 6) || (MD2Motor == 56))
			if (MD2Available56 == 0) return;

		/* Check for enabled MD-2 */
		MD2Status = 'E';
		if ((MD2Motor == 1) || (MD2Motor == 2) || (MD2Motor == 12))
			if (MD2Enabled12 == 0) return;
		if ((MD2Motor == 3) || (MD2Motor == 4) || (MD2Motor == 34))
			if (MD2Enabled34 == 0) return;
		if ((MD2Motor == 5) || (MD2Motor == 6) || (MD2Motor == 56))
			if (MD2Enabled56 == 0) return;

	}

	/* Check for valid motor # */
	MD2Status = 'B';
	if ((MD2Motor == 1) || (MD2Motor == 2) || (MD2Motor == 3)) MD2Status = 'O';
	if ((MD2Motor == 4) || (MD2Motor == 5) || (MD2Motor == 6)) MD2Status = 'O';
	if ((MD2Motor == 12) || (MD2Motor == 34) || (MD2Motor == 56)) MD2Status = 'O';
	if (MD2Status != 'O') return;

	/* Check for valid step type */
	MD2StepType = (char)toupper(MD2StepType);
	if ((MD2StepType != 'H') && (MD2StepType != 'D') && (MD2StepType != 'S')) {
		MD2Status = 'B';
		return;
	}

	/* Check for valid move type */
	MD2MoveType = (char)toupper(MD2MoveType);
	if ((MD2MoveType != 'A') && (MD2MoveType != 'R')) {
		MD2Status = 'B';
		return;
	}
                                                                    
	/* Set port addresses and motor numbers */
	if (MD2Motor == 1) {
		MD2MtrAdr = MD2MtrAdr12;
		MD2MotorL = 1;
		MD2MotorS = 0;
	}
	if (MD2Motor == 2) {
		MD2MtrAdr = MD2MtrAdr12;
		MD2MotorL = 2;
		MD2MotorS = 0;
	}
	if (MD2Motor == 3) {
		MD2MtrAdr = MD2MtrAdr34;
		MD2MotorL = 3;
		MD2MotorS = 0;
	}
	if (MD2Motor == 4) {
		MD2MtrAdr = MD2MtrAdr34;
		MD2MotorL = 4;
		MD2MotorS = 0;
	}
	if (MD2Motor == 5) {
		MD2MtrAdr = MD2MtrAdr56;
		MD2MotorL = 5;
		MD2MotorS = 0;
	}
	if (MD2Motor == 6) {
		MD2MtrAdr = MD2MtrAdr56;
		MD2MotorL = 6;
		MD2MotorS = 0;
	}
	if (MD2Motor == 12) {
		MD2MtrAdr = MD2MtrAdr12;
		MD2MotorL = 1;
		MD2MotorS = 2;
	}
	if (MD2Motor == 34) {
		MD2MtrAdr = MD2MtrAdr34;
		MD2MotorL = 3;
		MD2MotorS = 4;
	}
	if (MD2Motor == 56) {
		MD2MtrAdr = MD2MtrAdr56;
		MD2MotorL = 5;
		MD2MotorS = 6;
	}

	/* Set step patterns */
	if (MD2StepType == 'H') {       /* Half step mode */
		MD2StpPat[0] = 0x66; MD2StpPat[1] = 0x77;
		MD2StpPat[2] = 0x33; MD2StpPat[3] = 0xbb;
		MD2StpPat[4] = 0x99; MD2StpPat[5] = 0xdd;
		MD2StpPat[6] = 0xcc; MD2StpPat[7] = 0xee;
	}
	if (MD2StepType == 'S') {       /* Single phase full step mode */
		MD2StpPat[0] = 0x77; MD2StpPat[1] = 0xbb;
		MD2StpPat[2] = 0xdd; MD2StpPat[3] = 0xee;
		MD2StpPat[4] = 0x77; MD2StpPat[5] = 0xbb;
		MD2StpPat[6] = 0xdd; MD2StpPat[7] = 0xee;
	}
	if (MD2StepType == 'D') {       /* Double phase full step mode */
		MD2StpPat[0] = 0x66; MD2StpPat[1] = 0x33;
		MD2StpPat[2] = 0x99; MD2StpPat[3] = 0xcc;
		MD2StpPat[4] = 0x66; MD2StpPat[5] = 0x33;
		MD2StpPat[6] = 0x99; MD2StpPat[7] = 0xcc;
	}
                                
	/* Fix variables for 0 */
	MD2Target[0] = 0;               /* Motor 0 is no motor, no steps, etc */
	MD2Position[0] = 0;
	MD2Units[0] = 1;
  
	/* Set step count and direction for relative moves */
	if (MD2MoveType == 'R') {       /* Relative move? */
		/* If (limits exceeded) quit */
		if ((MD2Target[MD2MotorL] + MD2Position[MD2MotorL]) > MD2LimitF[MD2MotorL]) MD2Status = 'L';
		if ((MD2Target[MD2MotorL] + MD2Position[MD2MotorL]) < MD2LimitR[MD2MotorL]) MD2Status = 'L';
		if ((MD2Target[MD2MotorS] + MD2Position[MD2MotorS]) > MD2LimitF[MD2MotorS]) MD2Status = 'L';
		if ((MD2Target[MD2MotorS] + MD2Position[MD2MotorS]) < MD2LimitR[MD2MotorS]) MD2Status = 'L';
		if (MD2Status == 'L') return;
		/* Set the # of steps */
		MD2StepsL = (long int)((MD2Target[MD2MotorL] * MD2Units[MD2MotorL]));
		MD2StepsS = (long int)((MD2Target[MD2MotorS] * MD2Units[MD2MotorS]));
		/* Set the direction */
		if (MD2StepsL < 0) MD2DirL = -1; else MD2DirL = 1;
		if (MD2StepsS < 0) MD2DirS = -1; else MD2DirS = 1;
		/* Make step counts positive */
		MD2StepsL = labs(MD2StepsL);
		MD2StepsS = labs(MD2StepsS);
		/* Set completed positions and round */
		MD2Position[MD2MotorL] = MD2Position[MD2MotorL] + ((MD2StepsL * (1.0 / MD2Units[MD2MotorL])) * MD2DirL);
		MD2Position[MD2MotorL] = (long int)((MD2Position[MD2MotorL] * MD2Units[MD2MotorL]) + .5);
		MD2Position[MD2MotorL] = MD2Position[MD2MotorL] / MD2Units[MD2MotorL];

		MD2Position[MD2MotorS] = MD2Position[MD2MotorS] + ((MD2StepsS * (1.0 / MD2Units[MD2MotorS])) * MD2DirS);
		MD2Position[MD2MotorS] = (long int)((MD2Position[MD2MotorS] * MD2Units[MD2MotorS]) + .5);
		MD2Position[MD2MotorS] = MD2Position[MD2MotorS] / MD2Units[MD2MotorS];
	}
  
	/* Set step count and direction for absolute moves */
	if (MD2MoveType == 'A') {       /* Absolute move? */
		/* If (limits exceeded) quit */
		if (MD2Target[MD2MotorL] > MD2LimitF[MD2MotorL]) MD2Status = 'L';
		if (MD2Target[MD2MotorL] < MD2LimitR[MD2MotorL]) MD2Status = 'L';
		if (MD2Target[MD2MotorS] > MD2LimitF[MD2MotorS]) MD2Status = 'L';
		if (MD2Target[MD2MotorS] < MD2LimitR[MD2MotorS]) MD2Status = 'L';
		if (MD2Status == 'L') return;
		/* Set the # of steps */
		MD2StepsL = (long int)(((MD2Target[MD2MotorL] - MD2Position[MD2MotorL]) * MD2Units[MD2MotorL]));
		MD2StepsS = (long int)(((MD2Target[MD2MotorS] - MD2Position[MD2MotorS]) * MD2Units[MD2MotorS]));
		/* Set the direction */
		if (MD2StepsL < 0) MD2DirL = -1; else MD2DirL = 1;
		if (MD2StepsS < 0) MD2DirS = -1; else MD2DirS = 1;
		/* Make step counts positive */
		MD2StepsL = labs(MD2StepsL);
		MD2StepsS = labs(MD2StepsS);
		/* Set completed positions and round */
		MD2Position[MD2MotorL] = MD2Target[MD2MotorL];
		MD2Position[MD2MotorL] = (long int)((MD2Position[MD2MotorL] * MD2Units[MD2MotorL]) + .5);
		MD2Position[MD2MotorL] = MD2Position[MD2MotorL] / MD2Units[MD2MotorL];

		MD2Position[MD2MotorS] = MD2Target[MD2MotorS];
		MD2Position[MD2MotorS] = (long int)((MD2Position[MD2MotorS] * MD2Units[MD2MotorS]) + .5);
		MD2Position[MD2MotorS] = MD2Position[MD2MotorS] / MD2Units[MD2MotorS];
	}

	/* Swap directions if needed */
	if (MD2HomeDir[MD2MotorL] == -1) MD2DirL = MD2DirL * -1;
	if (MD2HomeDir[MD2MotorS] == -1) MD2DirS = MD2DirS * -1;

	/* Backlash compensation, adjust step counts */
	if (MD2DirL != MD2LastDir[MD2MotorL]) {
		MD2StepsL = MD2StepsL + (long int)((MD2Backlash[MD2MotorL] * MD2Units[MD2MotorL]));
		MD2LastDir[MD2MotorL] = MD2DirL;
	}
	if (MD2DirS != MD2LastDir[MD2MotorS]) {
		MD2StepsS = MD2StepsS + (long int)((MD2Backlash[MD2MotorS] * MD2Units[MD2MotorS]));
		MD2LastDir[MD2MotorS] = MD2DirS;
	}

	/* Swap parameters depending on longest/shortest step count */
	if (MD2StepsL < MD2StepsS) {
		MD2SwapL = MD2StepsL; MD2StepsL = MD2StepsS; MD2StepsS = MD2SwapL;
		MD2SwapL = (long int)MD2MotorL; MD2MotorL = MD2MotorS; MD2MotorS = (int)MD2SwapL;
		MD2SwapL = (long int)MD2DirL; MD2DirL = MD2DirS; MD2DirS = (int)MD2SwapL;
	}
   
	/* Initialize error accumulator and precalculate vars for interpolation*/
	MD2StepsL2 = MD2StepsL + MD2StepsL;
	MD2StepsS2 = MD2StepsS + MD2StepsS;
	MD2LiError = MD2StepsS2 - MD2StepsL;
                                                
	/* Set up pattern masks */
	if ((MD2MotorL == 1) || (MD2MotorL == 3) || (MD2MotorL == 5)) {
		MD2MaskL = 0xf;
		MD2MaskS = 0xf0;
	}
	else {
		MD2MaskL = 0xf0;
		MD2MaskS = 0xf;
	}

	/* Setup ramping---------------- */

	/* If (in calibration mode set ramping to maxspeed otherwise */
	/* Set ramping parameters normally */
	if (MD2CalMode) {
		MD2Mode = 0;            /* constant speed.*/
		MD2Decel = 0L;          /* never begin deceleration.*/
		MD2DelayCur = (long int)MD2MaxSpeed[MD2MotorL];
	}
	else {
		/* normal ramping. */
		/* find mindelay and maxdelay by searching velocity array and */
		/* interpolating between elements */
		/* setup delay count adjusments to be linearize ramp while stepping */
        
		/* Convert minspeed velocity to delay counts */
		for (MD2I = 2; MD2I <= 23; MD2I++) {    /*search for correct array elements.*/
			if ((MD2MinSpeed[MD2MotorL] * MD2Units[MD2MotorL]) >= MD2Velocity[MD2I]) break;
		}
		/* Interpolate between elements */
		MD2Way = ((MD2MinSpeed[MD2MotorL] * MD2Units[MD2MotorL]) - MD2Velocity[MD2I - 1]) / (MD2Velocity[MD2I] - MD2Velocity[MD2I - 1]);
		MD2MinDelay = (long int)((((MD2Delay[MD2I] - MD2Delay[MD2I - 1]) * MD2Way) + MD2Delay[MD2I - 1]) + .5);

		/* Convert maxspeed velocity to delay counts */
		for (MD2I = 2; MD2I <= 23; MD2I++) {    /*search for correct array elements.*/
			if ((MD2MaxSpeed[MD2MotorL] * MD2Units[MD2MotorL]) >= MD2Velocity[MD2I]) break;
		}
		/* Interpolate between elements */
		MD2Way = ((MD2MaxSpeed[MD2MotorL] * MD2Units[MD2MotorL]) - MD2Velocity[MD2I - 1]) / (MD2Velocity[MD2I] - MD2Velocity[MD2I - 1]);
		MD2MaxDelay = (long int)((((MD2Delay[MD2I] - MD2Delay[MD2I - 1]) * MD2Way) + MD2Delay[MD2I - 1]) + .5);

		/*set up ramping parameters.*/
		if ((MD2Slope[MD2MotorL] <= 0) || (MD2MinDelay <= MD2MaxDelay)) {
			/* Constant speed mode---------------- */

			/* Constant speed mode */
			MD2Mode = 2;

			/* Current speed */
			MD2DelayCur = MD2MaxDelay;
		}
		else {
			/* Ramping move----------------- */

			/* Acceleration mode */
			MD2Mode = -1;

			/* Current starting speed */
			MD2DelayCur = MD2MinDelay;

			/* Set step count to begin deceleration */
			if (MD2StepsL < (long int)((MD2Slope[MD2MotorL] * MD2Units[MD2MotorL]) * 2)) {
				/* Move too short to reach max speed */
				MD2Decel = (long int)((MD2StepsL / 2L) + .5);
			}
			else {
				/* Move long enough to reach max speed */
				MD2Decel = (long int)(MD2Slope[MD2MotorL] * MD2Units[MD2MotorL]);
			}
             
			/* Set slopes */
			MD2DelaySlope = (long int)(MD2Slope[MD2MotorL] * MD2Units[MD2MotorL]);
			MD2DelaySlope2 = MD2DelaySlope + MD2DelaySlope;

			/* Set amount of change in delay count */
			MD2DelayChange = MD2MinDelay - MD2MaxDelay;

			/* Different calculations for slope <=1 and >1 */
			if (MD2DelayChange > MD2DelaySlope) {
				/* Slope > 1 */
				/* Set delay adjustment that occurs every step */
				MD2DelayEach = (long int)(MD2DelayChange / MD2DelaySlope);
				/* Reset delaychange according to previous calc */
				MD2DelayChange = MD2DelayChange - (MD2DelaySlope * MD2DelayEach);
			}
			else {
				/* Slope <= 1 */
				/* Don't adjust every step */
				MD2DelayEach = 0;
			}
                  
			/* Precalculate this */
			MD2DelayChange2 = MD2DelayChange + MD2DelayChange;
             
			/* Preset adjustment flag */
			MD2DelayFlag = MD2DelayChange2 - MD2DelaySlope;
             
		}
	}

	/* Turn off com interrupts? */
	if (MD2Interrupts == 0) {
		MD2IntReg = inp(0x21);
		outp (0x21, MD2IntReg | 0x98);
	}

	/* Initialize status */
	MD2Status = 'O';
  
	/* Move loop */
	while (MD2StepsL != 0) {
		/* Move long motor */
        
		/* Point to the next step pattern */
		MD2PatPtr[MD2MotorL] = (MD2PatPtr[MD2MotorL] + MD2DirL) & 0x7;

		/* Get step pattern and mask off unneeded bits */
		MD2Pats = MD2StpPat[MD2PatPtr[MD2MotorL]] & MD2MaskL;

		/* Move short motor if needed */
		if (MD2LiError >= 0) {
			/* Point to the next step pattern */
			MD2PatPtr[MD2MotorS] = (MD2PatPtr[MD2MotorS] + MD2DirS) & 0x7;

			/* Or in short motor pattern */
			MD2Pats = MD2Pats | (MD2StpPat[MD2PatPtr[MD2MotorS]] & MD2MaskS);

			/* Update error for linear interpolation */
			MD2LiError = MD2LiError - MD2StepsL2;
		}
		else {
			/* Don't move short */
			MD2Pats = MD2Pats | (inp(MD2MtrAdr) & MD2MaskS);
		}
             
		/* Update error for linear interpolation */
		MD2LiError = MD2LiError + MD2StepsS2;
        
		/* Output the step pattern to move the motor */
		outp(MD2MtrAdr, MD2Pats);

		/* Delay between steps*/

		for (MD2DelayTmp = 1; MD2DelayTmp <= MD2DelayCur; MD2DelayTmp++) ;

		/* Decrement step count */
		MD2StepsL = MD2StepsL - 1L;
                            
		/* Calculate new speed */
        
		if (MD2Mode == 1) {     /* Deceleration */

			/* Begin constant if (reached min speed */
			if (MD2DelayCur >= MD2MinDelay) MD2Mode = 2;
             
			/* Decrease current speed */
			MD2DelayCur = MD2DelayCur + MD2DelayEach;
			if (MD2DelayFlag >= 0) {
				MD2DelayCur = MD2DelayCur + 1L;
				MD2DelayFlag = MD2DelayFlag - MD2DelaySlope2;
			}
			MD2DelayFlag = MD2DelayFlag + MD2DelayChange2;
		}
       
		if (MD2Mode == 0) {     /* Constant deceleration */

			/* Begin deceleration based on steps remaining */
			if (MD2StepsL <= MD2Decel) MD2Mode = 1;
		}

		if (MD2Mode == -1) {    /* Acceleration */

			/* Increase current speed */
			MD2DelayCur = MD2DelayCur - MD2DelayEach;
			if (MD2DelayFlag >= 0) {
				MD2DelayCur = MD2DelayCur - 1L;
				MD2DelayFlag = MD2DelayFlag - MD2DelaySlope2;
			}
			MD2DelayFlag = MD2DelayFlag + MD2DelayChange2;

			/* Begin deceleration based on steps remaining */
			if (MD2StepsL <= MD2Decel) MD2Mode = 1;
            
			/* Begin constant speed if (reached max speed */
			if (MD2DelayCur <= MD2MaxDelay) MD2Mode = 0;
		}
	}

	/* Turn com interrupts back on? */
	if (MD2Interrupts == 0) outp (0x21, MD2IntReg);

	/* Power off motor if desired */
	if (MD2Hold == 0) {
		outp (MD2MtrAdr, 0xff);
		/* Delay to allow step to take */
		for (MD2DelayTmp = 1; MD2DelayTmp <= MD2MinDelay; MD2DelayTmp++) ;
	}

	/* Keypress? */
	if (kbhit()) {
		getch();                /* Clear keyboard buffer */
		MD2Status = 'K';
	}

}

