'---------------------------------------------------------------------
'NAME:      MD2QI1S.BAS
'DESC:      MD-2 LEVEL 1 MOTION CONTROL SUBROUTINES FOR Q-BASIC.
'USAGE:     USE WHEN CREATING A CUSTOM MOTION CONTROL PROGRAM
'           THAT DOESN'T REQUIRE COMPLEX FEATURES SUCH AS RAMPING, ETC.
'           MOTOR PARAMETERS ARE PASSED VIA GLOBAL VARIABLES.
'           ALL VARIABLES AND SUBS BEGIN WITH MD2 TO AVOID CONFLICTS.
'BY:        COPYRIGHT (C) 1993 ARRICK ROBOTICS, ROGER ARRICK.
'DATE:      11/6/93
'EDIT:      12
'
'SUBROUTINES ---------------------------------------------------------
'
'    NAME           DESCRIPTION
'    ----           -----------
'    MD2SETUP       Used at the beginning of a program to initialize
'                   motor parameters to their default values.
'                   Use this subroutine before any other.
'    MD2ON          Turns on an MD-2 system.
'    MD2OFF         Turns off an MD-2 system.
'    MD2HOME        Moves a motor to the Home position by watching the
'                   home switch.  Current position is set to zero.
'    MD2MOVE        Moves a motor the number of steps in MD2TARGET(M)
'                   parameter.
'
'MOTOR PARAMETERS AND VARIABLES --------------------------------------
'
'    NAME            TYPE    DESCRIPTION
'    ----            ----    -----------
'
'    MD2HOLD         INTEGER -1=Leaves the motor energized after a move
'                            to cause the motor to hold its position.
'                            0 causes the motor to turn off after a move
'                            which conserves power and reduces heat.
'    MD2MOTOR        INTEGER The selected motor to act upon. 1,2,3,4,5 or 6.
'    MD2SPEED(M)     INTEGER Delay count between steps. Determines the
'                            motor's.  0=fast, 32766=slow.
'                            Actual motor speed depends on the computer.
'                            M is the motor number 1,2,3,4,5 or 6.
'    MD2POSITION(M)  LONG    Current motor position for each motor in steps
'                            relative to home. Negative = reverse from home
'                            and positive = forward from home.
'                            M is the motor number 1,2,3,4,5 or 6.
'    MD2STATUS       STRING  Completion status. O=motion completed OK,
'                            K=a keypress stopped the motion, B=Bad parameter.
'    MD2TARGET(M)    LONG    The number of steps to move. Positive #'s are
'                            forward, negative are reverse.
'                            M is the motor number 1,2,3,4,5 or 6.
'
'VARIABLES NEEDED BY SUBROUTINES BUT NOT BY PROGRAMMER ---------------
'
'    NAME            TYPE    DESCRIPTION
'    ----            ----    -----------
'
'    MD2DELAY        INTEGER Used for delay counter.
'    MD2TIMER        SINGLE  Used for timer.
'    MD2STEPS        LONG    Used for step counter.
'    MD2DIR          INTEGER Used as direction storage.
'    MD2PMASK        INTEGER Bit mask for selected motor.
'    MD2OMASK        INTEGER Bit mask for other motor.
'    MD2SMASK        INTEGER Bit mask for switch.
'    MD2MTRADR       INTEGER Selected Port address.
'    MD2MTRADR12     INTEGER Port address for motors 1 & 2.
'    MD2MTRADR34     INTEGER Port address for motors 3 & 4.
'    MD2MTRADR56     INTEGER Port address for motors 5 & 6.
'    MD2STPPAT(7)    INTEGER Step pattern array.
'    MD2PATPTR(6)    INTEGER Pattern pointer for each motor.
'    MD2PATTERN      INTEGER Selected step pattern.
'---------------------------------------------------------------------

'MOVE THESE STATEMENTS TO THE BEGINNING OF A PROGRAM.
DIM MD2DELAY AS INTEGER         'DELAY COUNTER.
DIM MD2TIMER AS SINGLE          'TIMER.
DIM MD2STEPS AS LONG            'STEP COUNTER.
DIM MD2DIR AS INTEGER           'DIRECTION.
DIM MD2PMASK AS INTEGER         'SELECTED MOTOR PATTERN BIT MASK.
DIM MD2OMASK AS INTEGER         'OTHER MOTOR PATTERN BIT MASK.
DIM MD2SMASK AS INTEGER         'SWITCH BIT MASK.
DIM MD2MTRADR AS INTEGER        'Selected Port address.
DIM MD2MTRADR12 AS INTEGER      'PORT FOR MOTORS 1 & 2.
DIM MD2MTRADR34 AS INTEGER      'PORT FOR MOTORS 3 & 4.
DIM MD2MTRADR56 AS INTEGER      'PORT FOR MOTORS 5 & 6.
DIM MD2STPPAT(7) AS INTEGER     'STEP PATTERN ARRAY.
DIM MD2PATPTR(6) AS INTEGER     'STEP PATTERN POINTERS.
DIM MD2PATTERN AS INTEGER       'SELECTED STEP PATTERN.
DIM MD2STATUS AS STRING         'COMPLETION STATUS.
DIM MD2HOLD AS INTEGER          'HOLD MOTORS WHEN DONE.
DIM MD2POSITION(6) AS LONG      'MOTOR POSITIONS.
DIM MD2MOTOR AS INTEGER         'SELECTED MOTOR NUMBER.
DIM MD2SPEED(6) AS INTEGER      'MOTOR SPEED.
DIM MD2TARGET(6) AS LONG        'TARGET DISTANCE AND DIRECTION.


MD2SETUP:

	'---------------------------------------------------------------------
	'NAME:      MD2SETUP
	'DESC:      THE MD2SETUP PROCEDURE SETS DEFAULT MD-2 SYSTEM
	'           PARAMETERS SUCH AS MOTOR SPEED, CURRENT POSITION, ETC.
	'USAGE:     USE AT THE BEGINNING OF A MOTION CONTROL PROGRAM.
	'           MUST BE USED -BEFORE- ANY OTHER MOTION CONTROL SUBROUTINE!
	'INPUTS:    NONE.
	'OUTPUTS:   DEFAULT MOTOR PARAMETERS.
	'---------------------------------------------------------------------

	'SETUP PORT ADDRESSES.
	MD2MTRADR12 = &H3BC
	MD2MTRADR34 = &H378
	MD2MTRADR56 = &H278

	'SETUP MOTOR PARAMETER DEFAULTS.
	FOR MD2MOTOR = 1 TO 6
		MD2POSITION(MD2MOTOR) = 0       'POSITIONS = 0.
		MD2PATPTR(MD2MOTOR) = 0         'STEP PATTERN POINTERS.
		MD2TARGET(MD2MOTOR) = 0         'TARGET POSITION,STEPS.
		MD2SPEED(MD2MOTOR) = 5000       'SPEED.
	NEXT MD2MOTOR
	MD2MOTOR = 1                        'MOTOR NUMBER.
	MD2STATUS = "O"                     'STATUS=OK.
	MD2HOLD = 0                         'DON'T HOLD MOTORS.
   
	'SET HALF STEP PHASE PATTERNS.
	MD2STPPAT(0) = &H66
	MD2STPPAT(1) = &H77
	MD2STPPAT(2) = &H33
	MD2STPPAT(3) = &HBB
	MD2STPPAT(4) = &H99
	MD2STPPAT(5) = &HDD
	MD2STPPAT(6) = &HCC
	MD2STPPAT(7) = &HEE
   
	RETURN


MD2ON:
   
	'---------------------------------------------------------------------
	'NAME:      MD2ON
	'DESC:      THE MD2ON PROCEDURE INITIALIZES A PARALLEL PRINTER PORT
	'           AND TURNS ON AN MD-2.
	'USAGE:     USE AT THE BEGINNING OF A MOTION CONTROL PROGRAM BUT
	'           AFTER THE MD2SETUP SUBROUTINE.
	'INPUTS:    MOTOR # DETERMINES PORT.
	'OUTPUTS:   NONE.
	'---------------------------------------------------------------------
			   
	IF MD2MOTOR = 1 OR MD2MOTOR = 2 THEN
		OUT MD2MTRADR12, &HFF
		OUT MD2MTRADR12 + 2, &H5
	END IF
  
	IF MD2MOTOR = 3 OR MD2MOTOR = 4 THEN
		OUT MD2MTRADR34, &HFF
		OUT MD2MTRADR34 + 2, &H5
	END IF
  
	IF MD2MOTOR = 5 OR MD2MOTOR = 6 THEN
		OUT MD2MTRADR56, &HFF
		OUT MD2MTRADR56 + 2, &H5
	END IF

	RETURN


MD2OFF:

	'---------------------------------------------------------------------
	'NAME:      MD2OFF
	'DESC:      THE MD2OFF PROCEDURE RETURNS A PARALLEL PRINTER PORT
	'           REFERENCED BY THE MOTOR # TO ITS PREVIOUS STATE READY
	'           FOR USE WITH A PRINTER AND DISABLES THE MD-2.
	'USAGE:     USE AT THE END OF A MOTION CONTROL PROGRAM.
	'INPUTS:    MOTOR # DETERMINES PORT.
	'OUTPUTS:   NONE.
	'---------------------------------------------------------------------
  
	'-STROBE PIN HIGH, -ALF PIN HIGH, -INIT PIN LOW, -SELIN PIN LOW, IRQ OFF.
	IF MD2MOTOR = 1 OR MD2MOTOR = 2 THEN OUT MD2MTRADR12 + 2, &H4
	IF MD2MOTOR = 3 OR MD2MOTOR = 4 THEN OUT MD2MTRADR34 + 2, &H4
	IF MD2MOTOR = 5 OR MD2MOTOR = 6 THEN OUT MD2MTRADR56 + 2, &H4

	'DELAY FOR .2 SECONDS.
	MD2TIMER = TIMER: DO: LOOP UNTIL TIMER > MD2TIMER + .2
  
	'TURN -INIT PIN HIGH.
	IF MD2MOTOR = 1 OR MD2MOTOR = 2 THEN OUT MD2MTRADR12 + 2, &HC
	IF MD2MOTOR = 3 OR MD2MOTOR = 4 THEN OUT MD2MTRADR34 + 2, &HC
	IF MD2MOTOR = 5 OR MD2MOTOR = 6 THEN OUT MD2MTRADR56 + 2, &HC
															
	RETURN


MD2HOME:

	'---------------------------------------------------------------------
	'NAME:      MD2HOME
	'DESC:      THE MD2HOME PROCEDURE IS USED TO MOVE THE STEPPER MOTOR
	'           TO A KNOWN HOME POSITION.  ALL OTHER MOVES ARE RELATIVE
	'           TO THIS HOME (ZERO) POSITION.  THE SELECTED MOTOR IS
	'           MOVED REVERSE UNTIL THE SWITCH IS ACTIVATED, THEN FORWARD
	'           UNTIL DEACTIVATED. THE CURRENT POSITION IS THEN SET TO
	'           ZERO - THIS IS THE HOME POSITION.
	'           ONLY 1 MOTOR (1-6) CAN BE HOME'D AT A TIME.
	'USAGE:     SET THE DESIRED MOTOR #, MOTOR PARAMETERS AND CALL.
	'INPUTS:    MOTOR # AND MOTOR PARAMETERS.
	'OUTPUTS:   CURRENT POSITION SET TO ZERO, MD2STATUS.
	'---------------------------------------------------------------------
  
	'SET DEFAULT RETURN STATUS.
	MD2STATUS = "O"

	'IF BAD MOTOR # THEN BAIL OUT.
	IF MD2MOTOR < 1 OR MD2MOTOR > 6 THEN MD2STATUS = "B": RETURN

	'SET UP ADDRESS.
	IF MD2MOTOR = 1 OR MD2MOTOR = 2 THEN MD2MTRADR = MD2MTRADR12
	IF MD2MOTOR = 3 OR MD2MOTOR = 4 THEN MD2MTRADR = MD2MTRADR34
	IF MD2MOTOR = 5 OR MD2MOTOR = 6 THEN MD2MTRADR = MD2MTRADR56

	'SET UP PATTERN MASK, OTHER MOTOR'S MASK AND SWITCH MASK.
	IF MD2MOTOR = 1 OR MD2MOTOR = 3 OR MD2MOTOR = 5 THEN
		MD2PMASK = &HF
		MD2OMASK = &HF0
		MD2SMASK = &H20
	ELSE
		MD2PMASK = &HF0
		MD2OMASK = &HF
		MD2SMASK = &H10
	END IF

	'------------------------------------------
	'MOVE MOTOR REVERSE UNTIL SWITCH ACTIVATED.
	'------------------------------------------
   
	'LOOP UNTIL SWITCH IS ACTIVATED.
	DO UNTIL (INP(MD2MTRADR + 1) AND MD2SMASK) = 0

		'QUIT MOVING IF KEY PRESSED.
		IF INKEY$ <> "" THEN MD2STATUS = "K": EXIT DO

		'POINT TO THE NEXT STEP PATTERN.
		MD2PATPTR(MD2MOTOR) = (MD2PATPTR(MD2MOTOR) - 1) AND &H7

		'GET STEP PATTERN AND MASK OFF UNNEEDED BITS.
		MD2PATTERN = MD2STPPAT(MD2PATPTR(MD2MOTOR)) AND MD2PMASK

		'DON'T DISTURB OTHER MOTORS BITS.
		MD2PATTERN = MD2PATTERN OR (INP(MD2MTRADR) AND MD2OMASK)

		'OUTPUT THE STEP PATTERN TO MOVE THE MOTOR.
		OUT MD2MTRADR, MD2PATTERN

		'DELAY BETWEEN STEPS.
		FOR MD2DELAY = 1 TO MD2SPEED(MD2MOTOR): NEXT MD2DELAY
   
	LOOP

	'------------------------------------------
	'MOVE MOTOR FORWARD UNTIL SWITCH DEACTIVATED.
	'------------------------------------------
   
	'LOOP UNTIL SWITCH IS DEACTIVATED.
	DO UNTIL (INP(MD2MTRADR + 1) AND MD2SMASK) <> 0

		'QUIT MOVING IF KEY PRESSED.
		IF INKEY$ <> "" THEN MD2STATUS = "K": EXIT DO

		'POINT TO THE NEXT STEP PATTERN.
		MD2PATPTR(MD2MOTOR) = (MD2PATPTR(MD2MOTOR) + 1) AND &H7
	  
		'GET STEP PATTERN AND MASK OFF UNNEEDED BITS.
		MD2PATTERN = MD2STPPAT(MD2PATPTR(MD2MOTOR)) AND MD2PMASK

		'DON'T DISTURB OTHER MOTORS BITS.
		MD2PATTERN = MD2PATTERN OR (INP(MD2MTRADR) AND MD2OMASK)
	  
		'OUTPUT THE STEP PATTERN TO MOVE THE MOTOR.
		OUT MD2MTRADR, MD2PATTERN

		'DELAY BETWEEN STEPS.
		FOR MD2DELAY = 1 TO MD2SPEED(MD2MOTOR): NEXT MD2DELAY
   
	LOOP

	'SET POSITION TO 0.
	MD2POSITION(MD2MOTOR) = 0

	'POWER OFF MOTOR IF DESIRED.
	IF MD2HOLD = 0 THEN OUT MD2MTRADR, &HFF
   
	RETURN


MD2MOVE:

	'---------------------------------------------------------------------
	'NAME:      MD2MOVE
	'DESC:      THIS PROCEDURE IS USED TO MOVE A MOTOR.
	'           THE HOME SWITCH IS IGNORED.  ANY KEYPRESS WILL STOP MOTION.
	'USAGE:     SET MOTOR #, MOTOR PARAMETERS AND CALL.
	'INPUTS:    MOTOR # (1,2,3,4,5 OR 6) AND MOTOR PARAMETERS.
	'OUTPUTS:   CURRENT POSITION, MD2STATUS.
	'---------------------------------------------------------------------
   
	'SET DEFAULT RETURN STATUS.
	MD2STATUS = "O"

	'IF BAD MOTOR # THEN BAIL OUT.
	IF MD2MOTOR < 1 OR MD2MOTOR > 6 THEN MD2STATUS = "B": RETURN

	'SET UP ADDRESS.
	IF MD2MOTOR = 1 OR MD2MOTOR = 2 THEN MD2MTRADR = MD2MTRADR12
	IF MD2MOTOR = 3 OR MD2MOTOR = 4 THEN MD2MTRADR = MD2MTRADR34
	IF MD2MOTOR = 5 OR MD2MOTOR = 6 THEN MD2MTRADR = MD2MTRADR56

	'SET UP PATTERN MASK AND OTHER MOTOR'S MASK.
	IF MD2MOTOR = 1 OR MD2MOTOR = 3 OR MD2MOTOR = 5 THEN
		MD2PMASK = &HF
		MD2OMASK = &HF0
	ELSE
		MD2PMASK = &HF0
		MD2OMASK = &HF
	END IF

	'SET THE # OF STEPS.
	MD2STEPS = ABS(MD2TARGET(MD2MOTOR))

	'SET THE DIRECTION.
	IF MD2TARGET(MD2MOTOR) < 0 THEN MD2DIR = -1 ELSE MD2DIR = 1

	'SET THE FINAL MOTOR POSITION.
	MD2POSITION(MD2MOTOR) = MD2POSITION(MD2MOTOR) + MD2TARGET(MD2MOTOR)
  
	'MOVE LOOP.
	DO UNTIL MD2STEPS = 0

		'QUIT MOVING IF KEY PRESSED.
		IF INKEY$ <> "" THEN MD2STATUS = "K": EXIT DO

		'POINT TO THE NEXT STEP PATTERN.
		MD2PATPTR(MD2MOTOR) = (MD2PATPTR(MD2MOTOR) + MD2DIR) AND &H7

		'GET STEP PATTERN AND MASK OFF UNNEEDED BITS.
		MD2PATTERN = MD2STPPAT(MD2PATPTR(MD2MOTOR)) AND MD2PMASK

		'DON'T DISTURB OTHER MOTORS BITS.
		MD2PATTERN = MD2PATTERN OR (INP(MD2MTRADR) AND MD2OMASK)

		'OUTPUT THE STEP PATTERN TO MOVE THE MOTOR.
		OUT MD2MTRADR, MD2PATTERN

		'DELAY BETWEEN STEPS.
		FOR MD2DELAY = 1 TO MD2SPEED(MD2MOTOR): NEXT MD2DELAY
	 
		'DECREMENT STEP COUNT.
		MD2STEPS = MD2STEPS - 1
	  
	LOOP
  
	'UPDATE POSITION.
	MD2POSITION(MD2MOTOR) = MD2POSITION(MD2MOTOR) - (MD2STEPS * MD2DIR)
   
	'POWER OFF MOTOR IF DESIRED.
	IF MD2HOLD = 0 THEN OUT MD2MTRADR, &HFF
   
	RETURN
