'---------------------------------------------------------------------
'NAME:      MD2VW1S.BAS
'DESC:      MD-2 LEVEL 1 MOTION CONTROL SUBROUTINES FOR VISUAL-BASIC/WIN.
'USAGE:     USE WHEN CREATING A CUSTOM MOTION CONTROL PROGRAM
'           THAT DOESN'T REQUIRE COMPLEX FEATURES SUCH AS RAMPING, ETC.
'           MOTOR PARAMETERS ARE PASSED VIA GLOBAL VARIABLES.
'           ALL VARIABLES AND SUBS BEGIN WITH MD2 TO AVOID CONFLICTS.
'BY:        COPYRIGHT (C) 1993 ARRICK ROBOTICS, ROGER ARRICK.
'DATE:      12/20/93
'EDIT:      3
'
'SUBROUTINES ---------------------------------------------------------
'
'    NAME           DESCRIPTION
'    ----           -----------
'    MD2SETUP       Used at the beginning of a program to initialize
'                   motor parameters to their default values.
'                   Use this subroutine before any other.
'    MD2ON          Turns on an MD-2 system.
'    MD2OFF         Turns off an MD-2 system.
'    MD2HOME        Moves a motor to the Home position by watching the
'                   home switch.  Current position is set to zero.
'    MD2MOVE        Moves a motor the number of steps in MD2TARGET(M)
'                   parameter.
'
'MOTOR PARAMETERS AND VARIABLES --------------------------------------
'
'    NAME            TYPE    DESCRIPTION
'    ----            ----    -----------
'
'    MD2HOLD         INTEGER -1=Leaves the motor energized after a move
'                            to cause the motor to hold its position.
'                            0 causes the motor to turn off after a move
'                            which conserves power and reduces heat.
'    MD2MOTOR        INTEGER The selected motor to act upon. 1,2,3,4,5 or 6.
'    MD2SPEED(M)     INTEGER Delay count between steps. Determines the
'                            motor's.  0=fast, 32766=slow.
'                            Actual motor speed depends on the computer.
'                            M is the motor number 1,2,3,4,5 or 6.
'    MD2POSITION(M)  LONG    Current motor position for each motor in steps
'                            relative to home. Negative = reverse from home
'                            and positive = forward from home.
'                            M is the motor number 1,2,3,4,5 or 6.
'    MD2STATUS       STRING  Completion status. O=motion completed OK,
'                            K=a keypress stopped the motion, B=Bad parameter.
'    MD2TARGET(M)    LONG    The number of steps to move. Positive #'s are
'                            forward, negative are reverse.
'                            M is the motor number 1,2,3,4,5 or 6.
'
'VARIABLES NEEDED BY SUBROUTINES BUT NOT BY PROGRAMMER ---------------
'
'    NAME            TYPE    DESCRIPTION
'    ----            ----    -----------
'
'    MD2DELAY        INTEGER Used for delay counter.
'    MD2TIMER        SINGLE  Used for timer.
'    MD2STEPS        LONG    Used for step counter.
'    MD2DIR          INTEGER Used as direction storage.
'    MD2PMASK        INTEGER Bit mask for selected motor.
'    MD2OMASK        INTEGER Bit mask for other motor.
'    MD2SMASK        INTEGER Bit mask for switch.
'    MD2MTRADR       INTEGER Selected port address.
'    MD2MTRADR12     INTEGER Port address for motors 1 & 2.
'    MD2MTRADR34     INTEGER Port address for motors 3 & 4.
'    MD2MTRADR56     INTEGER Port address for motors 5 & 6.
'    MD2STPPAT(7)    INTEGER Step pattern array.
'    MD2PATPTR(6)    INTEGER Pattern pointer for each motor.
'    MD2PATTERN      INTEGER Selected step pattern.
'---------------------------------------------------------------------

Option Explicit

'THIS DLL FROM MICROHELP PROVIDES INP AND OUT FUNCTIONS TO VISUAL-BASIC.
'MAKE SURE MHELP.VBX CAN BE FOUND BY THIS PROGRAM.
Declare Function INP% Lib "MHELP.VBX" (ByVal PortNum%)
Declare Sub OUT Lib "MHELP.VBX" (ByVal PortNum%, ByVal DataByte%)

'GLOBAL VARIABLES USED BY ALL MODULES.
Global MD2STATUS As String              'COMPLETION STATUS.
Global MD2HOLD As Integer               'HOLD MOTORS WHEN DONE.
Global MD2POSITION(6) As Long           'MOTOR POSITIONS.
Global MD2MOTOR As Integer              'SELECTED MOTOR NUMBER.
Global MD2SPEED(6) As Integer           'MOTOR SPEED.
Global MD2TARGET(6) As Long             'TARGET DISTANCE AND DIRECTION.

'THESE VARIABLES ARE ONLY USED BY SUBS IN THIS MODULE.
Dim MD2DELAY As Integer                 'DELAY COUNTER.
Dim MD2TIMER As Single                  'TIMER.
Dim MD2STEPS As Long                    'STEP COUNTER.
Dim MD2DIR As Integer                   'DIRECTION.
Dim MD2PMASK As Integer                 'SELECTED MOTOR PATTERN BIT MASK.
Dim MD2OMASK As Integer                 'OTHER MOTOR PATTERN BIT MASK.
Dim MD2SMASK As Integer                 'SWITCH BIT MASK.
Dim MD2MTRADR As Integer                'SELECTED PORT ADDRESS.
Dim MD2MTRADR12 As Integer              'PORT FOR MOTORS 1 & 2.
Dim MD2MTRADR34 As Integer              'PORT FOR MOTORS 3 & 4.
Dim MD2MTRADR56 As Integer              'PORT FOR MOTORS 5 & 6.
Dim MD2STPPAT(7) As Integer             'STEP PATTERN ARRAY.
Dim MD2PATPTR(6) As Integer             'STEP PATTERN POINTERS.
Dim MD2PATTERN As Integer               'SELECTED STEP PATTERN.

Sub MD2HOME ()

	'---------------------------------------------------------------------
	'NAME:      MD2HOME
	'DESC:      THE MD2HOME PROCEDURE IS USED TO MOVE THE STEPPER MOTOR
	'           TO A KNOWN HOME POSITION.  ALL OTHER MOVES ARE RELATIVE
	'           TO THIS HOME (ZERO) POSITION.  THE SELECTED MOTOR IS
	'           MOVED REVERSE UNTIL THE SWITCH IS ACTIVATED, THEN FORWARD
	'           UNTIL DEACTIVATED. THE CURRENT POSITION IS THEN SET TO
	'           ZERO - THIS IS THE HOME POSITION.
	'           ONLY 1 MOTOR (1-6) CAN BE HOME'D AT A TIME.
	'USAGE:     SET THE DESIRED MOTOR #, MOTOR PARAMETERS AND CALL.
	'INPUTS:    MOTOR # AND MOTOR PARAMETERS.
	'OUTPUTS:   CURRENT POSITION SET TO ZERO, MD2STATUS.
	'---------------------------------------------------------------------

	'SET DEFAULT RETURN STATUS.
	MD2STATUS = "O"

	'IF BAD MOTOR # THEN BAIL OUT.
	If MD2MOTOR < 1 Or MD2MOTOR > 6 Then MD2STATUS = "B": Return

	'SET UP ADDRESS.
	If MD2MOTOR = 1 Or MD2MOTOR = 2 Then MD2MTRADR = MD2MTRADR12
	If MD2MOTOR = 3 Or MD2MOTOR = 4 Then MD2MTRADR = MD2MTRADR34
	If MD2MOTOR = 5 Or MD2MOTOR = 6 Then MD2MTRADR = MD2MTRADR56

	'SET UP PATTERN MASK, OTHER MOTOR'S MASK AND SWITCH MASK.
	If MD2MOTOR = 1 Or MD2MOTOR = 3 Or MD2MOTOR = 5 Then
		MD2PMASK = &HF
		MD2OMASK = &HF0
		MD2SMASK = &H20
	Else
		MD2PMASK = &HF0
		MD2OMASK = &HF
		MD2SMASK = &H10
	End If

	'------------------------------------------
	'MOVE MOTOR REVERSE UNTIL SWITCH ACTIVATED.
	'------------------------------------------

	'LOOP UNTIL SWITCH IS ACTIVATED.
	Do Until (INP(MD2MTRADR + 1) And MD2SMASK) = 0

		'POINT TO THE NEXT STEP PATTERN.
		MD2PATPTR(MD2MOTOR) = (MD2PATPTR(MD2MOTOR) - 1) And &H7

		'GET STEP PATTERN AND MASK OFF UNNEEDED BITS.
		MD2PATTERN = MD2STPPAT(MD2PATPTR(MD2MOTOR)) And MD2PMASK

		'DON'T DISTURB OTHER MOTORS BITS.
		MD2PATTERN = MD2PATTERN Or (INP(MD2MTRADR) And MD2OMASK)

		'OUTPUT THE STEP PATTERN TO MOVE THE MOTOR.
		OUT MD2MTRADR, MD2PATTERN

		'DELAY BETWEEN STEPS.
		For MD2DELAY = 1 To MD2SPEED(MD2MOTOR): Next MD2DELAY

	Loop

	'------------------------------------------
	'MOVE MOTOR FORWARD UNTIL SWITCH DEACTIVATED.
	'------------------------------------------

	'LOOP UNTIL SWITCH IS DEACTIVATED.
	Do Until (INP(MD2MTRADR + 1) And MD2SMASK) <> 0

		'POINT TO THE NEXT STEP PATTERN.
		MD2PATPTR(MD2MOTOR) = (MD2PATPTR(MD2MOTOR) + 1) And &H7

		'GET STEP PATTERN AND MASK OFF UNNEEDED BITS.
		MD2PATTERN = MD2STPPAT(MD2PATPTR(MD2MOTOR)) And MD2PMASK

		'DON'T DISTURB OTHER MOTORS BITS.
		MD2PATTERN = MD2PATTERN Or (INP(MD2MTRADR) And MD2OMASK)

		'OUTPUT THE STEP PATTERN TO MOVE THE MOTOR.
		OUT MD2MTRADR, MD2PATTERN

		'DELAY BETWEEN STEPS.
		For MD2DELAY = 1 To MD2SPEED(MD2MOTOR): Next MD2DELAY

	Loop

	'SET POSITION TO 0.
	MD2POSITION(MD2MOTOR) = 0

	'POWER OFF MOTOR IF DESIRED.
	If MD2HOLD = 0 Then OUT MD2MTRADR, &HFF

End Sub

Sub MD2MOVE ()

	'---------------------------------------------------------------------
	'NAME:      MD2MOVE
	'DESC:      THIS PROCEDURE IS USED TO MOVE A MOTOR.
	'           THE HOME SWITCH IS IGNORED.
	'USAGE:     SET MOTOR #, MOTOR PARAMETERS AND CALL.
	'INPUTS:    MOTOR # (1,2,3,4,5 OR 6) AND MOTOR PARAMETERS.
	'OUTPUTS:   CURRENT POSITION, MD2STATUS.
	'---------------------------------------------------------------------

	'SET DEFAULT RETURN STATUS.
	MD2STATUS = "O"

	'IF BAD MOTOR # THEN BAIL OUT.
	If MD2MOTOR < 1 Or MD2MOTOR > 6 Then MD2STATUS = "B": Return

	'SET UP ADDRESS.
	If MD2MOTOR = 1 Or MD2MOTOR = 2 Then MD2MTRADR = MD2MTRADR12
	If MD2MOTOR = 3 Or MD2MOTOR = 4 Then MD2MTRADR = MD2MTRADR34
	If MD2MOTOR = 5 Or MD2MOTOR = 6 Then MD2MTRADR = MD2MTRADR56

	'SET UP PATTERN MASK AND OTHER MOTOR'S MASK.
	If MD2MOTOR = 1 Or MD2MOTOR = 3 Or MD2MOTOR = 5 Then
		MD2PMASK = &HF
		MD2OMASK = &HF0
	Else
		MD2PMASK = &HF0
		MD2OMASK = &HF
	End If

	'SET THE # OF STEPS.
	MD2STEPS = Abs(MD2TARGET(MD2MOTOR))

	'SET THE DIRECTION.
	If MD2TARGET(MD2MOTOR) < 0 Then MD2DIR = -1 Else MD2DIR = 1

	'SET THE FINAL MOTOR POSITION.
	MD2POSITION(MD2MOTOR) = MD2POSITION(MD2MOTOR) + MD2TARGET(MD2MOTOR)

	'MOVE LOOP.
	Do Until MD2STEPS = 0

		'POINT TO THE NEXT STEP PATTERN.
		MD2PATPTR(MD2MOTOR) = (MD2PATPTR(MD2MOTOR) + MD2DIR) And &H7

		'GET STEP PATTERN AND MASK OFF UNNEEDED BITS.
		MD2PATTERN = MD2STPPAT(MD2PATPTR(MD2MOTOR)) And MD2PMASK

		'DON'T DISTURB OTHER MOTORS BITS.
		MD2PATTERN = MD2PATTERN Or (INP(MD2MTRADR) And MD2OMASK)

		'OUTPUT THE STEP PATTERN TO MOVE THE MOTOR.
		OUT MD2MTRADR, MD2PATTERN

		'DELAY BETWEEN STEPS.
		For MD2DELAY = 1 To MD2SPEED(MD2MOTOR): Next MD2DELAY

		'DECREMENT STEP COUNT.
		MD2STEPS = MD2STEPS - 1

	Loop

	'UPDATE POSITION.
	MD2POSITION(MD2MOTOR) = MD2POSITION(MD2MOTOR) - (MD2STEPS * MD2DIR)

	'POWER OFF MOTOR IF DESIRED.
	If MD2HOLD = 0 Then OUT MD2MTRADR, &HFF

End Sub

Sub MD2OFF ()

	'---------------------------------------------------------------------
	'NAME:      MD2OFF
	'DESC:      THE MD2OFF PROCEDURE RETURNS A PARALLEL PRINTER PORT
	'           REFERENCED BY THE MOTOR # TO ITS PREVIOUS STATE READY
	'           FOR USE WITH A PRINTER AND DISABLES THE MD-2.
	'USAGE:     USE AT THE END OF A MOTION CONTROL PROGRAM.
	'INPUTS:    MOTOR # DETERMINES PORT.
	'OUTPUTS:   NONE.
	'---------------------------------------------------------------------

	'-STROBE PIN HIGH, -ALF PIN HIGH, -INIT PIN LOW, -SELIN PIN LOW, IRQ OFF.
	If MD2MOTOR = 1 Or MD2MOTOR = 2 Then OUT MD2MTRADR12 + 2, &H4
	If MD2MOTOR = 3 Or MD2MOTOR = 4 Then OUT MD2MTRADR34 + 2, &H4
	If MD2MOTOR = 5 Or MD2MOTOR = 6 Then OUT MD2MTRADR56 + 2, &H4

	'DELAY FOR .2 SECONDS.
	MD2TIMER = Timer: Do: Loop Until Timer > MD2TIMER + .2

	'TURN -INIT PIN HIGH.
	If MD2MOTOR = 1 Or MD2MOTOR = 2 Then OUT MD2MTRADR12 + 2, &HC
	If MD2MOTOR = 3 Or MD2MOTOR = 4 Then OUT MD2MTRADR34 + 2, &HC
	If MD2MOTOR = 5 Or MD2MOTOR = 6 Then OUT MD2MTRADR56 + 2, &HC

End Sub

Sub MD2ON ()

	'---------------------------------------------------------------------
	'NAME:      MD2ON
	'DESC:      THE MD2ON PROCEDURE INITIALIZES A PARALLEL PRINTER PORT
	'           AND TURNS ON AN MD-2.
	'USAGE:     USE AT THE BEGINNING OF A MOTION CONTROL PROGRAM BUT
	'           AFTER THE MD2SETUP SUBROUTINE.
	'INPUTS:    MOTOR # DETERMINES PORT.
	'OUTPUTS:   NONE.
	'---------------------------------------------------------------------

	If MD2MOTOR = 1 Or MD2MOTOR = 2 Then
		OUT MD2MTRADR12, &HFF
		OUT MD2MTRADR12 + 2, &H5
	End If

	If MD2MOTOR = 3 Or MD2MOTOR = 4 Then
		OUT MD2MTRADR34, &HFF
		OUT MD2MTRADR34 + 2, &H5
	End If

	If MD2MOTOR = 5 Or MD2MOTOR = 6 Then
		OUT MD2MTRADR56, &HFF
		OUT MD2MTRADR56 + 2, &H5
	End If

End Sub

Sub MD2SETUP ()

	'---------------------------------------------------------------------
	'NAME:      MD2SETUP
	'DESC:      THE MD2SETUP PROCEDURE SETS DEFAULT MD-2 SYSTEM
	'           PARAMETERS SUCH AS MOTOR SPEED, CURRENT POSITION, ETC.
	'USAGE:     USE AT THE BEGINNING OF A MOTION CONTROL PROGRAM.
	'           MUST BE USED -BEFORE- ANY OTHER MOTION CONTROL SUBROUTINE!
	'INPUTS:    NONE.
	'OUTPUTS:   DEFAULT MOTOR PARAMETERS.
	'---------------------------------------------------------------------

	'SETUP PORT ADDRESSES.
	MD2MTRADR12 = &H3BC
	MD2MTRADR34 = &H378
	MD2MTRADR56 = &H278

	'SETUP MOTOR PARAMETER DEFAULTS.
	For MD2MOTOR = 1 To 6
		MD2POSITION(MD2MOTOR) = 0       'POSITIONS = 0.
		MD2PATPTR(MD2MOTOR) = 0         'STEP PATTERN POINTERS.
		MD2TARGET(MD2MOTOR) = 10        'TARGET POSITION,STEPS.
		MD2SPEED(MD2MOTOR) = 5000       'SPEED.
	Next MD2MOTOR
	MD2MOTOR = 1                        'MOTOR NUMBER.
	MD2STATUS = "O"                     'STATUS=OK.
	MD2HOLD = 0                         'DON'T HOLD MOTORS.

	'SET HALF STEP PHASE PATTERNS.
	MD2STPPAT(0) = &H66
	MD2STPPAT(1) = &H77
	MD2STPPAT(2) = &H33
	MD2STPPAT(3) = &HBB
	MD2STPPAT(4) = &H99
	MD2STPPAT(5) = &HDD
	MD2STPPAT(6) = &HCC
	MD2STPPAT(7) = &HEE

End Sub

