'---------------------------------------------------------------------
'NAME:      MD2VW2S.BAS
'DESC:      MD-2 LEVEL 2 VISUAL-BASIC/WINDOWS MOTION CONTROL SUBROUTINES
'USAGE:     USE WHEN CREATING COMPLEX MOTION CONTROL PROGRAMS.
'           USE LEVEL 1 LIBRARY WHEN CREATING SIMPLE PROGRAMS.
'
'           FEATURES:
'               DUAL MOTOR MOVES.
'               RELATIVE/ABSOLUTE MOVES.
'               ADJUSTABLE ACCELERATION/DECELERATION.
'               2-AXIS LINEAR INTERPOLATION.
'               2-AXIS CIRCULAR INTERPOLATION.
'               ARC AND ELLIPSE MOVES.
'               2-AXIS GRID MOVES.
'               USER DEFINED UNITS.
'               BACKLASH COMPENSATION.
'               SOFT LIMITS.
'               SPEED GIVEN IN UNITS PER SECOND.
'               PARAMETER EDITOR WITH SAVE & LOAD.
'               MOTION SEQUENCE PROGRAM INTERPRETER.
'               INPUT/OUTPUT PORT CONTROL.
'
'           ALL MOTOR PARAMETERS ARE PASSED TO SUBROUTINES VIA
'           GLOBAL VARIABLES.  ALL VARIABLES AND SUBS BEGIN WITH
'           MD2 TO AVOID CONFLICTS.
'           SEE .TXT FILE FOR COMPLETE DESCRIPTION.
'BY:        COPYRIGHT (C) 1994 ARRICK ROBOTICS, ROGER ARRICK.
'DATE:      5/14/95
'EDIT:      21
'
'PUBLIC SUBROUTINES ---------------------------------------------------
'
'    NAME           DESCRIPTION
'    ----           -----------
'    MD2SETUP       Used at the beginning of a program to initialize
'                   motor parameters to their default values and checks
'                   for parallel port availability. Loads calibration file.
'                   Use this SUB before any other.
'    MD2CALIBRATE   Calibrates motor speeds with computer speeds and saves
'                   to disk.  Use this SUB once on your cpu before moving.
'    MD2ON          Turns on an MD-2 system. Use before any moves.
'    MD2OFF         Turns off an MD-2 system. Use at end of program.
'    MD2HOME        Moves a motor to the Home position by watching the
'                   home switch.  Current position is set to zero.
'    MD2MOVE        Moves a motor to the desired target position using
'                   the various motor parameters.  Also moves two motors
'                   together providing linear interpolation.
'    MD2CIRCLE      Performs 2-axis circular, arc and ellipise moves.
'    MD2GRID        Moves 2 motors using grid coordinates.
'    MD2OUTPUTS     Turns on and off output pins to control tools.
'    MD2INPUTS      Reads condition of input pins and home switches.
'    MD2STANDBYON   Puts MD-2 into standby mode which reduces motor current
'                   and heat while maintaining some holding torque.
'    MD2STANDBYOFF  Turns off standby mode giving full current to motors.
'    MD2PARLOAD     Loads motor parameters from a disk file.
'    MD2PARSAVE     Saves motor parameters to a disk file.
'    MD2SEQLOAD     Loads a sequence of motion commands from a disk file.
'    MD2SEQSAVE     Saves a sequence of motion commands to a disk file.
'    MD2SEQRUN      Runs a sequence of motion commands.
'
'
'PUBLIC MOTOR PARAMETERS AND VARIABLES ----------------------------------
'
'    (M) indicates motor # 1,2,3,4,5 or 6.
'
'    NAME            TYPE    DESCRIPTION
'    ----            ----    -----------
'
'    MD2BACKLASH(M)  SINGLE  Backlash compensation in units.  Adds this
'                            amount to every move that changes direction.
'                            Uses MD2LASTDIR(M).
'                            Value is in Units.
'    MD2ENABLED12    INTEGER Port 3BC for motors 1 & 2 has been enabled
'                            by the MD2ON sub. 0=not enabled, -1=available.
'    MD2ENABLED34    INTEGER Port 3BC for motors 3 & 4 has been enabled
'                            by the MD2ON sub. 0=not enabled, -1=available.
'    MD2ENABLED56    INTEGER Port 3BC for motors 5 & 6 has been enabled
'                            by the MD2ON sub. 0=not enabled, -1=available.
'    MD2HOLD         INTEGER -1=Leaves the motor energized after a move
'                            to cause the motor to hold its position.
'                            0 causes the motor to turn off after a move
'                            which conserves power and reduces heat.
'    MD2HOMEOFFSET(M)SINGLE  Distance from home switch to call home.
'                            Value is in units.
'    MD2HOMEDIR(M)   INTEGER Direction to home switch, 0=rev, -1=fwd.
'                            Reverses meaning of forward and reverse
'                            target directions.  All directions are reversed.
'    MD2INTERRUPTS   INTEGER 0=turns COM port interrupts off during moves
'                            which prevents mouse & modem interference.
'                            This can be important under WINDOWS.
'                            -1 leaves interrupts enabled.
'    MD2LIMITF(M)    SINGLE  Forward soft limit in units.  Prevents moves
'                            beyond this value in the forward direction.
'                            Value is in Units.
'    MD2LIMITR(M)    SINGLE  Reverse soft limit in units.  Prevents moves
'                            beyond this value in the reverse direction
'                            Value is in Units.
'    MD2MOTOR        INTEGER The selected motor to act upon. 1,2,3,4,5 or 6.
'                            Set to 12, 34, 56 for dual motor moves.
'    MD2MOTORNAME(M) STRING  User definable motor name.
'    MD2MAXSPEED(M)  SINGLE  Motor fastest speed after accelerating.
'                            In Units per Second.
'    MD2MINSPEED(M)  SINGLE  Motor speed at the beginning and end of ramping.
'                            In Units per Second.
'    MD2MOVETYPE     STRING  Tells the MD2MOVE sub what type of move.
'                            R=relative, A=absolute. Affects all motors.
'                            With relative moves, the MD2TARGET parameter
'                            tells how many steps to move from the current
'                            position. Negative numbers are reverse, positive
'                            numbers are forward.
'                            With absolute moves, the MD2TARGET parameter
'                            tells what position to move to.  The number of
'                            steps and direction are determined by the
'                            current position. Negative numbers are reverse
'                            from home and positive numbers are forward.
'                            negative then reverse, positive then forward.
'    MD2AVAILABLE12  INTEGER Port 3BC availability for motors 1 & 2 set by
'                            the MD2SETUP sub. 0=not available, -1=available.
'    MD2AVAILABLE34  INTEGER Port 378 availability for motors 3 & 4 set by
'                            the MD2SETUP sub. 0=not available, -1=available.
'    MD2AVAILABLE56  INTEGER Port 278 availability for motors 5 & 6 set by
'                            the MD2SETUP sub. 0=not available, -1=available.
'    MD2POSITION(M)  SINGLE  Current motor position for each motor (M=motor #)
'                            relative to home. Negative = reverse from home
'                            and positive = forward from home.
'                            Value is in units.
'    MD2SLOPE(M)     SINGLE  The slope of the ramp (acceleration/deceleration)
'                            in number Units.  A 2 would mean that the
'                            motor would accelerate from the MD2MINSPEED to
'                            the MD2MAXSPEED in 2 units.
'    MD2STATUS       STRING  Completion status.
'                            O=motion completed OK,
'                            K=a keypress stopped the motion,
'                            B=bad parameter,
'                            L=soft limit was exceeded.
'                            E=MD-2 not enabled.
'                            P=Port not available.
'                            F=file error.
'    MD2STEPTYPE     STRING  H=Half step, D=Double phase full step,
'                            S=Single phase full step.
'                            Affects all motors.
'    MD2TARGET(M)    SINGLE  The number of units to move in relative moves,
'                            the position to move to in absolute moves.
'                            Value is in units which is converted to steps.
'    MD2UNITS(M)     LONG    Converts user definable units to steps
'                            for MD2MOVE subroutine.
'                            This is number of steps in each unit.
'    MD2UNITNAME(M)  STRING  This is the name of the units such as
'                            steps, inches, degrees, etc.
'
'PUBLIC INPUT/OUTPUT PORT PARAMETERS----------------------------------
'
'    MD2OUTPUTCODE   INTEGER Tells the MD2OUTPUTS sub which MD-2, which
'                            output pin and on/off action.
'    MD2INPUT(CODE)  INTEGER Contains status of input pins and home switches
'                            after using MD2INPUTS sub.
'
'PUBLIC PARAMETER & SEQUENCE FILE VARIABLES---------------------------
'
'    MD2COMMAND      STRING  Single command to execute.
'    MD2SEQUENCE     STRING  Contains a motion sequence.
'    MD2SEQFILE      STRING  Sequence file name.
'    MD2PARFILE      STRING  Parameter file name.
'    MD2POINTER      STRING  Sequence file pointer.
'
'PUBLIC GRID MOVE PARAMETERS-------------------------------------------
'
'    MD2GRIDBEGINX   SINGLE  Grid X start position in units.
'    MD2GRIDBEGINY   SINGLE  Grid Y start position in units.
'    MD2GRIDSPACEX   SINGLE  Grid X spacing in units.
'    MD2GRIDSPACEY   SINGLE  Grid Y spacing in units.
'    MD2GRIDTARGETX  INTEGER Grid X target.
'    MD2GRIDTARGETY  INTEGER Grid Y target.
'
'PUBLIC CIRCLE/ARC MOVE PARAMETERS------------------------------------
'
'    MD2CIRCLERADIUSX   SINGLE  Circle X radius in units.
'    MD2CIRCLERADIUSY   SINGLE  Circle Y radius in units.
'    MD2CIRCLECENTERX   SINGLE  Circle X center in units.
'    MD2CIRCLECENTERY   SINGLE  Circle Y center in units.
'    MD2CIRCLESTART     INTEGER Circle start angle in degrees.
'    MD2CIRCLEARC       INTEGER Circle arc angle in degrees. -=CW,+=CCW.
'    MD2CIRCLECHORD     INTEGER Circle chord angle in degrees.
'
'PUBLIC MOTOR SPEED CALIBRATION PARAMETERS--------------------------------
'
'    MD2VELOCITY(24)    SINGLE  Fastest possible motor speed in steps/sec.
'    MD2DELAY(24)       SINGLE  Slowest possible motor speed in steps/sec.
'    MD2CALFILE         STRING  Calibration file name.
'    MD2CALIBRATED      INTEGER Calibrated status, 0=false, -1=true.
'
'---------------------------------------------------------------------

Option Explicit

'THIS DLL FROM MICROHELP PROVIDES INP AND OUT FUNCTIONS TO VISUAL-BASIC.
'MAKE SURE MHELP.VBX CAN BE FOUND BY THIS PROGRAM.
Declare Function INP% Lib "MHELP.VBX" (ByVal PortNum%)
Declare Sub OUT Lib "MHELP.VBX" (ByVal PortNum%, ByVal DataByte%)

'GLOBAL MOTION CONTROL VARIABLES USED BY ALL MODULES.
Global MD2AVAILABLE12 As Integer            '-1=3BC PORT AVAILABLE, 0=NOT.
Global MD2AVAILABLE34 As Integer            '-1=378 PORT AVAILABLE, 0=NOT.
Global MD2AVAILABLE56 As Integer            '-1=278 PORT AVAILABLE, 0=NOT.
Global MD2BACKLASH(6) As Single             'BACKLASH COMPENSATION IN UNITS.
Global MD2ENABLED12 As Integer              '-1=3BC MD-2 ENABLED, 0=NOT.
Global MD2ENABLED34 As Integer              '-1=378 MD-2 ENABLED, 0=NOT.
Global MD2ENABLED56 As Integer              '-1=278 MD-2 ENABLED, 0=NOT.
Global MD2HOLD As Integer                   'HOLD MOTOR AFTER MOVES 0=NO,-1=YES.
Global MD2HOMEOFFSET(6) As Single           'HOME OFFSET (OVER TRAVEL).
Global MD2HOMEDIR(6) As Integer             'DIRECTION OF HOME, 1=FWD, -1=REV.
Global MD2INPUT(35) As Integer              'MD-2 INPUTS, 0=OFF,-1=ACTIVE.
Global MD2INTERRUPTS As Integer             'INTERRUPTS DURING MOVES, 0=OFF,-1=ON.
Global MD2LIMITF(6) As Single               'FORWARD SOFT LIMIT IN UNITS.
Global MD2LIMITR(6) As Single               'REVERSE SOFT LIMIT IN UNITS.
Global MD2MAXSPEED(6) As Single             'FAST TOP RAMP SPEED.
Global MD2MINSPEED(6) As Single             'SLOW STARTING RAMP SPEED.
Global MD2MOTOR As Integer                  'SELECTED MOTOR,1-6,12,34 OR 56.
Global MD2MOVETYPE As String                'MOVE TYPE R=RELATIVE,A=ABSOLUTE.
Global MD2MOTORNAME(6) As String            'USER DEFINABLE MOTOR NAME.
Global MD2OUTPUTCODE As Integer             'OUTPUT PIN CONTROL ACTION CODE.
Global MD2POSITION(6) As Single             'MOTOR POSITION IN UNITS.
Global MD2SLOPE(6) As Single                '# OF UNITS TO ACHEIVE ACCEL/DECEL.
Global MD2STATUS As String                  'RETURN STATUS, O=OK,B=BAD PARM,ETC.
Global MD2STEPTYPE As String                'H=HALF,D=DOUBLE FULL,S=SINGLE FULL.
Global MD2TARGET(6) As Single               'TARGET POSITION OR DISTANCE IN UNITS.
Global MD2UNITS(6) As Long                  '# OF STEPS PER UNIT.
Global MD2UNITNAME(6) As String             'NAME OF UNIT.

'GLOBAL SEQUENCE FILE AND PARAMETER FILE VARIABLES USED BY ALL MODULES.
Global MD2COMMAND As String                 'SINGLE COMMAND USED BY MD2SEQRUN.
Global MD2SEQUENCE As String                'COMMAND SEQUENCE USED BY MD2SEQRUN.
Global MD2PARFILE As String                 'MOTOR PARAMETER FILE NAME.
Global MD2SEQFILE As String                 'SEQUENCE FILE NAME.
Global MD2POINTER As Integer                'SEQUENCE POINTER USED BY MD2SEQRUN.

'GLOBAL GRID PARAMETERS USED BY ALL MODULES.
Global MD2GRIDBEGINX As Single              'GRID X BEGIN POSITION IN UNITS.
Global MD2GRIDBEGINY As Single              'GRID Y BEGIN POSITION IN UNITS.
Global MD2GRIDSPACEX As Single              'GRID X SPACING IN UNITS.
Global MD2GRIDSPACEY As Single              'GRID Y SPACING IN UNITS.
Global MD2GRIDTARGETX As Integer            'GRID X TARGET.
Global MD2GRIDTARGETY As Integer            'GRID Y TARGET.

'GLOBAL CIRCLE PARAMETERS USED BY MD2CIRCLE.
Global MD2CIRCLERADIUSX As Single           'CIRCLE X RADIUS IN UNITS.
Global MD2CIRCLERADIUSY As Single           'CIRCLE Y RADIUS IN UNITS.
Global MD2CIRCLECENTERX As Single           'CENTER X COORDINATE IN UNITS.
Global MD2CIRCLECENTERY As Single           'CENTER Y COORDINATE IN UNITS.
Global MD2CIRCLESTART As Integer            'START ANGLE IN DEGREES.
Global MD2CIRCLEARC As Integer              'ARC ANGLE IN DEGREES. 360=CIRCLE.
Global MD2CIRCLECHORD As Integer            'CHORD ANGLE IN DEGREES.

'GLOBAL CALIBRATION PARAMETERS.
Global MD2CALFILE As String                 'CALIBRATION FILE NAME.
Global MD2DELAY(24) As Long                 'DELAY/VELOCITY CONVERSION ARRAY.
Global MD2VELOCITY(24) As Single            'DELAY/VELOCITY CONVERSION ARRAY(SPS).
Global MD2CALIBRATED As Integer             'CALIBRATED STATUS, -1=TRUE, 0=FALSE.

'PRIVATE VARIABLES USED ONLY BY SUBS IN THIS MODULE.
Dim MD2MTRADR12 As Integer                  'PORT FOR MOTORS 1 & 2.
Dim MD2MTRADR34 As Integer                  'PORT FOR MOTORS 3 & 4.
Dim MD2MTRADR56 As Integer                  'PORT FOR MOTORS 5 & 6.
Dim MD2STPPAT(7) As Integer                 'STEP PATTERN ARRAY.
Dim MD2PATPTR(6) As Integer                 'STEP PATTERN POINTERS.
Dim MD2LASTDIR(6) As Integer                'LAST DIRECTION -1=REV,1=FWD.
Dim MD2CALMODE As Integer                   'CALIBRATION MODE. 0=OFF,-1=ON.

Sub MD2CALIBRATE ()

	'---------------------------------------------------------------------
	'NAME:      MD2CALIBRATE
	'DESC:      THIS PROCEDURE CALIBRATES MOTOR SPEEDS ON THIS PARTICULAR
	'           COMPUTER.  THE CALIBRATION INFORMATION IS STORED IN THE
	'           ARRAYS MD2VELOCITY(24) AND MD2DELAY(24) THEN SAVED TO DISK
	'           IN THE FILE NAME STORED IN MD2CALFILE.  CALIBRATION
	'           INFORMATION IS READ BY THE MD2SETUP SUB AT THE BEGINNING
	'           OF EACH MOTION PROGRAM AND IS USED WHEN MOVING MOTORS.
	'           THIS IS TRANSPARENT TO THE USER AND PROGRAMMER.
	'           WITHOUT CALIBRATION, MOTOR SPEEDS WOULD VARY DEPENDING ON
	'           COMPUTER SPEED AND OTHER FACTORS.
	'           IT IS ONLY NECESSARY TO CALIBRATE ON A PARTICULAR COMPUTER
	'           ONCE.  MD2SETUP MUST BE USED FIRST.
	'           MD2VELOCITY(1)=FASTEST POSSIBLE SPEED IN STEPS PER SECOND.
	'           MD2VELOCITY(24)=SLOWEST POSSIBLE SPEED IN STEPS PER SECOND
	'USAGE:     SIMPLY CALL AND WAIT SEVERAL MINUITES.
	'INPUTS:    MD2CALFILE = CALIBRATION FILE NAME USUALLY "MD2.CAL"
	'OUTPUTS:   FILE CONTAINING CALIBRATION PARAMETERS,ARRAYS,MD2STATUS.
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	Dim MD2ELEMENT As Integer       'ELEMENT COUNTER.
	Dim MD2STIME As Single          'BEGINNING TIME.
	Dim MD2ETIME As Single          'ENDING TIME.
	Dim MD2DELAYLOOP As Long        'DELAY.
	Dim MD2FILENUM As Integer       'FREE FILE NUMBER.
	Dim MD2TMPMOTOR As Integer      'TEMP VARIABLE STORAGE.
	Dim MD2TMPMOVETYPE As String
	Dim MD2TMPSLOPE As Single
	Dim MD2TMPBACKLASH As Single
	Dim MD2TMPUNITS As Long
	Dim MD2TMPLIMITF As Single
	Dim MD2TMPLIMITR As Single
	Dim MD2TMPTARGET As Single
	Dim MD2TMPPOSITION As Single
					  
	'CHECK FOR SETUP.
	If MD2MTRADR12 <> &H3BC Then MD2STATUS = "S": Exit Sub
	
	'SAVE PARAMETERS.
	MD2TMPMOTOR = MD2MOTOR
	MD2TMPMOVETYPE = MD2MOVETYPE
	MD2TMPSLOPE = MD2SLOPE(1)
	MD2TMPBACKLASH = MD2BACKLASH(1)
	MD2TMPUNITS = MD2UNITS(1)
	MD2TMPLIMITF = MD2LIMITF(1)
	MD2TMPLIMITR = MD2LIMITR(1)
	MD2TMPTARGET = MD2TARGET(1)
	MD2TMPPOSITION = MD2POSITION(1)
   
	'INITIALIZE MOTOR PARAMETERS AND VARIABLES.
	MD2MOTOR = 1
	MD2MOVETYPE = "R"
	MD2SLOPE(1) = 0
	MD2BACKLASH(1) = 0
	MD2UNITS(1) = 1
	MD2LIMITF(1) = 2000000000
	MD2LIMITR(1) = -2000000000
	MD2DELAYLOOP = 1
	MD2CALMODE = -1
	MD2STATUS = "O"
	MD2OFF

	'FOR EACH DELAY (1,2,4,8,ETC), FIND VELOCITY AND PLACE IN ARRAY.
	For MD2ELEMENT = 1 To 24
	   
		'PRINT ".";

		'INITIALIZE VARIABLES.
		MD2TARGET(1) = 1
		MD2MAXSPEED(1) = MD2DELAYLOOP
		MD2DELAY(MD2ELEMENT) = MD2DELAYLOOP

MD2CALOOP:
	   
		'TIME THE MOVE TO DETERMINE SPEED.
		MD2STIME = Timer: Do: Loop Until MD2STIME <> Timer: MD2STIME = Timer
		MD2MOVE
		MD2ETIME = Timer

		'IF MOVE ERROR THEN BAIL OUT.
		If MD2STATUS <> "O" Then Exit For

		'TRY AGAIN IF NOT 10 SECONDS OR MORE OF TEST.
		If (MD2ETIME - MD2STIME) <= 10 Then
			MD2TARGET(1) = MD2TARGET(1) + MD2TARGET(1)
			GoTo MD2CALOOP
		End If

		'PUT VELOCITY IN ARRAY.
		MD2VELOCITY(MD2ELEMENT) = MD2TARGET(1) / (MD2ETIME - MD2STIME)
	   
		'QUIT IF LAST MEASUREMENT WAS SLOWER THAN .01SPS.
		If MD2VELOCITY(MD2ELEMENT) <= .01 Then
			'FILL REMAINING ELEMENTS WITH LAST ENTRY.
			Do
				MD2DELAY(MD2ELEMENT) = MD2DELAYLOOP
				MD2VELOCITY(MD2ELEMENT) = MD2TARGET(1) / (MD2ETIME - MD2STIME)
				MD2ELEMENT = MD2ELEMENT + 1
			Loop Until MD2ELEMENT = 25
			Exit For
		End If
	   
		'DO NEXT DELAY.
		MD2DELAYLOOP = MD2DELAYLOOP + MD2DELAYLOOP

	Next MD2ELEMENT
   
	MD2CALMODE = 0
  
	If MD2STATUS = "O" Then
		'SAVE CALIBRATION PARAMETERS IF NO FAILURE.
		On Error GoTo MD2CALIBRATEERROR
		MD2FILENUM = FreeFile
		Open MD2CALFILE For Output As MD2FILENUM
		For MD2ELEMENT = 1 To 24
			If MD2STATUS = "F" Then Exit For
			Print #MD2FILENUM, MD2DELAY(MD2ELEMENT); ","; MD2VELOCITY(MD2ELEMENT)
		Next MD2ELEMENT
		Close MD2FILENUM
		On Error GoTo 0
		MD2CALIBRATED = -1
	Else
		MD2CALIBRATED = 0
	End If

	'RESTORE PARAMETERS.
	MD2MOVETYPE = MD2TMPMOVETYPE
	MD2MOTOR = MD2TMPMOTOR
	MD2SLOPE(1) = MD2TMPSLOPE
	MD2BACKLASH(1) = MD2TMPBACKLASH
	MD2UNITS(1) = MD2TMPUNITS
	MD2LIMITF(1) = MD2TMPLIMITF
	MD2LIMITR(1) = MD2TMPLIMITR
	MD2TARGET(1) = MD2TMPTARGET
	MD2POSITION(1) = MD2TMPPOSITION
	Exit Sub

'FILE ERROR HANDLER.
MD2CALIBRATEERROR:
	MD2STATUS = "F"
	Resume Next
	
End Sub

Sub MD2CIRCLE ()

	'---------------------------------------------------------------------
	'NAME:      MD2CIRCLE
	'DESC:      THE MD2CIRCLE WILL MOVE 2 MOTORS WHICH ARE ATTACHED TO AN
	'           XY POSITIONING TABLE IN A CIRCULAR, ARC OR ELLIPSE PATTERN.
	'           THE 2 MOTORS MUST BE 1 & 2, 3 & 4 OR 5 & 6.
	'           THE USER CAN SET THE CHORD ANGLE IN DEGREES WHICH DETERMINES
	'           THE RESOLUTION OF THE CIRCLE.  A CHORD ANGLE OF 45 WILL
	'           PRODUCE AN OCTAGON.  A CHORD ANGLE OF 1 DEGREE WILL PRODUCE
	'           A CIRCLE WITH 360 SIDES.
	'           VARIOUS ARC ANGLES CAN ALSO BE GIVEN WHICH WILL PRODUCE
	'           PARTS OF A CIRCLE. AN ARC ANGLE OF 90 WILL PRODUCE AN ARC
	'           OF 90 DEGREES.  360 WILL PRODUCE A COMPLETE CIRCLE.
	'           X & Y RADIUS VALUES ARE PROVIDED TO PRODUCE AN ELLIPSE.
	'           IF BOTH X & Y RADIUS VALUES ARE THE SAME, A CIRCLE WILL
	'           RESULT.
	'           POSITIVE ARC ANGLES WILL PRODUCE COUNTER-CLOCKWISE ROTATION,
	'           NEGATIVE VALUES CLOCKWISE.
	'           MOVES ARE PERFORMED AT THE MINIMUM SPEED WITH NO RAMPING.
	'           TO MAKE A CIRCLE OR ARC AT AN ABSOLUTE POSITION, SET THE
	'           CENTERX,Y PARAMETERS TO THE ABSOLUTE CENTER POSITIONS AND
	'           SET MD2MOVETYPE TO "A".
	'           TO MAKE A CIRCLE OR ARC RELATIVE TO THE CURRENT POSITION,
	'           SET CENTERX,Y TO OFFSETS FROM CURRENT MOTOR POSITIONS AND
	'           SET MD2MOVETYPE TO "R".
	'           RAMPING IS DISABLED.  MOTORS WILL MOVE AT MINIMUM SPEED.
	'
	'              PARAMETER            DESCRIPTION
	'           ----------------    -----------------------
	'           MD2CIRCLERADIUSX    X-AXIS RADIUS IN UNITS.
	'           MD2CIRCLERADIUSY    Y-AXIS RADIUS IN UNITS.
	'           MD2CIRCLECENTERX    X-AXIS CENTER IN UNITS, REL OR ABS.
	'           MD2CIRCLECENTERY    Y-AXIS CENTER IN UNITS, REL OR ABS.
	'           MD2CIRCLESTART      STARTING ANGLE IN DEGREES.
	'           MD2CIRCLEARC        ARC ANGLE IN DEGREES. 360=CIRCLE.
	'                               +=CCW, -=CW ROTATION DIRECTION.
	'           MD2CIRCLECHORD      CHORD ANGLE IN DEGREES.
	'           MD2MOTOR            12, 34 OR 45.
	'           MD2MOVETYPE         "A"=ABSOLUTE, "R"=RELATIVE.
	'
	'USAGE:     SET CIRCLE PARAMETERS AND MOTOR PARAMETERS THEN CALL.
	'INPUTS:    ALL MOTOR PARAMETERS, CIRCLE PARAMETERS.
	'OUTPUTS:   MD2STATUS, MD2TARGET(M), MD2POSITION(M)
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	Dim MD2ENDANGLE As Integer      'END ANGLE.
	Dim MD2ANGLE As Integer         'CURRENT ANGLE.
	Dim MD2ANGLER As Single         'CURRENT ANGLE IN RADIANS.
	Dim MD2MOTOR1 As Integer        '1ST MOTOR #.
	Dim MD2MOTOR2 As Integer        '2ND MOTOR #.
	Dim MD2TMPMOVETYPE As String    'SAVE MOVETYPE
	Dim MD2TMPSLOPE1 As Single      'SAVE SLOPE 1.
	Dim MD2TMPSLOPE2 As Single      'SAVE SLOPE 2.
	Dim MD2TMPSPEED1 As Single      'SAVE MAX SPEED 1.
	Dim MD2TMPSPEED2 As Single      'SAVE MAX SPEED 2.
	Dim MD2TMPHOLD As Integer       'SAVE HOLD.
	Dim MD2CNTRX As Single          'CENTER X.
	Dim MD2CNTRY As Single          'CENTER Y.

	'CHECK FOR BAD PARAMETERS.
	MD2STATUS = "B"
	If MD2MOTOR <> 12 And MD2MOTOR <> 34 And MD2MOTOR <> 56 Then Exit Sub
	If MD2CIRCLEARC = 0 Or Abs(MD2CIRCLEARC) > 360 Then Exit Sub
	If MD2CIRCLECHORD <= 0 Or MD2CIRCLECHORD > 180 Then Exit Sub
	If MD2CIRCLESTART > 360 Or MD2CIRCLESTART < 0 Then Exit Sub
	MD2MOVETYPE = UCase$(MD2MOVETYPE)
	If MD2MOVETYPE <> "A" And MD2MOVETYPE <> "R" Then Exit Sub
	MD2STATUS = "O"

	'INITIALIZE VARIABLES.
	If MD2MOTOR = 12 Then MD2MOTOR1 = 1: MD2MOTOR2 = 2
	If MD2MOTOR = 34 Then MD2MOTOR1 = 3: MD2MOTOR2 = 4
	If MD2MOTOR = 56 Then MD2MOTOR1 = 5: MD2MOTOR2 = 6
   
	'SAVE PARAMETERS.
	MD2TMPMOVETYPE = MD2MOVETYPE
	MD2TMPSLOPE1 = MD2SLOPE(MD2MOTOR1)
	MD2TMPSLOPE2 = MD2SLOPE(MD2MOTOR2)
	MD2TMPSPEED1 = MD2MAXSPEED(MD2MOTOR1)
	MD2TMPSPEED2 = MD2MAXSPEED(MD2MOTOR2)
	MD2TMPHOLD = MD2HOLD

	'SETUP MOTOR PARAMETERS.
	If MD2MOVETYPE = "R" Then
		'RELATIVE CENTER POSITIONS.
		MD2CNTRX = MD2POSITION(MD2MOTOR1) + MD2CIRCLECENTERX
		MD2CNTRY = MD2POSITION(MD2MOTOR2) + MD2CIRCLECENTERY
	Else
		'ABSOLUTE CENTER POSITIONS.
		MD2CNTRX = MD2CIRCLECENTERX
		MD2CNTRY = MD2CIRCLECENTERY
	End If
	MD2MOVETYPE = "A"
	MD2SLOPE(MD2MOTOR1) = 0
	MD2SLOPE(MD2MOTOR2) = 0
	MD2MAXSPEED(MD2MOTOR1) = MD2MINSPEED(MD2MOTOR1)
	MD2MAXSPEED(MD2MOTOR2) = MD2MINSPEED(MD2MOTOR2)
	MD2HOLD = -1
   
	'SET ANGLE AND END ANGLE.
	MD2ENDANGLE = MD2CIRCLESTART + MD2CIRCLEARC
	MD2ANGLE = MD2CIRCLESTART
   
	'PLOT BASED ON DIRECTION.
	If MD2CIRCLEARC > 0 Then
	   
		'CCW PLOT LOOP.
		Do
			GoSub MD2CIRCLEDO
			If MD2STATUS <> "O" Then Exit Do
			MD2ANGLE = MD2ANGLE + MD2CIRCLECHORD
		Loop Until MD2ANGLE >= MD2ENDANGLE
   
	Else
	   
		'CW PLOT LOOP.
		Do
			GoSub MD2CIRCLEDO
			If MD2STATUS <> "O" Then Exit Do
			MD2ANGLE = MD2ANGLE - MD2CIRCLECHORD
		Loop Until MD2ANGLE <= MD2ENDANGLE
   
	End If

	'PLOT LAST ANGLE UNLESS PREVIOUS ERROR.
	If MD2STATUS = "O" Then
		MD2ANGLE = MD2ENDANGLE
		GoSub MD2CIRCLEDO
	End If

	'RESTORE PARAMETERS.
	MD2MOVETYPE = MD2TMPMOVETYPE
	MD2SLOPE(MD2MOTOR1) = MD2TMPSLOPE1
	MD2SLOPE(MD2MOTOR2) = MD2TMPSLOPE2
	MD2MAXSPEED(MD2MOTOR1) = MD2TMPSPEED1
	MD2MAXSPEED(MD2MOTOR2) = MD2TMPSPEED2
	MD2HOLD = MD2TMPHOLD

	'POWER OFF MOTOR IF DESIRED.
	If MD2HOLD = 0 Then
		If MD2MOTOR = 12 Then OUT MD2MTRADR12, &HFF
		If MD2MOTOR = 34 Then OUT MD2MTRADR34, &HFF
		If MD2MOTOR = 56 Then OUT MD2MTRADR56, &HFF
	End If

	Exit Sub

'MOVE TO ANGLE DESIRED ANGLE.
'
MD2CIRCLEDO:
	'CONVERT TO RADIANS.
	MD2ANGLER = MD2ANGLE * (3.1416 / 180)

	'CALCULATE X & Y AND MOVE.
	MD2TARGET(MD2MOTOR1) = (MD2CIRCLERADIUSX * Cos(MD2ANGLER)) + MD2CNTRX
	MD2TARGET(MD2MOTOR2) = (MD2CIRCLERADIUSY * Sin(MD2ANGLER)) + MD2CNTRY

	'MOVE THE MOTORS.
	MD2MOVE
   
	'DEBUG CODE.
	'LINE -((MD2TARGET(MD2MOTOR1) * 30) + 160, ABS((MD2TARGET(MD2MOTOR2) * 30) - 199) - 99)
	'PRESET ((MD2TARGET(MD2MOTOR1) * 30) + 160, ABS((MD2TARGET(MD2MOTOR2) * 30) - 199) - 99)
	'PRINT MD2TARGET(MD2MOTOR1), MD2TARGET(MD2MOTOR2): INPUT "", A$
	'DO: LOOP WHILE INKEY$ = ""

	Return

End Sub

Sub MD2GRID ()

	'---------------------------------------------------------------------
	'NAME:      MD2GRID
	'DESC:      THE MD2GRID PROCEDURE IS USED TO MOVE MOTORS TO A LOCATION
	'           OF A GRID.  A GRID IS AN X Y MATRIX OF LOCATIONS LIKE
	'           A CHECKER BOARD.  THIS ALLOWS A PROGRAMMER TO MOVE TO
	'           A ROW & COLUMN OF A GRID INSTEAD OF MOVING BY PASSING
	'           MD2GRID WORKS WITH ABSOLUTE TARGET POSITIONS ONLY.
	'           THE SPECIFIC TARGET LOCATION IN STEPS OR INCHES, ETC.
	'           THE FOLLOWING PARAMETERS DEFINE THE GRID.
	'
	'               PARAMETER       DESCRIPTION
	'               ---------       -----------
	'               MD2GRIDBEGINX   X BEGIN ABSOLUTE POSITION IN UNITS
	'               MD2GRIDBEGINY   Y BEGIN ABSOLUTE POSITION IN UNITS
	'               MD2GRIDSPACEX   X SPACING IN UNITS.
	'               MD2GRIDSPACEY   Y SPACING IN UNITS.
	'               MD2GRIDTARGETX  X TARGET, ABSOLUTE.
	'               MD2GRIDTARGETY  Y TARGET, ABSOLUTE.
	'
	'USAGE:     SET GRID PARAMETERS TO DEFINE THE GRID, SET MD2MOTOR
	'           TO A VALID MOTOR PAIR (12,34 OR 56), SET MOTOR PARAMETERS
	'           THEN CALL.
	'INPUTS:    ALL GRID AND MOTOR PARAMETERS.
	'OUTPUTS:   MD2STATUS,MD2TARGET(M),MD2POSITION(M).
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	Dim MD2X As Integer        'X MOTOR #.
	Dim MD2Y As Integer        'Y MOTOR #.
	Dim MD2MT As String        'MOVETYPE STORAGE.

	'CHECK FOR BAD PARAMETERS.
	MD2STATUS = "B"
	If MD2MOTOR <> 12 And MD2MOTOR <> 34 And MD2MOTOR <> 56 Then Exit Sub
	MD2STATUS = "O"

	'SET MOTOR #.
	If MD2MOTOR = 12 Then MD2X = 1: MD2Y = 2
	If MD2MOTOR = 34 Then MD2X = 3: MD2Y = 4
	If MD2MOTOR = 56 Then MD2X = 5: MD2Y = 6

	'SET TARGET POSITIONS.
	MD2TARGET(MD2X) = MD2GRIDBEGINX + ((MD2GRIDTARGETX - 1) * MD2GRIDSPACEX)
	MD2TARGET(MD2Y) = MD2GRIDBEGINY + ((MD2GRIDTARGETY - 1) * MD2GRIDSPACEY)

	'GRID MOVE.
	MD2MT = MD2MOVETYPE     'SAVE MOVETYPE.
	MD2MOVETYPE = "A"       'ABSOLUTE MOVES.
	MD2MOVE                 'MOVE TO THE GRID COORDINATES.
	MD2MOVETYPE = MD2MT     'RESTORE MOVETYPE.

End Sub

Sub MD2HOME ()

	'---------------------------------------------------------------------
	'NAME:      MD2HOME
	'DESC:      THE MD2HOME PROCEDURE IS USED TO MOVE THE STEPPER MOTOR
	'           TO A KNOWN HOME POSITION.  ALL OTHER MOVES ARE RELATIVE
	'           TO THIS HOME (ZERO) POSITION.  THE SELECTED MOTOR IS
	'           MOVED REVERSE UNTIL THE SWITCH IS ACTIVATED, THEN FORWARD
	'           UNTIL DEACTIVATED, THEN MOVED FORWARD ACCORDING TO THE
	'           MD2HOMEOFFSET(M) PARAMETER. THE CURRENT POSITION IS THEN
	'           SET TO ZERO - THIS IS THE HOME POSITION.  THE MD2MINSPEED
	'           PARAMETER IS USED FOR THE MOTOR SPEED - NO RAMPING IS DONE.
	'           ONLY 1 MOTOR (1-6) CAN BE HOME'D AT A TIME.
	'USAGE:     SET THE DESIRED MOTOR #, MOTOR PARAMETERS AND CALL.
	'INPUTS:    MOTOR # AND MOTOR PARAMETERS.
	'OUTPUTS:   CURRENT POSITION SET TO ZERO, MD2STATUS.
	'---------------------------------------------------------------------
   
	'LOCAL VARIABLES.
	Dim MD2DELAYLOOP As Long        'DELAY COUNTER BETWEEN STEPS.
	Dim MD2DIR As Integer       'DIRECTION. -1=REV, 1=FWD.
	Dim MD2MTRADR As Integer    'MOTOR PORT ADDRESS.
	Dim MD2PATS As Integer      'CALCULATED STEP PATTERN.
	Dim MD2PMASK As Integer     'SELECTED STEP PATTERN MASK.
	Dim MD2OMASK As Integer     'OTHER MOTOR PATTERN MASK.
	Dim MD2SMASK As Integer     'SWITCH MASK.
	Dim MD2INTREG As Integer    'INTERRUPT REGISTER.
	Dim MD2I As Long            'TEMPORARY.
	Dim MD2WAY As Single        'TEMP FOR VELOCITY INTEROLATION.
	Dim MD2MINDELAY As Long     'CALCULATED DELAY COUNTER.
	Dim MD2KEYIN As Integer     'KEYBOARD ENTRY.

	'CHECK FOR SETUP.
	If MD2MTRADR12 <> &H3BC Then MD2STATUS = "S": Exit Sub

	'CHECK FOR CALIBRATION.
	If Not MD2CALIBRATED Then MD2STATUS = "C": Exit Sub

	'CHECK FOR PORT AVAILABILITY.
	MD2STATUS = "P"
	If (MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 12) And Not MD2AVAILABLE12 Then Exit Sub
	If (MD2MOTOR = 3 Or MD2MOTOR = 4 Or MD2MOTOR = 34) And Not MD2AVAILABLE34 Then Exit Sub
	If (MD2MOTOR = 5 Or MD2MOTOR = 6 Or MD2MOTOR = 56) And Not MD2AVAILABLE56 Then Exit Sub

	'CHECK FOR ENABLED MD-2.
	MD2STATUS = "E"
	If (MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 12) And Not MD2ENABLED12 Then Exit Sub
	If (MD2MOTOR = 3 Or MD2MOTOR = 4 Or MD2MOTOR = 34) And Not MD2ENABLED34 Then Exit Sub
	If (MD2MOTOR = 5 Or MD2MOTOR = 6 Or MD2MOTOR = 56) And Not MD2ENABLED56 Then Exit Sub

	'CHECK FOR VALID MOTOR #.
	MD2STATUS = "B"
	If MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 3 Then MD2STATUS = "O"
	If MD2MOTOR = 4 Or MD2MOTOR = 5 Or MD2MOTOR = 6 Then MD2STATUS = "O"
	If MD2STATUS <> "O" Then Exit Sub

	'CHECK FOR VALID STEP TYPE.
	MD2STEPTYPE = UCase$(MD2STEPTYPE)
	If MD2STEPTYPE <> "H" And MD2STEPTYPE <> "D" And MD2STEPTYPE <> "S" Then MD2STATUS = "B": Exit Sub

	'INITIALIZE STATUS.
	MD2STATUS = "O"

	'SET STEP PATTERNS.
	If MD2STEPTYPE = "H" Then   'HALF STEP MODE.
		MD2STPPAT(0) = &H66: MD2STPPAT(1) = &H77
		MD2STPPAT(2) = &H33: MD2STPPAT(3) = &HBB
		MD2STPPAT(4) = &H99: MD2STPPAT(5) = &HDD
		MD2STPPAT(6) = &HCC: MD2STPPAT(7) = &HEE
	End If
	If MD2STEPTYPE = "S" Then   'SINGLE PHASE FULL STEP MODE.
		MD2STPPAT(0) = &H77: MD2STPPAT(1) = &HBB
		MD2STPPAT(2) = &HDD: MD2STPPAT(3) = &HEE
		MD2STPPAT(4) = &H77: MD2STPPAT(5) = &HBB
		MD2STPPAT(6) = &HDD: MD2STPPAT(7) = &HEE
	End If
	If MD2STEPTYPE = "D" Then   'DOUBLE PHASE FULL STEP MODE.
		MD2STPPAT(0) = &H66: MD2STPPAT(1) = &H33
		MD2STPPAT(2) = &H99: MD2STPPAT(3) = &HCC
		MD2STPPAT(4) = &H66: MD2STPPAT(5) = &H33
		MD2STPPAT(6) = &H99: MD2STPPAT(7) = &HCC
	End If

	'SET UP ADDRESS.
	If MD2MOTOR = 1 Or MD2MOTOR = 2 Then MD2MTRADR = MD2MTRADR12
	If MD2MOTOR = 3 Or MD2MOTOR = 4 Then MD2MTRADR = MD2MTRADR34
	If MD2MOTOR = 5 Or MD2MOTOR = 6 Then MD2MTRADR = MD2MTRADR56

	'SET UP PATTERN MASK, OTHER MOTOR'S MASK AND SWITCH MASK.
	If MD2MOTOR = 1 Or MD2MOTOR = 3 Or MD2MOTOR = 5 Then
		MD2PMASK = &HF
		MD2OMASK = &HF0
		MD2SMASK = &H20
	Else
		MD2PMASK = &HF0
		MD2OMASK = &HF
		MD2SMASK = &H10
	End If

	'CONVERT MINSPEED VELOCITY TO DELAY COUNTS.
	For MD2I = 2 To 23     'SEARCH FOR CORRECT ARRAY ELEMENTS.
		If (MD2MINSPEED(MD2MOTOR) * MD2UNITS(MD2MOTOR)) >= MD2VELOCITY(MD2I) Then Exit For
	Next MD2I
	'INTERPOLATE.
	MD2WAY = ((MD2MINSPEED(MD2MOTOR) * MD2UNITS(MD2MOTOR)) - MD2VELOCITY(MD2I - 1)) / (MD2VELOCITY(MD2I) - MD2VELOCITY(MD2I - 1))
	MD2MINDELAY = Int((((MD2DELAY(MD2I) - MD2DELAY(MD2I - 1)) * MD2WAY) + MD2DELAY(MD2I - 1)) + .5)

	'SELECT DIRECTION TO HOME.
	If MD2HOMEDIR(MD2MOTOR) = 0 Then MD2DIR = -1 Else MD2DIR = 1
   
	'GET KEYBOARD STATUS.
	MD2KEYIN = INP(&H60) And &H80

	'TURN OFF COM INTERRUPTS?
	If MD2INTERRUPTS = 0 Then
		MD2INTREG = INP(&H21)
		OUT &H21, MD2INTREG Or &H98
	End If

	'---------------------------------------------
	'MOVE MOTOR REVERSE UNTIL SWITCH ACTIVATED.
	'---------------------------------------------

	'LOOP UNTIL SWITCH IS ACTIVATED.
	Do Until (INP(MD2MTRADR + 1) And MD2SMASK) = 0

		'QUIT MOVING IF KEY PRESSED. (IF KEY MAKE CODE RECEIVED.)
		If (INP(&H60) And &H80) = 0 And MD2KEYIN = &H80 Then
			MD2STATUS = "K"
			Exit Do
		Else
			MD2KEYIN = INP(&H60) And &H80
		End If
		
		'POINT TO THE NEXT STEP PATTERN.
		MD2PATPTR(MD2MOTOR) = (MD2PATPTR(MD2MOTOR) + MD2DIR) And &H7

		'GET STEP PATTERN AND MASK OFF UNNEEDED BITS.
		MD2PATS = MD2STPPAT(MD2PATPTR(MD2MOTOR)) And MD2PMASK

		'DON'T DISTURB OTHER MOTORS BITS.
		MD2PATS = MD2PATS Or (INP(MD2MTRADR) And MD2OMASK)

		'OUTPUT THE STEP PATTERN TO MOVE THE MOTOR.
		OUT MD2MTRADR, MD2PATS

		'DELAY BETWEEN STEPS.
		For MD2DELAYLOOP = 1 To MD2MINDELAY: Next MD2DELAYLOOP
	
	Loop

	'----------------------------------------------
	'MOVE MOTOR FORWARD UNTIL SWITCH DEACTIVATED.
	'----------------------------------------------

	'LOOP UNTIL SWITCH IS DEACTIVATED.
	Do Until (INP(MD2MTRADR + 1) And MD2SMASK) <> 0

		'QUIT MOVING IF KEY PRESSED.
		If MD2STATUS = "K" Then Exit Do
		If (INP(&H60) And &H80) = 0 And MD2KEYIN = &H80 Then
			MD2STATUS = "K"
			Exit Do
		Else
			MD2KEYIN = INP(&H60) And &H80
		End If
		
		'POINT TO THE NEXT STEP PATTERN.
		MD2PATPTR(MD2MOTOR) = (MD2PATPTR(MD2MOTOR) - MD2DIR) And &H7
	   
		'GET STEP PATTERN AND MASK OFF UNNEEDED BITS.
		MD2PATS = MD2STPPAT(MD2PATPTR(MD2MOTOR)) And MD2PMASK

		'DON'T DISTURB OTHER MOTORS BITS.
		MD2PATS = MD2PATS Or (INP(MD2MTRADR) And MD2OMASK)
	   
		'OUTPUT THE STEP PATTERN TO MOVE THE MOTOR.
		OUT MD2MTRADR, MD2PATS

		'DELAY BETWEEN STEPS.
		For MD2DELAYLOOP = 1 To MD2MINDELAY: Next MD2DELAYLOOP
	
	Loop
   
	'-----------------------------------------------
	'OVER STEP HOME POSITION USING MD2HOMEOFFSET(M).
	'-----------------------------------------------

	'STEP UNTIL COUNT DEPLETED.
	MD2I = Int(MD2HOMEOFFSET(MD2MOTOR) * MD2UNITS(MD2MOTOR))
	Do Until MD2I = 0
	   
		'QUIT MOVING IF KEY PRESSED.
		If MD2STATUS = "K" Then Exit Do
		If ((INP(&H60) And &H80)) = 0 And (MD2KEYIN = &H80) Then
			MD2STATUS = "K"
			Exit Do
		Else
			MD2KEYIN = INP(&H60) And &H80
		End If
		
		'POINT TO THE NEXT STEP PATTERN.
		MD2PATPTR(MD2MOTOR) = (MD2PATPTR(MD2MOTOR) - MD2DIR) And &H7
	  
		'GET STEP PATTERN AND MASK OFF UNNEEDED BITS.
		MD2PATS = MD2STPPAT(MD2PATPTR(MD2MOTOR)) And MD2PMASK

		'DON'T DISTURB OTHER MOTORS BITS.
		MD2PATS = MD2PATS Or (INP(MD2MTRADR) And MD2OMASK)
	  
		'OUTPUT THE STEP PATTERN TO MOVE THE MOTOR.
		OUT MD2MTRADR, MD2PATS

		'DELAY BETWEEN STEPS.
		For MD2DELAYLOOP = 1 To MD2MINDELAY: Next MD2DELAYLOOP

		'UPDATE STEP COUNT.
		MD2I = MD2I - 1
   
	Loop

	'TURN COM INTERRUPTS BACK ON?
	If MD2INTERRUPTS = 0 Then OUT &H21, MD2INTREG

	'SET LAST DIRECTION ACCORDING TO HOMEDIR.
	If MD2HOMEDIR(MD2MOTOR) = 0 Then
		MD2LASTDIR(MD2MOTOR) = 1    'HOMEDIR=REV, SET LASTDIR TO FORWARD.
	Else
		MD2LASTDIR(MD2MOTOR) = -1   'HOMEDIR=FWD, SET LASTDIR TO REVERSE.
	End If

	'SET POSITION TO 0.
	MD2POSITION(MD2MOTOR) = 0

	'POWER OFF MOTOR IF DESIRED.
	If MD2HOLD = 0 Then OUT MD2MTRADR, &HFF

End Sub

Sub MD2INPUTS ()

	'---------------------------------------------------------------------
	'NAME:      MD2INPUTS
	'DESC:      THE MD2INPUTS PROCEDURE READS ALL INPUTS FROM ALL MD-2
	'           SYSTEMS INCLUDING HOME SWITCHES AND GENERAL PURPOSE INPUTS
	'           AND SETS VARIABLES ACCORDINGLY.  VARIABLES FOR PORTS THAT
	'           ARE NOT CONNECTED TO MD-2 SYSTEMS WILL CONTAIN USELESS
	'           INFORMATION.  EACH MD-2 SYSTEM HAS 2 HOME SWITCHES (ONE
	'           ASSOCIATED WITH EACH MOTOR) WHICH ARE INPUTS AND SOME
	'           MD-2 SYSTEMS HAVE 3 GENERAL PURPOSE INPUTS WHICH ARE
	'           USER DEFINED.  CHECK YOUR MANUAL TO SEE IF YOU HAVE
	'           GENERAL PURPOSE INPUTS.
	'USAGE:     USE TO READ THE CONDITION OF HOME SWITCHES, SENSORS
	'           AND OTHER INPUTS.
	'INPUTS:    NONE.
	'OUTPUTS:   INPUT VARIABLES -1=ACTIVE(LOGIC 0), 0=NOT ACTIVE(LOGIC 1).
	'
	'           VARIABLES  MD-2 PORT  INPUT           CONDITION
	'           ---------  ---------  -----           ---------
	'           MD2INPUT(11)  3BC     HOME SWITCH #1  -1=SWITCH ACTIVE (LOW).
	'           MD2INPUT(12)  3BC     HOME SWITCH #2  -1=SWITCH ACTIVE (LOW).
	'           MD2INPUT(13)  3BC     GEN PURP IN #1  -1=INPUT ACTIVE (LOW).
	'           MD2INPUT(14)  3BC     GEN PURP IN #2  -1=INPUT ACTIVE (LOW).
	'           MD2INPUT(15)  3BC     GEN PURP IN #3  -1=INPUT ACTIVE (LOW).
	'
	'           MD2INPUT(21)  378     HOME SWITCH #1  -1=SWITCH ACTIVE (LOW).
	'           MD2INPUT(22)  378     HOME SWITCH #2  -1=SWITCH ACTIVE (LOW).
	'           MD2INPUT(23)  378     GEN PURP IN #1  -1=INPUT ACTIVE (LOW).
	'           MD2INPUT(24)  378     GEN PURP IN #2  -1=INPUT ACTIVE (LOW).
	'           MD2INPUT(25)  378     GEN PURP IN #3  -1=INPUT ACTIVE (LOW).
	'
	'           MD2INPUT(31)  278     HOME SWITCH #1  -1=SWITCH ACTIVE (LOW).
	'           MD2INPUT(32)  278     HOME SWITCH #2  -1=SWITCH ACTIVE (LOW).
	'           MD2INPUT(33)  278     GEN PURP IN #1  -1=INPUT ACTIVE (LOW).
	'           MD2INPUT(34)  278     GEN PURP IN #2  -1=INPUT ACTIVE (LOW).
	'           MD2INPUT(35)  278     GEN PURP IN #3  -1=INPUT ACTIVE (LOW).
	'---------------------------------------------------------------------

	'FIRST CLEAR ALL VARIABLES.
	MD2INPUT(11) = 0
	MD2INPUT(12) = 0
	MD2INPUT(13) = 0
	MD2INPUT(14) = 0
	MD2INPUT(15) = 0
	MD2INPUT(21) = 0
	MD2INPUT(22) = 0
	MD2INPUT(23) = 0
	MD2INPUT(24) = 0
	MD2INPUT(25) = 0
	MD2INPUT(31) = 0
	MD2INPUT(32) = 0
	MD2INPUT(33) = 0
	MD2INPUT(34) = 0
	MD2INPUT(35) = 0

	'READ INPUTS AND SET VARIABLES IF INPUTS ARE ACTIVE (LOGIC ZERO).

	'MD-2 AT 3BC.
	If (INP(&H3BC + 1) And &H20) = 0 Then MD2INPUT(11) = -1
	If (INP(&H3BC + 1) And &H10) = 0 Then MD2INPUT(12) = -1
	If (INP(&H3BC + 1) And &H80) <> 0 Then MD2INPUT(13) = -1
	If (INP(&H3BC + 1) And &H40) <> 0 Then MD2INPUT(14) = -1
	If (INP(&H3BC + 1) And &H8) <> 0 Then MD2INPUT(15) = -1

	'MD-2 AT 378.
	If (INP(&H378 + 1) And &H20) = 0 Then MD2INPUT(21) = -1
	If (INP(&H378 + 1) And &H10) = 0 Then MD2INPUT(22) = -1
	If (INP(&H378 + 1) And &H80) <> 0 Then MD2INPUT(23) = -1
	If (INP(&H378 + 1) And &H40) <> 0 Then MD2INPUT(24) = -1
	If (INP(&H378 + 1) And &H8) <> 0 Then MD2INPUT(25) = -1
   
	'MD-2 AT 278.
	If (INP(&H278 + 1) And &H20) = 0 Then MD2INPUT(31) = -1
	If (INP(&H278 + 1) And &H10) = 0 Then MD2INPUT(32) = -1
	If (INP(&H278 + 1) And &H80) <> 0 Then MD2INPUT(33) = -1
	If (INP(&H278 + 1) And &H40) <> 0 Then MD2INPUT(34) = -1
	If (INP(&H278 + 1) And &H8) <> 0 Then MD2INPUT(35) = -1
												   
End Sub

Sub MD2MOVE ()

	'---------------------------------------------------------------------
	'NAME:      MD2MOVE
	'DESC:      THIS PROCEDURE IS USED TO MOVE MOTORS.
	'           SET MD2MOTOR TO 1,2,3,4,5 OR 6 FOR SINGLE MOTOR MOVES OR
	'           TO 12, 34 OR 56 FOR DUAL MOTOR MOVES.
	'           WHEN MOVING 2 MOTORS, BOTH MOTORS START AND STOP AT THE
	'           SAME TIME RESULTING IN STRAIGHT LINE MOTION ALSO KNOWN
	'           AS LINEAR INTERPOLATION.
	'           SPEED AND RAMPING OF THE MOTOR WITH THE LONGEST
	'           TRAVEL WILL BE USED.
	'           SOFT LIMIT VALUES WILL PREVENT OVERTRAVEL.
	'           THE HOME SWITCH IS IGNORED.
	'           IF SLOPE=0 THEN THE MOTOR WILL MOVE AT MD2MAXSPEED
	'           IF KEY PRESSED DURING MOVE, MOVE IS COMPLETED AND MD2STATUS=K
	'USAGE:     SET MOTOR #, MOTOR PARAMETERS AND CALL.
	'INPUTS:    MOTOR # (1-6,12,34 OR 56) AND MOTOR PARAMETERS.
	'OUTPUTS:   CURRENT POSITION, MD2STATUS.
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	Dim MD2MTRADR As Integer    'MOTOR PORT ADDRESS.
	Dim MD2PATS As Integer      'CALCULATED STEP PATTERN.
	Dim MD2INTREG As Integer    'INTERRUPT REGISTER.
	Dim MD2DIRSWAPL As Integer  'LONG MOTOR SWAP DIR. -1=YES, 1=NO.
	Dim MD2DIRSWAPS As Integer  'LONG MOTOR SWAP DIR. -1=YES, 1=NO.
	Dim MD2KEYIN As Integer     'KEYBOARD VALUE.

	'RAMPING VARIABLES.
	Dim MD2DELAYTMP As Long     'TEMPORARY DELAY LOOP COUNTER.
	Dim MD2DELAYCUR As Long     'CURRENT SPEED DELAY COUNT.
	Dim MD2DELAYFLAG As Long    'FLAG INDICATING DELAY ADJUSTMENT.
	Dim MD2DELAYEACH As Long    'DELAY ADJUSTMENT EACH STEP.
	Dim MD2DELAYCHANGE As Long  'AMOUNT OF CHANGE IN DELAY.
	Dim MD2DELAYCHANGE2 As Long 'DELAYCHANGE * 2.
	Dim MD2DECEL As Long        'STEP COUNT TO BEGIN DECELERATION MODE.
	Dim MD2DELAYSLOPE As Long   'SLOPE.
	Dim MD2DELAYSLOPE2 As Long  'SLOPE * 2.
	Dim MD2MODE As Integer      'SPEED MODE. 1=DECEL,0=CONSTANT,-1=ACCEL.
	Dim MD2MINDELAY As Long     'MINIMUM SPEED DELAY COUNT.
	Dim MD2MAXDELAY As Long     'MAXIMUM SPEED DELAY COUNT.
	Dim MD2WAY As Single        'TEMP FOR VELOCITY INTEROLATION.
	Dim MD2I As Integer         'ARRAY ELEMENT COUNTER.

	'LINEAR INTERPOLATION VARIABLES.
	Dim MD2LIERROR As Long      'ERROR VALUE FOR LINEAR INTERPOLATION.
	Dim MD2MOTORL As Integer    'MOTOR # TRAVELING THE LONGEST.
	Dim MD2MOTORS As Integer    'MOTOR # TRAVELING THE SHORTEST.
	Dim MD2DIRL As Integer      'LONG MOTOR DIRECTION. -1=REV,1=FWD.
	Dim MD2DIRS As Integer      'SHORT MOTOR DIRECTION.
	Dim MD2MASKL As Integer     'LONG MOTOR PATTERN MASK.
	Dim MD2MASKS As Integer     'SHORT MOTOR PATTERN MASK.
	Dim MD2STEPSL As Long       '# OF STEPS FOR LONG MOTOR.
	Dim MD2STEPSS As Long       '# OF STEPS FOR SHORT MOTOR.
	Dim MD2STEPSL2 As Long      'MD2STEPSL * 2.
	Dim MD2STEPSS2 As Long      'MD2STEPSS * 2.
	Dim MD2SWAPL As Long        'SWAP TEMPORARY VARIABLE.

	'CHECK FOR SETUP.
	If MD2MTRADR12 <> &H3BC Then MD2STATUS = "S": Exit Sub

	'IGNORE SOME CHECKS IF IN CALIBRATION MODE.
	If Not MD2CALMODE Then

	  'CHECK FOR CALIBRATION.
	  If Not MD2CALIBRATED Then MD2STATUS = "C": Exit Sub

	  'CHECK FOR PORT AVAILABILITY.
	  MD2STATUS = "P"
	  If (MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 12) And Not MD2AVAILABLE12 Then Exit Sub
	  If (MD2MOTOR = 3 Or MD2MOTOR = 4 Or MD2MOTOR = 34) And Not MD2AVAILABLE34 Then Exit Sub
	  If (MD2MOTOR = 5 Or MD2MOTOR = 6 Or MD2MOTOR = 56) And Not MD2AVAILABLE56 Then Exit Sub

	  'CHECK FOR ENABLED MD-2.
	  MD2STATUS = "E"
	  If (MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 12) And Not MD2ENABLED12 Then Exit Sub
	  If (MD2MOTOR = 3 Or MD2MOTOR = 4 Or MD2MOTOR = 34) And Not MD2ENABLED34 Then Exit Sub
	  If (MD2MOTOR = 5 Or MD2MOTOR = 6 Or MD2MOTOR = 56) And Not MD2ENABLED56 Then Exit Sub

	End If

	'CHECK FOR VALID MOTOR #.
	MD2STATUS = "B"
	If MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 3 Then MD2STATUS = "O"
	If MD2MOTOR = 4 Or MD2MOTOR = 5 Or MD2MOTOR = 6 Then MD2STATUS = "O"
	If MD2MOTOR = 12 Or MD2MOTOR = 34 Or MD2MOTOR = 56 Then MD2STATUS = "O"
	If MD2STATUS <> "O" Then Exit Sub

	'CHECK FOR VALID STEP TYPE.
	MD2STEPTYPE = UCase$(MD2STEPTYPE)
	If MD2STEPTYPE <> "H" And MD2STEPTYPE <> "D" And MD2STEPTYPE <> "S" Then MD2STATUS = "B": Exit Sub

	'CHECK FOR VALID MOVE TYPE.
	MD2MOVETYPE = UCase$(MD2MOVETYPE)
	If MD2MOVETYPE <> "A" And MD2MOVETYPE <> "R" Then MD2STATUS = "B": Exit Sub
												   
	'SET PORT ADDRESSES AND MOTOR NUMBERS.
	If MD2MOTOR = 1 Then
		MD2MTRADR = MD2MTRADR12
		MD2MOTORL = 1
		MD2MOTORS = 0
	End If
	If MD2MOTOR = 2 Then
		MD2MTRADR = MD2MTRADR12
		MD2MOTORL = 2
		MD2MOTORS = 0
	End If
	If MD2MOTOR = 3 Then
		MD2MTRADR = MD2MTRADR34
		MD2MOTORL = 3
		MD2MOTORS = 0
	End If
	If MD2MOTOR = 4 Then
		MD2MTRADR = MD2MTRADR34
		MD2MOTORL = 4
		MD2MOTORS = 0
	End If
	If MD2MOTOR = 5 Then
		MD2MTRADR = MD2MTRADR56
		MD2MOTORL = 5
		MD2MOTORS = 0
	End If
	If MD2MOTOR = 6 Then
		MD2MTRADR = MD2MTRADR56
		MD2MOTORL = 6
		MD2MOTORS = 0
	End If
	If MD2MOTOR = 12 Then
		MD2MTRADR = MD2MTRADR12
		MD2MOTORL = 1
		MD2MOTORS = 2
	End If
	If MD2MOTOR = 34 Then
		MD2MTRADR = MD2MTRADR34
		MD2MOTORL = 3
		MD2MOTORS = 4
	End If
	If MD2MOTOR = 56 Then
		MD2MTRADR = MD2MTRADR56
		MD2MOTORL = 5
		MD2MOTORS = 6
	End If

	'SET STEP PATTERNS.
	If MD2STEPTYPE = "H" Then   'HALF STEP MODE.
		MD2STPPAT(0) = &H66: MD2STPPAT(1) = &H77
		MD2STPPAT(2) = &H33: MD2STPPAT(3) = &HBB
		MD2STPPAT(4) = &H99: MD2STPPAT(5) = &HDD
		MD2STPPAT(6) = &HCC: MD2STPPAT(7) = &HEE
	End If
	If MD2STEPTYPE = "S" Then   'SINGLE PHASE FULL STEP MODE.
		MD2STPPAT(0) = &H77: MD2STPPAT(1) = &HBB
		MD2STPPAT(2) = &HDD: MD2STPPAT(3) = &HEE
		MD2STPPAT(4) = &H77: MD2STPPAT(5) = &HBB
		MD2STPPAT(6) = &HDD: MD2STPPAT(7) = &HEE
	End If
	If MD2STEPTYPE = "D" Then   'DOUBLE PHASE FULL STEP MODE.
		MD2STPPAT(0) = &H66: MD2STPPAT(1) = &H33
		MD2STPPAT(2) = &H99: MD2STPPAT(3) = &HCC
		MD2STPPAT(4) = &H66: MD2STPPAT(5) = &H33
		MD2STPPAT(6) = &H99: MD2STPPAT(7) = &HCC
	End If
						  
	'FIX VARIABLES FOR 0.
	MD2TARGET(0) = 0    'MOTOR 0 IS NO MOTOR, NO STEPS, ETC.
	MD2POSITION(0) = 0
	MD2UNITS(0) = 1
  
	'SET STEP COUNT AND DIRECTION FOR RELATIVE MOVES.
	If MD2MOVETYPE = "R" Then   'RELATIVE MOVE?
		'IF LIMITS EXCEEDED THEN QUIT.
		If (MD2TARGET(MD2MOTORL) + MD2POSITION(MD2MOTORL)) > MD2LIMITF(MD2MOTORL) Then MD2STATUS = "L"
		If (MD2TARGET(MD2MOTORL) + MD2POSITION(MD2MOTORL)) < MD2LIMITR(MD2MOTORL) Then MD2STATUS = "L"
		If (MD2TARGET(MD2MOTORS) + MD2POSITION(MD2MOTORS)) > MD2LIMITF(MD2MOTORS) Then MD2STATUS = "L"
		If (MD2TARGET(MD2MOTORS) + MD2POSITION(MD2MOTORS)) < MD2LIMITR(MD2MOTORS) Then MD2STATUS = "L"
		If MD2STATUS = "L" Then Exit Sub
		'SET THE # OF STEPS.
		MD2STEPSL = Int((MD2TARGET(MD2MOTORL) * MD2UNITS(MD2MOTORL)) + .5)
		MD2STEPSS = Int((MD2TARGET(MD2MOTORS) * MD2UNITS(MD2MOTORS)) + .5)
		'SET THE DIRECTION.
		If MD2STEPSL < 0 Then MD2DIRL = -1 Else MD2DIRL = 1
		If MD2STEPSS < 0 Then MD2DIRS = -1 Else MD2DIRS = 1
		'MAKE STEP COUNTS POSITIVE.
		MD2STEPSL = Abs(MD2STEPSL)
		MD2STEPSS = Abs(MD2STEPSS)
		'SET COMPLETED POSITIONS AND ROUND.
		MD2POSITION(MD2MOTORL) = MD2POSITION(MD2MOTORL) + ((MD2STEPSL * (1 / MD2UNITS(MD2MOTORL))) * MD2DIRL)
		MD2POSITION(MD2MOTORL) = Int((MD2POSITION(MD2MOTORL) * MD2UNITS(MD2MOTORL)) + .5)
		MD2POSITION(MD2MOTORL) = MD2POSITION(MD2MOTORL) / MD2UNITS(MD2MOTORL)

		MD2POSITION(MD2MOTORS) = MD2POSITION(MD2MOTORS) + ((MD2STEPSS * (1 / MD2UNITS(MD2MOTORS))) * MD2DIRS)
		MD2POSITION(MD2MOTORS) = Int((MD2POSITION(MD2MOTORS) * MD2UNITS(MD2MOTORS)) + .5)
		MD2POSITION(MD2MOTORS) = MD2POSITION(MD2MOTORS) / MD2UNITS(MD2MOTORS)
	End If
  
	'SET STEP COUNT AND DIRECTION FOR ABSOLUTE MOVES.
	If MD2MOVETYPE = "A" Then   'ABSOLUTE MOVE?
		'IF LIMITS EXCEEDED THEN QUIT.
		If MD2TARGET(MD2MOTORL) > MD2LIMITF(MD2MOTORL) Then MD2STATUS = "L"
		If MD2TARGET(MD2MOTORL) < MD2LIMITR(MD2MOTORL) Then MD2STATUS = "L"
		If MD2TARGET(MD2MOTORS) > MD2LIMITF(MD2MOTORS) Then MD2STATUS = "L"
		If MD2TARGET(MD2MOTORS) < MD2LIMITR(MD2MOTORS) Then MD2STATUS = "L"
		If MD2STATUS = "L" Then Exit Sub
		'SET THE # OF STEPS.
		MD2STEPSL = Int(((MD2TARGET(MD2MOTORL) - MD2POSITION(MD2MOTORL)) * MD2UNITS(MD2MOTORL)) + .5)
		MD2STEPSS = Int(((MD2TARGET(MD2MOTORS) - MD2POSITION(MD2MOTORS)) * MD2UNITS(MD2MOTORS)) + .5)
		'SET THE DIRECTION.
		If MD2STEPSL < 0 Then MD2DIRL = -1 Else MD2DIRL = 1
		If MD2STEPSS < 0 Then MD2DIRS = -1 Else MD2DIRS = 1
		'MAKE STEP COUNTS POSITIVE.
		MD2STEPSL = Abs(MD2STEPSL)
		MD2STEPSS = Abs(MD2STEPSS)
		'SET COMPLETED POSITIONS AND ROUND.
		MD2POSITION(MD2MOTORL) = MD2TARGET(MD2MOTORL)
		MD2POSITION(MD2MOTORL) = Int((MD2POSITION(MD2MOTORL) * MD2UNITS(MD2MOTORL)) + .5)
		MD2POSITION(MD2MOTORL) = MD2POSITION(MD2MOTORL) / MD2UNITS(MD2MOTORL)

		MD2POSITION(MD2MOTORS) = MD2TARGET(MD2MOTORS)
		MD2POSITION(MD2MOTORS) = Int((MD2POSITION(MD2MOTORS) * MD2UNITS(MD2MOTORS)) + .5)
		MD2POSITION(MD2MOTORS) = MD2POSITION(MD2MOTORS) / MD2UNITS(MD2MOTORS)
	End If

	'SWAP DIRECTIONS IF NEEDED.
	If MD2HOMEDIR(MD2MOTORL) = -1 Then MD2DIRL = MD2DIRL * -1
	If MD2HOMEDIR(MD2MOTORS) = -1 Then MD2DIRS = MD2DIRS * -1

	'BACKLASH COMPENSATION, ADJUST STEP COUNTS.
	If MD2DIRL <> MD2LASTDIR(MD2MOTORL) Then
		MD2STEPSL = MD2STEPSL + Int((MD2BACKLASH(MD2MOTORL) * MD2UNITS(MD2MOTORL)) + .5)
		MD2LASTDIR(MD2MOTORL) = MD2DIRL
	End If
	If MD2DIRS <> MD2LASTDIR(MD2MOTORS) Then
		MD2STEPSS = MD2STEPSS + Int((MD2BACKLASH(MD2MOTORS) * MD2UNITS(MD2MOTORS)) + .5)
		MD2LASTDIR(MD2MOTORS) = MD2DIRS
	End If

	'SWAP PARAMETERS DEPENDING ON LONGEST/SHORTEST STEP COUNT.
	If MD2STEPSL < MD2STEPSS Then
		MD2SWAPL = MD2STEPSL: MD2STEPSL = MD2STEPSS: MD2STEPSS = MD2SWAPL
		MD2SWAPL = MD2MOTORL: MD2MOTORL = MD2MOTORS: MD2MOTORS = MD2SWAPL
		MD2SWAPL = MD2DIRL: MD2DIRL = MD2DIRS: MD2DIRS = MD2SWAPL
	End If
   
	'INITIALIZE ERROR ACCUMULATOR AND PRECALCULATE VARS FOR INTERPOLATION.
	MD2STEPSL2 = MD2STEPSL + MD2STEPSL
	MD2STEPSS2 = MD2STEPSS + MD2STEPSS
	MD2LIERROR = MD2STEPSS2 - MD2STEPSL
									
	'SET UP PATTERN MASKS.
	If MD2MOTORL = 1 Or MD2MOTORL = 3 Or MD2MOTORL = 5 Then
		MD2MASKL = &HF
		MD2MASKS = &HF0
	Else
		MD2MASKL = &HF0
		MD2MASKS = &HF
	End If

	'SETUP RAMPING----------------.

	'IF IN CALIBRATION MODE SET RAMPING TO MAXSPEED OTHERWISE
	'SET RAMPING PARAMETERS NORMALLY.
	If MD2CALMODE Then

		MD2MODE = 0     'CONSTANT SPEED.
		MD2DECEL = 0    'NEVER BEGIN DECELERATION.
		MD2DELAYCUR = MD2MAXSPEED(MD2MOTORL)

	Else
		'NORMAL RAMPING.
		'FIND MINDELAY AND MAXDELAY BY SEARCHING VELOCITY ARRAY AND
		'INTERPOLATING BETWEEN ELEMENTS.
		'SETUP DELAY COUNT ADJUSMENTS TO BE LINEARIZE RAMP WHILE STEPPING.
	   
		'CONVERT MINSPEED VELOCITY TO DELAY COUNTS.
		For MD2I = 2 To 23     'SEARCH FOR CORRECT ARRAY ELEMENTS.
			If (MD2MINSPEED(MD2MOTORL) * MD2UNITS(MD2MOTORL)) >= MD2VELOCITY(MD2I) Then Exit For
		Next MD2I
		'INTERPOLATE BETWEEN ELEMENTS.
		MD2WAY = ((MD2MINSPEED(MD2MOTORL) * MD2UNITS(MD2MOTORL)) - MD2VELOCITY(MD2I - 1)) / (MD2VELOCITY(MD2I) - MD2VELOCITY(MD2I - 1))
		MD2MINDELAY = Int((((MD2DELAY(MD2I) - MD2DELAY(MD2I - 1)) * MD2WAY) + MD2DELAY(MD2I - 1)) + .5)

		'CONVERT MAXSPEED VELOCITY TO DELAY COUNTS.
		For MD2I = 2 To 23     'SEARCH FOR CORRECT ARRAY ELEMENTS.
			If (MD2MAXSPEED(MD2MOTORL) * MD2UNITS(MD2MOTORL)) >= MD2VELOCITY(MD2I) Then Exit For
		Next MD2I
		'INTERPOLATE BETWEEN ELEMENTS.
		MD2WAY = ((MD2MAXSPEED(MD2MOTORL) * MD2UNITS(MD2MOTORL)) - MD2VELOCITY(MD2I - 1)) / (MD2VELOCITY(MD2I) - MD2VELOCITY(MD2I - 1))
		MD2MAXDELAY = Int((((MD2DELAY(MD2I) - MD2DELAY(MD2I - 1)) * MD2WAY) + MD2DELAY(MD2I - 1)) + .5)

		'SET UP RAMPING PARAMETERS.
		If (MD2SLOPE(MD2MOTORL) <= 0) Or (MD2MINDELAY <= MD2MAXDELAY) Then

			'CONSTANT SPEED MODE----------------.

			'CONSTANT SPEED MODE.
			MD2MODE = 2

			'CURRENT SPEED.
			MD2DELAYCUR = MD2MAXDELAY

		Else

			'RAMPING MOVE-----------------.

			'ACCELERATION MODE.
			MD2MODE = -1

			'CURRENT STARTING SPEED.
			MD2DELAYCUR = MD2MINDELAY

			'SET STEP COUNT TO BEGIN DECELERATION.
			If MD2STEPSL < ((MD2SLOPE(MD2MOTORL) * MD2UNITS(MD2MOTORL)) * 2) Then
				'MOVE TOO SHORT TO REACH MAX SPEED.
				MD2DECEL = Int((MD2STEPSL / 2) + .5)
			Else
				'MOVE LONG ENOUGH TO REACH MAX SPEED.
				MD2DECEL = Int(MD2SLOPE(MD2MOTORL) * MD2UNITS(MD2MOTORL))
			End If
		   
			'SET SLOPES.
			MD2DELAYSLOPE = Int(MD2SLOPE(MD2MOTORL) * MD2UNITS(MD2MOTORL))
			MD2DELAYSLOPE2 = MD2DELAYSLOPE + MD2DELAYSLOPE

			'SET AMOUNT OF CHANGE IN DELAY COUNT.
			MD2DELAYCHANGE = MD2MINDELAY - MD2MAXDELAY

			'DIFFERENT CALCULATIONS FOR SLOPE <=1 AND >1.
			If MD2DELAYCHANGE > MD2DELAYSLOPE Then
				'SLOPE > 1.
				'SET DELAY ADJUSTMENT THAT OCCURS EVERY STEP.
				MD2DELAYEACH = Int(MD2DELAYCHANGE / MD2DELAYSLOPE)
				'RESET DELAYCHANGE ACCORDING TO PREVIOUS CALC.
				MD2DELAYCHANGE = MD2DELAYCHANGE - (MD2DELAYSLOPE * MD2DELAYEACH)
			Else
				'SLOPE <= 1.
				'DON'T ADJUST EVERY STEP.
				MD2DELAYEACH = 0
			End If
			   
			'PRECALCULATE THIS.
			MD2DELAYCHANGE2 = MD2DELAYCHANGE + MD2DELAYCHANGE
		   
			'PRESET ADJUSTMENT FLAG.
			MD2DELAYFLAG = MD2DELAYCHANGE2 - MD2DELAYSLOPE
		   
		End If
		
	End If

	'GET KEYBOARD INPUT STATUS.
	MD2KEYIN = INP(&H60) And &H80

	'TURN OFF COM INTERRUPTS?
	If MD2INTERRUPTS = 0 Then
		MD2INTREG = INP(&H21)
		OUT &H21, MD2INTREG Or &H98
	End If

	'INITIALIZE STATUS.
	MD2STATUS = "O"
  
	'MOVE LOOP.
	Do Until MD2STEPSL = 0
		
		'SET STATUS IF KEY PRESSED.
		If (INP(&H60) And &H80) = 0 And MD2KEYIN = &H80 Then
			MD2STATUS = "K"
		Else
			MD2KEYIN = INP(&H60) And &H80
		End If
		'MOVE LONG MOTOR.
	   
		'POINT TO THE NEXT STEP PATTERN.
		MD2PATPTR(MD2MOTORL) = (MD2PATPTR(MD2MOTORL) + MD2DIRL) And &H7

		'GET STEP PATTERN AND MASK OFF UNNEEDED BITS.
		MD2PATS = MD2STPPAT(MD2PATPTR(MD2MOTORL)) And MD2MASKL

		'MOVE SHORT MOTOR IF NEEDED.
		If MD2LIERROR >= 0 Then
		   
			'POINT TO THE NEXT STEP PATTERN.
			MD2PATPTR(MD2MOTORS) = (MD2PATPTR(MD2MOTORS) + MD2DIRS) And &H7
	   
			'OR IN SHORT MOTOR PATTERN.
			MD2PATS = MD2PATS Or (MD2STPPAT(MD2PATPTR(MD2MOTORS)) And MD2MASKS)

			'UPDATE ERROR FOR LINEAR INTERPOLATION.
			MD2LIERROR = MD2LIERROR - MD2STEPSL2
		   
		Else

			'DON'T MOVE SHORT.
			MD2PATS = MD2PATS Or (INP(MD2MTRADR) And MD2MASKS)
	   
		End If
		   
		'UPDATE ERROR FOR LINEAR INTERPOLATION.
		MD2LIERROR = MD2LIERROR + MD2STEPSS2
	   
		'OUTPUT THE STEP PATTERN TO MOVE THE MOTOR.
		OUT MD2MTRADR, MD2PATS

		'DELAY BETWEEN STEPS.
		For MD2DELAYTMP = 1 To MD2DELAYCUR: Next MD2DELAYTMP
	  
		'DECREMENT STEP COUNT.
		MD2STEPSL = MD2STEPSL - 1
					   
		'CALCULATE NEW SPEED.
	   
		If MD2MODE = 1 Then   'DECELERATION.

			'BEGIN CONSTANT IF REACHED MIN SPEED.
			If MD2DELAYCUR >= MD2MINDELAY Then MD2MODE = 2
		   
			'DECREASE CURRENT SPEED.
			MD2DELAYCUR = MD2DELAYCUR + MD2DELAYEACH
			If MD2DELAYFLAG >= 0 Then
				MD2DELAYCUR = MD2DELAYCUR + 1
				MD2DELAYFLAG = MD2DELAYFLAG - MD2DELAYSLOPE2
			End If
			MD2DELAYFLAG = MD2DELAYFLAG + MD2DELAYCHANGE2

		End If
	  
		If MD2MODE = 0 Then   'CONSTANT THEN DECELERATION.
			'BEGIN DECELERATION BASED ON STEPS REMAINING.
			If MD2STEPSL <= MD2DECEL Then MD2MODE = 1
		End If
	   
		If MD2MODE = -1 Then  'ACCELERATION.

			'INCREASE CURRENT SPEED.
			MD2DELAYCUR = MD2DELAYCUR - MD2DELAYEACH
			If MD2DELAYFLAG >= 0 Then
				MD2DELAYCUR = MD2DELAYCUR - 1
				MD2DELAYFLAG = MD2DELAYFLAG - MD2DELAYSLOPE2
			End If
			MD2DELAYFLAG = MD2DELAYFLAG + MD2DELAYCHANGE2

			'BEGIN DECELERATION BASED ON STEPS REMAINING.
			If MD2STEPSL <= MD2DECEL Then MD2MODE = 1
		  
			'BEGIN CONSTANT SPEED IF REACHED MAX SPEED.
			If MD2DELAYCUR <= MD2MAXDELAY Then MD2MODE = 0
	  
		End If
	   
	Loop
   
	'TURN COM INTERRUPTS BACK ON?
	If MD2INTERRUPTS = 0 Then OUT &H21, MD2INTREG

	'POWER OFF MOTOR IF DESIRED.
	If MD2HOLD = 0 Then
		OUT MD2MTRADR, &HFF
		'DELAY TO ALLOW STEP TO TAKE.
		For MD2DELAYTMP = 1 To MD2MINDELAY: Next MD2DELAYTMP
	End If

End Sub

Sub MD2OFF ()

	'---------------------------------------------------------------------
	'NAME:      MD2OFF
	'DESC:      THE MD2OFF PROCEDURE RETURNS A PARALLEL PRINTER PORT
	'           REFERENCED BY THE MOTOR # TO ITS PREVIOUS STATE READY
	'           FOR USE WITH A PRINTER AND DISABLES THE MD-2.
	'USAGE:     USE AT THE END OF A MOTION CONTROL PROGRAM.
	'INPUTS:    MOTOR # DETERMINES PORT.
	'OUTPUTS:   NONE.
	'---------------------------------------------------------------------
   
	'LOCAL VARIABLES.
	Dim MD2TIMER As Single

	'CHECK FOR SETUP.
	If MD2MTRADR12 <> &H3BC Then MD2STATUS = "S": Exit Sub

	'CHECK FOR VALID MOTOR #.
	MD2STATUS = "B"
	If MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 3 Then MD2STATUS = "O"
	If MD2MOTOR = 4 Or MD2MOTOR = 5 Or MD2MOTOR = 6 Then MD2STATUS = "O"
	If MD2MOTOR = 12 Or MD2MOTOR = 34 Or MD2MOTOR = 56 Then MD2STATUS = "O"
	If MD2STATUS <> "O" Then Exit Sub

	'CHECK FOR PORT AVAILABILITY.
	MD2STATUS = "P"
	If (MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 12) And Not MD2AVAILABLE12 Then Exit Sub
	If (MD2MOTOR = 3 Or MD2MOTOR = 4 Or MD2MOTOR = 34) And Not MD2AVAILABLE34 Then Exit Sub
	If (MD2MOTOR = 5 Or MD2MOTOR = 6 Or MD2MOTOR = 56) And Not MD2AVAILABLE56 Then Exit Sub
		 
	'-STROBE PIN HIGH, -ALF PIN HIGH, -INIT PIN LOW, -SELIN PIN LOW, IRQ OFF.
	If MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 12 Then
		OUT MD2MTRADR12 + 2, &H4
	End If
	If MD2MOTOR = 3 Or MD2MOTOR = 4 Or MD2MOTOR = 34 Then
		OUT MD2MTRADR34 + 2, &H4
	End If
	If MD2MOTOR = 5 Or MD2MOTOR = 6 Or MD2MOTOR = 56 Then
		OUT MD2MTRADR56 + 2, &H4
	End If

	'DELAY FOR .2 SECONDS.
	MD2TIMER = Timer: Do: Loop Until Timer > MD2TIMER + .2
   
	'TURN -INIT PIN HIGH.
	If MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 12 Then
		OUT MD2MTRADR12 + 2, &HC
		MD2ENABLED12 = 0
	End If
	If MD2MOTOR = 3 Or MD2MOTOR = 4 Or MD2MOTOR = 34 Then
		OUT MD2MTRADR34 + 2, &HC
		MD2ENABLED34 = 0
	End If
	If MD2MOTOR = 5 Or MD2MOTOR = 6 Or MD2MOTOR = 56 Then
		OUT MD2MTRADR56 + 2, &HC
		MD2ENABLED56 = 0
	End If
															
	MD2STATUS = "O"

End Sub

Sub MD2ON ()

	'---------------------------------------------------------------------
	'NAME:      MD2ON
	'DESC:      THE MD2ON PROCEDURE INITIALIZES A PARALLEL PRINTER PORT
	'           AND TURNS ON AN MD-2.
	'USAGE:     USE AT THE BEGINNING OF A MOTION CONTROL PROGRAM BUT
	'           AFTER THE MD2SETUP SUBROUTINE.
	'INPUTS:    MOTOR # DETERMINES PORT.
	'OUTPUTS:   NONE.
	'---------------------------------------------------------------------

	'CHECK FOR SETUP.
	If MD2MTRADR12 <> &H3BC Then MD2STATUS = "S": Exit Sub

	'CHECK FOR VALID MOTOR #.
	MD2STATUS = "B"
	If MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 3 Then MD2STATUS = "O"
	If MD2MOTOR = 4 Or MD2MOTOR = 5 Or MD2MOTOR = 6 Then MD2STATUS = "O"
	If MD2MOTOR = 12 Or MD2MOTOR = 34 Or MD2MOTOR = 56 Then MD2STATUS = "O"
	If MD2STATUS <> "O" Then Exit Sub

	'CHECK FOR PORT AVAILABILITY.
	MD2STATUS = "P"
	If (MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 12) And Not MD2AVAILABLE12 Then Exit Sub
	If (MD2MOTOR = 3 Or MD2MOTOR = 4 Or MD2MOTOR = 34) And Not MD2AVAILABLE34 Then Exit Sub
	If (MD2MOTOR = 5 Or MD2MOTOR = 6 Or MD2MOTOR = 56) And Not MD2AVAILABLE56 Then Exit Sub

	'TURN OFF ALL MOTOR PHASES,
	'SET -STROBE LOW (MD-2 ENABLED),
	'    -ALF LOW (OUTPUT #1 OFF HIGH),
	'    -INIT LOW (STANDBY OFF),
	'    -SELIN LOW (OUTPUT #2 OF HIGH),
	'    IRQ DISABLED.
   
	If MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 12 Then
		OUT MD2MTRADR12, &HFF
		OUT MD2MTRADR12 + 2, &H5
		MD2ENABLED12 = -1
		MD2STATUS = "O"
	End If
   
	If MD2MOTOR = 3 Or MD2MOTOR = 4 Or MD2MOTOR = 34 Then
		OUT MD2MTRADR34, &HFF
		OUT MD2MTRADR34 + 2, &H5
		MD2ENABLED34 = -1
		MD2STATUS = "O"
	End If
   
	If MD2MOTOR = 5 Or MD2MOTOR = 6 Or MD2MOTOR = 56 Then
		OUT MD2MTRADR56, &HFF
		OUT MD2MTRADR56 + 2, &H5
		MD2ENABLED56 = -1
		MD2STATUS = "O"
	End If

End Sub

Sub MD2OUTPUTS ()
 
	'---------------------------------------------------------------------
	'NAME:      MD2OUTPUTS
	'DESC:      THE MD2OUTPUTS PROCEDURE TURNS OUTPUT PINS ON AND OFF.
	'           THE INPUT/OUTPUT PORT ON THE BACK OF THE MD-2 SYSTEM
	'           HAS 2 OUTPUTS AND 3 INPUTS.  SOME MD-2 SYSTEMS DO NOT
	'           HAVE THIS I/O PORT.  CHECK YOUR MANUAL FOR MORE INFORMATION.
	'           WHEN TURNED ON, THE OUTPUT PIN WILL BE DRIVEN TO GROUND
	'           TO TURN ON SINKING LOADS SUCH AS RELAYS WHICH CAN CONTROL
	'           LARGER LOADS.  WHEN TURNED OFF, THEN OUTPUT PIN WILL
	'           RETURN TO A LOGIC 1 WHICH IS 5 VOLTS DC.
	'           MAKE SURE NOT TO EXCEED MAXIMUM CURRENT DRAW ON OUTPUT PINS.
	'           THE MD2OUTPUTCODE DETERMINES WHICH MD-2, PIN AND ON/OFF.
	'           USE THE FOLLOWING CHART TO DETERMINE WHICH CODE TO USE.
	'           NOTICE THE FIRST DIGIT OF THE CODE SELECTS WHICH MD-2 SYSTEM,
	'           THE SECOND DIGIT SELECTS THE PIN, AND THE THIRD SELECTS
	'           ON OR OFF (1 OR 0).
	'
	'           MD2OUTPUTCODE    MD-2 PORT    OUTPUT  ACTION
	'           -------------    ---------    ------  ------
	'               110             3BC         #1      OFF
	'               111             3BC         #1      ON
	'               120             3BC         #2      OFF
	'               121             3BC         #2      ON
	'
	'               210             378         #1      OFF
	'               211             378         #1      ON
	'               220             378         #2      OFF
	'               221             378         #2      ON
	'
	'               310             278         #1      OFF
	'               311             278         #1      ON
	'               320             278         #2      OFF
	'               321             278         #2      ON
	'
	'USAGE:     SET THE MD2OUTPUTCODE PARAMETER AND CALL.  USE TO CONTROL
	'           TOOLING SUCH AS DRILLS AND DISPENSERS.
	'INPUTS:    MD2OUTPUTCODE DETERMINES WHICH, MD-2, WHICH PIN, ON/OFF.
	'OUTPUTS:   NONE.
	'---------------------------------------------------------------------

	'PORT 3BC, OUTPUT #1, OFF
	If MD2OUTPUTCODE = 110 Then
		OUT MD2MTRADR12 + 2, INP(MD2MTRADR12 + 2) And &HF7
	End If

	'PORT 3BC, OUTPUT #1, ON
	If MD2OUTPUTCODE = 111 Then
		OUT MD2MTRADR12 + 2, INP(MD2MTRADR12 + 2) Or &H8
	End If

	'PORT 3BC, OUTPUT #2, OFF
	If MD2OUTPUTCODE = 120 Then
		OUT MD2MTRADR12 + 2, INP(MD2MTRADR12 + 2) And &HFD
	End If

	'PORT 3BC, OUTPUT #2, ON
	If MD2OUTPUTCODE = 121 Then
		OUT MD2MTRADR12 + 2, INP(MD2MTRADR12 + 2) Or &H2
	End If
  

	'PORT 378, OUTPUT #1, OFF
	If MD2OUTPUTCODE = 210 Then
		OUT MD2MTRADR34 + 2, INP(MD2MTRADR34 + 2) And &HF7
	End If

	'PORT 378, OUTPUT #1, ON
	If MD2OUTPUTCODE = 211 Then
		OUT MD2MTRADR34 + 2, INP(MD2MTRADR34 + 2) Or &H8
	End If

	'PORT 378, OUTPUT #2, OFF
	If MD2OUTPUTCODE = 220 Then
		OUT MD2MTRADR34 + 2, INP(MD2MTRADR34 + 2) And &HFD
	End If

	'PORT 378, OUTPUT #2, ON
	If MD2OUTPUTCODE = 221 Then
		OUT MD2MTRADR34 + 2, INP(MD2MTRADR34 + 2) Or &H2
	End If
  

	'PORT 278, OUTPUT #1, OFF
	If MD2OUTPUTCODE = 310 Then
		OUT MD2MTRADR56 + 2, INP(MD2MTRADR56 + 2) And &HF7
	End If

	'PORT 278, OUTPUT #1, ON
	If MD2OUTPUTCODE = 311 Then
		OUT MD2MTRADR56 + 2, INP(MD2MTRADR56 + 2) Or &H8
	End If

	'PORT 278, OUTPUT #2, OFF
	If MD2OUTPUTCODE = 320 Then
		OUT MD2MTRADR56 + 2, INP(MD2MTRADR56 + 2) And &HFD
	End If

	'PORT 278, OUTPUT #2, ON
	If MD2OUTPUTCODE = 321 Then
		OUT MD2MTRADR56 + 2, INP(MD2MTRADR56 + 2) Or &H2
	End If

	MD2STATUS = "O"

End Sub

Sub MD2PARLOAD ()

	'---------------------------------------------------------------------
	'NAME:      MD2PARLOAD
	'DESC:      THIS PROCEDURE IS USED TO LOAD ALL MOTOR PARAMETERS
	'           FROM A DISK FILE.  WHEN USED WITH THE MD2PARSAVE PROCEDURE,
	'           A PROGRAM CAN BE CREATED WHICH SAVES ITS STATUS BEFORE
	'           POWERING DOWN AND RESTORES WHEN COMMING BACK ON ELIMINATING
	'           THE NEED FOR THE USER TO RE-ENTER ALL MOTOR PARAMETERS.
	'USAGE:     SET THE DESIRED FILE NAME IN MD2PARFILE AND CALL.
	'INPUTS:    FILE NAME AND ALL MOTOR PARAMETERS.
	'OUTPUTS:   STATUS = O FOR OK OR E FOR ERROR.
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	Dim MD2MTR As Integer      'MOTOR COUNTER.
	Dim MD2XS As String        'DUMMY STRING VARIABLE.
	Dim MD2XI As Integer       'DUMMY INTEGER VARIABLE.
	Dim MD2FILENUM As Integer  'FREE FILE NUMBER.

	'DEFAULT RETURN STATUS TO OK.
	MD2STATUS = "O"
  
	'LOAD PARAMETERS.
	On Error GoTo MD2PARLOADERROR
	MD2FILENUM = FreeFile
	Open MD2PARFILE For Input As MD2FILENUM
   
	'MOTOR PARAMETERS.
	Input #MD2FILENUM, MD2XS     'FILE NAME
	Input #MD2FILENUM, MD2HOLD
	Input #MD2FILENUM, MD2INTERRUPTS
	Input #MD2FILENUM, MD2MOTOR
	Input #MD2FILENUM, MD2MOVETYPE
	Input #MD2FILENUM, MD2STEPTYPE

	'CHECK FOR ERRORS INDICATING CORRUPT PARAMETER FILE.
	If MD2HOLD <> 0 And MD2HOLD <> -1 Then MD2STATUS = "F"
	If MD2MOVETYPE <> "A" And MD2MOVETYPE <> "R" Then MD2STATUS = "F"
   
	'MOTOR PARAMETER ARRAYS.
	For MD2MTR = 1 To 6
		If MD2STATUS <> "O" Then Exit For
		Input #MD2FILENUM, MD2XI
		Input #MD2FILENUM, MD2BACKLASH(MD2MTR)
		Input #MD2FILENUM, MD2HOMEDIR(MD2MTR)
		Input #MD2FILENUM, MD2HOMEOFFSET(MD2MTR)
		Input #MD2FILENUM, MD2LIMITF(MD2MTR)
		Input #MD2FILENUM, MD2LIMITR(MD2MTR)
		Input #MD2FILENUM, MD2MAXSPEED(MD2MTR)
		If MD2MAXSPEED(MD2MTR) < 0 Then MD2STATUS = "F"
		Input #MD2FILENUM, MD2MINSPEED(MD2MTR)
		If MD2MINSPEED(MD2MTR) < 0 Then MD2STATUS = "F"
		Input #MD2FILENUM, MD2MOTORNAME(MD2MTR)
		Input #MD2FILENUM, MD2POSITION(MD2MTR)
		Input #MD2FILENUM, MD2SLOPE(MD2MTR)
		If MD2SLOPE(MD2MTR) < 0 Then MD2STATUS = "F"
		Input #MD2FILENUM, MD2TARGET(MD2MTR)
		Input #MD2FILENUM, MD2UNITS(MD2MTR)
		If MD2UNITS(MD2MTR) < 0 Then MD2STATUS = "F"
		Input #MD2FILENUM, MD2UNITNAME(MD2MTR)
		Input #MD2FILENUM, MD2PATPTR(MD2MTR)
		Input #MD2FILENUM, MD2LASTDIR(MD2MTR)
	Next MD2MTR
   
	'GRID PARAMETERS.
	Input #MD2FILENUM, MD2GRIDBEGINX
	Input #MD2FILENUM, MD2GRIDBEGINY
	Input #MD2FILENUM, MD2GRIDSPACEX
	Input #MD2FILENUM, MD2GRIDSPACEY
	Input #MD2FILENUM, MD2GRIDTARGETX
	Input #MD2FILENUM, MD2GRIDTARGETY

	'CIRCLE PARAMETERS.
	Input #MD2FILENUM, MD2CIRCLERADIUSX
	Input #MD2FILENUM, MD2CIRCLERADIUSY
	Input #MD2FILENUM, MD2CIRCLECENTERX
	Input #MD2FILENUM, MD2CIRCLECENTERY
	Input #MD2FILENUM, MD2CIRCLESTART
	Input #MD2FILENUM, MD2CIRCLEARC
	Input #MD2FILENUM, MD2CIRCLECHORD
   
	'FINISH UP.
	Close MD2FILENUM
	On Error GoTo 0
	Exit Sub

'FILE ERROR HANDLER.
MD2PARLOADERROR:
	MD2STATUS = "F"
	Resume Next

End Sub

Sub MD2PARSAVE ()

	'---------------------------------------------------------------------
	'NAME:      MD2PARSAVE
	'DESC:      THIS PROCEDURE IS USED TO SAVE ALL MOTOR PARAMETERS
	'           TO A DISK FILE.  WHEN USED WITH THE MD2PARLOAD PROCEDURE,
	'           A PROGRAM CAN BE CREATED WHICH SAVES ITS STATUS BEFORE
	'           POWERING DOWN AND RESTORES WHEN COMMING BACK ON ELIMINATING
	'           THE NEED FOR THE USER TO RE-ENTER ALL MOTOR PARAMETERS.
	'USAGE:     SET THE DESIRED FILE NAME IN MD2PARFILE AND CALL.
	'INPUTS:    FILE NAME AND ALL MOTOR PARAMETERS.
	'OUTPUTS:   STATUS = O FOR OK OR E FOR ERROR.
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	Dim MD2MTR As Integer           'MOTOR COUNTER.
	Dim MD2FILENUM As Integer       'FREE FILE NUMBER.
 
	'DEFAULT RETURN STATUS TO OK.
	MD2STATUS = "O"
 
	'SAVE PARAMETERS.
	On Error GoTo MD2PARSAVEERROR
	MD2FILENUM = FreeFile
	Open MD2PARFILE For Output As MD2FILENUM
   
	'MOTOR PARAMETERS.
	Write #MD2FILENUM, MD2PARFILE
	Write #MD2FILENUM, MD2HOLD, MD2INTERRUPTS, MD2MOTOR, MD2MOVETYPE, MD2STEPTYPE
   
	'MOTOR PARAMETER ARRAYS.
	For MD2MTR = 1 To 6
		If MD2STATUS = "F" Then Exit For
		Print #MD2FILENUM, MD2MTR; ",";
		Print #MD2FILENUM, MD2BACKLASH(MD2MTR); ",";
		Print #MD2FILENUM, MD2HOMEDIR(MD2MTR); ",";
		Print #MD2FILENUM, MD2HOMEOFFSET(MD2MTR); ",";
		Print #MD2FILENUM, MD2LIMITF(MD2MTR); ",";
		Print #MD2FILENUM, MD2LIMITR(MD2MTR); ",";
		Print #MD2FILENUM, MD2MAXSPEED(MD2MTR); ",";
		Print #MD2FILENUM, MD2MINSPEED(MD2MTR); ",";
		Print #MD2FILENUM, Chr$(34); MD2MOTORNAME(MD2MTR); Chr$(34); ",";
		Print #MD2FILENUM, MD2POSITION(MD2MTR); ",";
		Print #MD2FILENUM, MD2SLOPE(MD2MTR); ",";
		Print #MD2FILENUM, MD2TARGET(MD2MTR); ",";
		Print #MD2FILENUM, MD2UNITS(MD2MTR); ",";
		Print #MD2FILENUM, Chr$(34); MD2UNITNAME(MD2MTR); Chr$(34); ",";
		Print #MD2FILENUM, MD2PATPTR(MD2MTR); ",";
		Print #MD2FILENUM, MD2LASTDIR(MD2MTR)
	Next MD2MTR
  
	'GRID PARAMETERS.
	Print #MD2FILENUM, MD2GRIDBEGINX; ",";
	Print #MD2FILENUM, MD2GRIDBEGINY; ",";
	Print #MD2FILENUM, MD2GRIDSPACEX; ",";
	Print #MD2FILENUM, MD2GRIDSPACEY; ",";
	Print #MD2FILENUM, MD2GRIDTARGETX; ",";
	Print #MD2FILENUM, MD2GRIDTARGETY

	'CIRCLE PARAMETERS.
	Print #MD2FILENUM, MD2CIRCLERADIUSX; ",";
	Print #MD2FILENUM, MD2CIRCLERADIUSY; ",";
	Print #MD2FILENUM, MD2CIRCLECENTERX; ",";
	Print #MD2FILENUM, MD2CIRCLECENTERY; ",";
	Print #MD2FILENUM, MD2CIRCLESTART; ",";
	Print #MD2FILENUM, MD2CIRCLEARC; ",";
	Print #MD2FILENUM, MD2CIRCLECHORD

	'FINISH UP.
	Close MD2FILENUM
	On Error GoTo 0
	Exit Sub

'FILE ERROR HANDLER.
MD2PARSAVEERROR:
	MD2STATUS = "F"
	Resume Next

End Sub

Sub MD2SEQLOAD ()

	'---------------------------------------------------------------------
	'NAME:      MD2SEQLOAD
	'DESC:      THIS PROCEDURE IS USED TO LOAD A MOTION SEQUENCE
	'           FROM A DISK FILE AND STORES IN MD2SEQUENCE STRING.
	'           THIS SEQUENCE FILE CAN BE EXECUTED USING THE MD2SEQRUN
	'           SUBROUTINE.
	'USAGE:     SET THE DESIRED FILE NAME IN MD2SEQFILE AND CALL.
	'INPUTS:    FILE NAME AND ALL MOTOR PARAMETERS.
	'OUTPUTS:   STATUS = O FOR OK OR E FOR ERROR.
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	Dim MD2FILENUM As Integer       'FREE FILE NUMBER.
 
	'INITIALIZE VARIABLES.
	MD2STATUS = "O"                 'DEFAULT RETURN STATUS TO OK.
	MD2SEQUENCE = ""                'CLEAR COMMANDS.
 
	'LOAD SEQUENCE FILE.
	On Error GoTo MD2SEQLOADERROR
	MD2FILENUM = FreeFile
	Open MD2SEQFILE For Input As MD2FILENUM
	MD2SEQUENCE = Input$(LOF(MD2FILENUM), MD2FILENUM)

	'FINISH UP.
	Close MD2FILENUM
	On Error GoTo 0
	Exit Sub

'FILE ERROR HANDLER.
MD2SEQLOADERROR:
	MD2STATUS = "F"
	Resume Next

End Sub

Sub MD2SEQRUN ()

	'---------------------------------------------------------------------
	'NAME:      MD2SEQRUN
	'DESC:      THIS PROCEDURE WILL EXECUTE A SEQUENCE OF COMMANDS
	'           WHICH ARE STORED IN THE MD2SEQUENCE VARIABLE.
	'           THE LENGTH OF THE SEQUENCE IS LIMITED TO 32,766.
	'           COMMANDS ARE LIKE BASIC LANGUAGE COMMANDS SUCH AS
	'           MD2POSITION(2)=1000 OR SIMPLY MD2MOVE.  ALLOWS
	'           MOTOR PARAMETERS TO BE SET AND MD-2 SUBROUTINES
	'           TO BE RUN SUCH AS MD2MOVE, MD2HOME, MD2PARLOAD, ETC.
	'           EACH INDIVIDUAL COMMAND IS EXTRACTED FROM THE MD2SEQUENCE
	'           VARIABLE AND PASSED TO MD2SEQRUNDO FOR EXECUTION.
	'           EXECUTION BEGINS WITH MD2POINTER.
	'USAGE:     LOAD MD2SEQUENCE, SET MD2POINTER, THEN CALL MD2SEQRUN.
	'INPUTS:    MD2SEQUENCE,MD2POINTER AND ALL MOTOR PARAMETERS.
	'OUTPUTS:   ALL MOTOR PARAMETERS, MD2STATUS, MD2POINTER.
	'           DISTROYS MD2COMMAND.
	'           WHEN DONE, MD2POINTER POINTS TO THE BEGINNING OF THE
	'           COMMAND AFTER THE LAST ONE EXECUTED.
	'           IF AN ERROR OCCURED (MD2STATUS<>"O") THEN MD2COMMAND
	'           CONTAINS THE COMMAND WITH THE ERROR.
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	Dim MD2CHAR As String           'CURRENT CHARACTER.
	Dim MD2PTR As Integer           'COMMAND POINTER.
	Dim MD2CMD As String            'WORKING COMMAND.
	Dim MD2TIMETEMP As Single       'TIMER STORAGE.
	Dim MD2NUMBER As String         'VALID NUMBER CHECK.
	Dim MD2OPENP As Integer         'OPEN PAREN LOCATION.
	Dim MD2CLOSEDP As Integer       'CLOSED PAREN LOCATION.
	Dim MD2EQUAL As Integer         'EQUAL SIGN LOCATION.
	Dim MD2VALUE As String          'VALID VALUE.
	Dim MD2SUBSCRIPT As String      'VALID SUBSCRIPT.
	Dim MD2VALIDCHAR As Integer     'VALID CHARACTER COUNTER.
	Dim MD2KEYIN As Integer         'KEYBOARD VALUE.
	
	'INITIALIZE.
	MD2COMMAND = ""                 'CLEAR COMMAND.
	MD2STATUS = "B"                 'BAD COMMAND.

	'CHECK FOR EMPTY SEQUENCE AND INVALID POINTER.
	If MD2SEQUENCE = "" Then Exit Sub
	If MD2POINTER = 0 Then MD2POINTER = 1
	If MD2POINTER > Len(MD2SEQUENCE) Then Exit Sub

'EXTRACT COMMANDS AND SEND TO MD2SEQRUNDO.
MD2SEQRUNEXTRACT:

	'GET THE CURRENT CHARACTER.
	MD2CHAR = Mid$(MD2SEQUENCE, MD2POINTER, 1)

	'POINT TO NEXT CHARACTER.
	MD2POINTER = MD2POINTER + 1

	'IF COMMENT THEN IGNORE REST OF LINE.
	'POINTER = NEXT CHARACTER WHEN DONE.
	If MD2CHAR = "'" Then
		Do
			'END OF SEQUENCE?
			If MD2POINTER > Len(MD2SEQUENCE) Then Exit Do
		   
			'GET CHARACTER.
			MD2CHAR = Mid$(MD2SEQUENCE, MD2POINTER, 1)
		   
			'POINT TO NEXT CHARACTER.
			MD2POINTER = MD2POINTER + 1
		   
			'DONE WITH COMMENT IF CR, LF, FF.
			If MD2CHAR = Chr$(10) Then Exit Do
			If MD2CHAR = Chr$(12) Then Exit Do
			If MD2CHAR = Chr$(13) Then Exit Do
		Loop
	End If

	'END-OF-COMMAND CHARACTER?
	If MD2CHAR = Chr$(10) Then GoTo MD2SEQRUNSEND
	If MD2CHAR = Chr$(12) Then GoTo MD2SEQRUNSEND
	If MD2CHAR = Chr$(13) Then GoTo MD2SEQRUNSEND
	If MD2CHAR = ":" Then GoTo MD2SEQRUNSEND
   
	'ADD CHARACTER.
	MD2COMMAND = MD2COMMAND + MD2CHAR
   
	'END OF SEQUENCE?
	If MD2POINTER > Len(MD2SEQUENCE) Then GoTo MD2SEQRUNSEND
   
	GoTo MD2SEQRUNEXTRACT


'DO THE COMMAND, CHECK FOR ERRORS, END AND SETUP FOR NEXT ONE.
'
MD2SEQRUNSEND:

	'DO THE COMMAND.
	GoSub MD2SEQRUNDO

	'CHECK FOR ERRORS.
	If MD2STATUS <> "O" Then Exit Sub

	'END?
	If MD2CMD = "END" Then Exit Sub

	'END OF SEQUENCE?
	If MD2POINTER > Len(MD2SEQUENCE) Then Exit Sub

	'SETUP FOR NEXT COMMAND.
	MD2COMMAND = ""
	GoTo MD2SEQRUNEXTRACT

'PARS EQUATION----------------------------------------------------
'RETURN MD2SUBSCRIPT INSIDE (), IF NOT AVAILABLE OR NOT 1-6 THEN = "".
'RETURN MD2VALUE AFTER =, IF NOT AVAIABLE OR VALID THEN = "".

MD2SEQRUNPARS:
	'INITIALIZE VARIABLES.
	MD2VALUE = ""
	MD2SUBSCRIPT = ""

	'GET CHARACTERS AFTER =. CHECK FOR VALIDITY AND EMPTYNESS.
	MD2EQUAL = InStr(MD2CMD, "=")
	If MD2EQUAL <> 0 Then
		MD2NUMBER = Right$(MD2CMD, Len(MD2CMD) - MD2EQUAL)
		GoSub MD2SEQRUNVALID
		If MD2NUMBER <> "" Then MD2VALUE = MD2NUMBER
	End If

	'GET CHARACTERS INSIDE ().

	'IF NO ( THEN DONE.
	MD2OPENP = InStr(MD2CMD, "(")
	If MD2OPENP = 0 Then Return

	'IF NO ) THEN ERROR. IF ) BEFORE ( THEN ERROR, IF EMPTY THEN ERROR.
	MD2CLOSEDP = InStr(MD2CMD, ")")
	If MD2CLOSEDP = 0 Then Return
	If MD2CLOSEDP < MD2OPENP Then Return
	If MD2CLOSEDP = MD2OPENP + 1 Then Return

	'EXTRACT THE SUBSCRIPT.
	MD2NUMBER = Mid$(MD2CMD, MD2OPENP + 1, (MD2CLOSEDP - MD2OPENP) - 1)

	'CHECK FOR VALID NUMBER.
	GoSub MD2SEQRUNVALID
	If MD2NUMBER = "" Then Return

	'IF NOT AN INTEGER THEN ERROR.
	If Val(MD2NUMBER) <> Int(Val(MD2NUMBER)) Then Return

	'IF NOT 1-6 THEN ERROR.
	If Val(MD2NUMBER) < 1 Or Val(MD2NUMBER) > 6 Then Return

	MD2SUBSCRIPT = MD2NUMBER
	Return


'VALID NUMBER?----------------------------------------------
'RETURN MD2NUMBER,  IF NOT VALID THEN MD2NUMBER=""
'
MD2SEQRUNVALID:
	If MD2NUMBER = "" Then Return
	For MD2VALIDCHAR = 1 To Len(MD2NUMBER)
		'GREATER THAN 57 ISN'T NUMBERS.
		If Asc(Mid$(MD2NUMBER, MD2VALIDCHAR, 1)) > 57 Then
			MD2NUMBER = ""
			Exit For
		End If
		'LESS THAN 45 ISN'T NUMBERS BUT LEAVES - AND .
		If Asc(Mid$(MD2NUMBER, MD2VALIDCHAR, 1)) < 45 Then
			MD2NUMBER = ""
			Exit For
		End If
		'IS IT A / ?
		If Asc(Mid$(MD2NUMBER, MD2VALIDCHAR, 1)) = 47 Then
			MD2NUMBER = ""
			Exit For
		End If
	Next MD2VALIDCHAR
	Return
		  

'-----------------------------------------------------------------------
'CLEAN COMMAND, AND EXECUTE.
'
MD2SEQRUNDO:

	'DEFAULT TO BAD PARAMETER.
	MD2STATUS = "B"

	'CHECK FOR EMPTY COMMAND.
	If MD2COMMAND = "" Then MD2STATUS = "O": Return
  
	'REMOVE WHITE SPACES AND DOUBLE QUOTES.
	MD2CMD = ""
	For MD2PTR = 1 To Len(MD2COMMAND)
	  
		'GET CURRENT CHARACTER AND CONVERT TO UPPER CASE.
		MD2CHAR = UCase$(Mid$(MD2COMMAND, MD2PTR, 1))

		'IGNORE CONTROL CHARS, QUOTES, GRAPHICS.
		If Asc(MD2CHAR) > 34 And Asc(MD2CHAR) < 128 Then
			MD2CMD = MD2CMD + MD2CHAR
		End If

	Next MD2PTR
  
	'EXTRACT NUMBER AFTER = AND SUBSCRIPT IF AVAILABLE.
	GoSub MD2SEQRUNPARS
	
	'SUBROUTINES ---------------------------------------------------
   
	If MD2CMD = "" Then MD2STATUS = "O": Return
	If MD2CMD = "MD2MOVE" Then MD2MOVE: Return
	If MD2CMD = "MD2HOME" Then MD2HOME: Return
	If MD2CMD = "MD2ON" Then MD2ON: Return
	If MD2CMD = "MD2OFF" Then MD2OFF: Return
	If MD2CMD = "MD2SETUP" Then MD2SETUP: Return
	If MD2CMD = "MD2PARLOAD" Then MD2PARLOAD: Return
	If MD2CMD = "MD2PARSAVE" Then MD2PARSAVE: Return
	If MD2CMD = "MD2OUTPUTS" Then MD2OUTPUTS: Return
	If MD2CMD = "MD2CIRCLE" Then MD2CIRCLE: Return
	If MD2CMD = "MD2GRID" Then MD2GRID: Return
	If MD2CMD = "MD2STANDBYON" Then MD2STANDBYON: Return
	If MD2CMD = "MD2STANDBYOFF" Then MD2STANDBYOFF: Return

	'COMMANDS -------------------------------------------------------

	'END.
	If MD2CMD = "END" Then
		MD2STATUS = "O": Return
	End If

	'SLEEP.
	If Left$(MD2CMD, 5) = "SLEEP" Then
		If Val(Right$(MD2CMD, Len(MD2CMD) - 5)) = 0 Then
			'Wait for Keypress (Sleep 0).
			MD2KEYIN = INP(&H60)
			Do
				'KEYPRESS CHECK.
				If INP(&H60) <> MD2KEYIN Then Exit Do
			Loop
		Else
			'Wait second(s).
			MD2TIMETEMP = Timer: Do: Loop Until Timer > MD2TIMETEMP + Val(Right$(MD2CMD, Len(MD2CMD) - 5))
		End If
		MD2STATUS = "O": Return
	End If
   
	'BEEP.
	If Left$(MD2CMD, 4) = "BEEP" Then
		Beep
		MD2STATUS = "O": Return
	End If
   
	'SET NON-ARRAY PARAMETERS ----------------------------------------
  
	'MOTOR.
	If Left$(MD2CMD, 9) = "MD2MOTOR=" Then
		If MD2VALUE = "" Then Return
		MD2MOTOR = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
  
	'INTERRUPTS.
	If Left$(MD2CMD, 14) = "MD2INTERRUPTS=" Then
		If MD2VALUE = "" Then Return
		MD2INTERRUPTS = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
  
	'HOLD.
	If Left$(MD2CMD, 8) = "MD2HOLD=" Then
		If MD2VALUE = "" Then Return
		MD2HOLD = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
  
	'MOVE TYPE.
	If Left$(MD2CMD, 12) = "MD2MOVETYPE=" Then
		MD2MOVETYPE = Right$(MD2CMD, Len(MD2CMD) - 12)
		MD2STATUS = "O"
		Return
	End If
  
	'STEP TYPE.
	If Left$(MD2CMD, 12) = "MD2STEPTYPE=" Then
		MD2STEPTYPE = Right$(MD2CMD, Len(MD2CMD) - 12)
		MD2STATUS = "O"
		Return
	End If
 
	'OUTPUT CODE.
	If Left$(MD2CMD, 14) = "MD2OUTPUTCODE=" Then
		If MD2VALUE = "" Then Return
		MD2OUTPUTCODE = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
  
	'MD2PARFILE.
	If Left$(MD2CMD, 11) = "MD2PARFILE=" Then
		MD2PARFILE = Right$(MD2CMD, Len(MD2CMD) - 11)
		MD2STATUS = "O"
		Return
	End If
   
	'MD2CALFILE.
	If Left$(MD2CMD, 11) = "MD2CALFILE=" Then
		MD2CALFILE = Right$(MD2CMD, Len(MD2CMD) - 11)
		MD2STATUS = "O"
		Return
	End If
   
	'MD2CIRCLERADIUSX.
	If Left$(MD2CMD, 17) = "MD2CIRCLERADIUSX=" Then
		If MD2VALUE = "" Then Return
		MD2CIRCLERADIUSX = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
			 
	'MD2CIRCLERADIUSY.
	If Left$(MD2CMD, 17) = "MD2CIRCLERADIUSY=" Then
		If MD2VALUE = "" Then Return
		MD2CIRCLERADIUSY = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
			
	'MD2CIRCLECENTERX.
	If Left$(MD2CMD, 17) = "MD2CIRCLECENTERX=" Then
		If MD2VALUE = "" Then Return
		MD2CIRCLECENTERX = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
			
	'MD2CIRCLECENTERY.
	If Left$(MD2CMD, 17) = "MD2CIRCLECENTERY=" Then
		If MD2VALUE = "" Then Return
		MD2CIRCLECENTERY = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
   
	'MD2CIRCLESTART.
	If Left$(MD2CMD, 15) = "MD2CIRCLESTART=" Then
		If MD2VALUE = "" Then Return
		MD2CIRCLESTART = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
		   
	'MD2CIRCLEARC.
	If Left$(MD2CMD, 13) = "MD2CIRCLEARC=" Then
		If MD2VALUE = "" Then Return
		MD2CIRCLEARC = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
		  
	'MD2CIRCLECHORD.
	If Left$(MD2CMD, 15) = "MD2CIRCLECHORD=" Then
		If MD2VALUE = "" Then Return
		MD2CIRCLECHORD = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
  
	'MD2GRIDBEGINX.
	If Left$(MD2CMD, 14) = "MD2GRIDBEGINX=" Then
		If MD2VALUE = "" Then Return
		MD2GRIDBEGINX = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
 
	'MD2GRIDBEGINY.
	If Left$(MD2CMD, 14) = "MD2GRIDBEGINY=" Then
		If MD2VALUE = "" Then Return
		MD2GRIDBEGINY = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
 
	'MD2GRIDSPACEX.
	If Left$(MD2CMD, 14) = "MD2GRIDSPACEX=" Then
		If MD2VALUE = "" Then Return
		MD2GRIDSPACEX = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If

	'MD2GRIDSPACEY.
	If Left$(MD2CMD, 14) = "MD2GRIDSPACEY=" Then
		If MD2VALUE = "" Then Return
		MD2GRIDSPACEY = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If

	'MD2GRIDTARGETX.
	If Left$(MD2CMD, 15) = "MD2GRIDTARGETX=" Then
		If MD2VALUE = "" Then Return
		MD2GRIDTARGETX = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If

	'MD2GRIDTARGETY.
	If Left$(MD2CMD, 15) = "MD2GRIDTARGETY=" Then
		If MD2VALUE = "" Then Return
		MD2GRIDTARGETY = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
		  
	'SET ARRAY PARAMETERS -----------------------------------------------

	'BACKLASH.
	If Left$(MD2CMD, 12) = "MD2BACKLASH(" Then
		If MD2VALUE = "" Or MD2SUBSCRIPT = "" Then Return
		MD2BACKLASH(Val(MD2SUBSCRIPT)) = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
  
	'HOMEDIR.
	If Left$(MD2CMD, 11) = "MD2HOMEDIR(" Then
		If MD2VALUE = "" Or MD2SUBSCRIPT = "" Then Return
		MD2HOMEDIR(Val(MD2SUBSCRIPT)) = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
  
	'HOMEOFFSET.
	If Left$(MD2CMD, 14) = "MD2HOMEOFFSET(" Then
		If MD2VALUE = "" Or MD2SUBSCRIPT = "" Then Return
		MD2HOMEOFFSET(Val(MD2SUBSCRIPT)) = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
  
	'LIMITF.
	If Left$(MD2CMD, 10) = "MD2LIMITF(" Then
		If MD2VALUE = "" Or MD2SUBSCRIPT = "" Then Return
		MD2LIMITF(Val(MD2SUBSCRIPT)) = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
  
	'LIMITR.
	If Left$(MD2CMD, 10) = "MD2LIMITR(" Then
		If MD2VALUE = "" Or MD2SUBSCRIPT = "" Then Return
		MD2LIMITR(Val(MD2SUBSCRIPT)) = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
  
	'MAXSPEED.
	If Left$(MD2CMD, 12) = "MD2MAXSPEED(" Then
		If MD2VALUE = "" Or MD2SUBSCRIPT = "" Then Return
		MD2MAXSPEED(Val(MD2SUBSCRIPT)) = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
  
	'MINSPEED.
	If Left$(MD2CMD, 12) = "MD2MINSPEED(" Then
		If MD2VALUE = "" Or MD2SUBSCRIPT = "" Then Return
		MD2MINSPEED(Val(MD2SUBSCRIPT)) = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
  
	'MOTORNAME.
	If Left$(MD2CMD, 13) = "MD2MOTORNAME(" Then
		If MD2SUBSCRIPT = "" Then Return
		MD2MOTORNAME(Val(MD2SUBSCRIPT)) = Right$(MD2CMD, Len(MD2CMD) - 16)
		MD2STATUS = "O": Return
	End If
  
	'POSITION.
	If Left$(MD2CMD, 12) = "MD2POSITION(" Then
		If MD2VALUE = "" Or MD2SUBSCRIPT = "" Then Return
		MD2POSITION(Val(MD2SUBSCRIPT)) = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
  
	'SLOPE.
	If Left$(MD2CMD, 9) = "MD2SLOPE(" Then
		If MD2VALUE = "" Or MD2SUBSCRIPT = "" Then Return
		MD2SLOPE(Val(MD2SUBSCRIPT)) = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
  
	'TARGET.
	If Left$(MD2CMD, 10) = "MD2TARGET(" Then
		If MD2VALUE = "" Or MD2SUBSCRIPT = "" Then Return
		MD2TARGET(Val(MD2SUBSCRIPT)) = Val(MD2VALUE)
		MD2STATUS = "O"
		Return
	End If
  
	'UNITS.
	If Left$(MD2CMD, 9) = "MD2UNITS(" Then
		If MD2VALUE = "" Or MD2SUBSCRIPT = "" Then Return
		MD2UNITS(Val(MD2SUBSCRIPT)) = Int(Val(MD2VALUE))
		MD2STATUS = "O"
		Return
	End If
 
	'UNITNAME.
	If Left$(MD2CMD, 12) = "MD2UNITNAME(" Then
		If MD2SUBSCRIPT = "" Then Return
		MD2UNITNAME(Val(MD2SUBSCRIPT)) = Right$(MD2CMD, Len(MD2CMD) - 15)
		MD2STATUS = "O"
		Return
	End If

	'MUST BE A BAD COMMAND ---------------------------------------------
   
	Return
  
End Sub

Sub MD2SEQSAVE ()

	'---------------------------------------------------------------------
	'NAME:      MD2SEQSAVE
	'DESC:      THIS PROCEDURE IS USED TO SAVE A MOTION SEQUENCE
	'           TO A DISK FILE FROM MD2SEQUENCE STRING.
	'USAGE:     SET THE DESIRED FILE NAME IN MD2SEQFILE AND CALL.
	'INPUTS:    FILE NAME, MD2SEQUENCE.
	'OUTPUTS:   STATUS = O FOR OK OR E FOR ERROR.
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	Dim MD2FILENUM As Integer      'FREE FILE NUMBER.

	'DEFAULT RETURN STATUS TO OK.
	MD2STATUS = "O"

	'SAVE PARAMETERS.
	On Error GoTo MD2SEQSAVEERROR
	MD2FILENUM = FreeFile
	Open MD2SEQFILE For Output As MD2FILENUM
	Print #MD2FILENUM, MD2SEQUENCE;

	'FINISH UP.
	Close MD2FILENUM
	On Error GoTo 0
	Exit Sub

'FILE ERROR HANDLER.
MD2SEQSAVEERROR:
	MD2STATUS = "F"
	Resume Next

End Sub

Sub MD2SETUP ()

	'---------------------------------------------------------------------
	'NAME:      MD2SETUP
	'DESC:      THE MD2SETUP PROCEDURE SETS DEFAULT MD-2 SYSTEM
	'           PARAMETERS SUCH AS MOTOR SPEED, CURRENT POSITION, ETC.
	'           AVAILABLE PARALLEL PRINTER PORTS ARE LOCATED AND
	'           VARIABLES ARE SET ACCORDINGLY.
	'USAGE:     USE AT THE BEGINNING OF A MOTION CONTROL PROGRAM.
	'           MUST BE USED -BEFORE- ANY OTHER MOTION CONTROL SUBROUTINE!
	'INPUTS:    NONE.
	'OUTPUTS:   DEFAULT MOTOR PARAMETERS, PORT AVAILABLE VARIABLES.
	'---------------------------------------------------------------------

	'LOCAL VARIABLES.
	Dim MD2ELEMENT As Integer           'ARRAY ELEMENT COUNTER.
	Dim MD2FILENUM As Integer           'FREE FILE NUMBER.
   
	'SETUP PORT ADDRESSES.
	MD2MTRADR12 = &H3BC
	MD2MTRADR34 = &H378
	MD2MTRADR56 = &H278

	'SETUP MOTOR PARAMETER DEFAULTS.
	For MD2MOTOR = 1 To 6
		MD2BACKLASH(MD2MOTOR) = 0       'BACKLASH COMPENSATION.
		MD2HOMEOFFSET(MD2MOTOR) = 0     'HOME OFFSET.
		MD2HOMEDIR(MD2MOTOR) = 0        'HOME DIRECTION.
		MD2LIMITF(MD2MOTOR) = 999999    'FORWARD LIMIT.
		MD2LIMITR(MD2MOTOR) = -999999   'REVERSE LIMIT.
		MD2LASTDIR(MD2MOTOR) = 1        'LAST DIRECTION - FORWARD.
		MD2POSITION(MD2MOTOR) = 0       'POSITIONS = 0.
		MD2PATPTR(MD2MOTOR) = 0         'STEP PATTERN POINTERS.
		MD2TARGET(MD2MOTOR) = 400       'TARGET POSITION,STEPS.
		MD2MINSPEED(MD2MOTOR) = 100     'MINIMUM SPEED.
		MD2MAXSPEED(MD2MOTOR) = 500     'MAXIMUM SPEED.
		MD2MOTORNAME(MD2MOTOR) = "#" + Str$(MD2MOTOR)    'NAME OF MOTOR.
		MD2SLOPE(MD2MOTOR) = 100        'RAMPING SLOPE IN STEPS.
		MD2UNITS(MD2MOTOR) = 1          '1 UNIT (STEPS) PER STEP.
		MD2UNITNAME(MD2MOTOR) = "Steps"
	Next MD2MOTOR
	MD2MOTOR = 1                'MOTOR NUMBER.
	MD2MOVETYPE = "R"           'RELATIVE MOVES.
	MD2STEPTYPE = "H"           'HALF STEP MODE.
	MD2STATUS = "O"             'STATUS=OK.
	MD2HOLD = 0                 'DONT HOLD MOTORS.
	MD2INTERRUPTS = 0           'INTERRUPTS OFF DURING MOVES.
	MD2PARFILE = "DEFAULTS.PAR" 'DEFAULT MOTOR PARAMETERS.
	MD2SEQFILE = ""             'SEQUENCE FILE.
	If MD2CALFILE = "" Then MD2CALFILE = "MD2.CAL"  'CALIBRATION FILE.
	MD2CALMODE = 0              'CALIBRATION MODE-OFF.

	'GRID DEFAULT PARAMETERS.
	MD2GRIDBEGINX = 0           'COLUMN BEGIN.
	MD2GRIDBEGINY = 0           'ROW BEGIN.
	MD2GRIDSPACEX = 1           'COLUMN SPACING.
	MD2GRIDSPACEY = 1           'ROW SPACING.
	MD2GRIDTARGETX = 0          'COLUMN TARGET.
	MD2GRIDTARGETY = 0          'ROW TARGET.
   
	'CIRCLE DEFAULT PARAMETERS.
	MD2CIRCLERADIUSX = 1        'RADIUS X.
	MD2CIRCLERADIUSY = 1        'RADIUS Y.
	MD2CIRCLECENTERX = 1        'CENTER X.
	MD2CIRCLECENTERY = 1        'CENTER Y.
	MD2CIRCLESTART = 0          'START ANGLE.
	MD2CIRCLEARC = 360          'ARC ANGLE (360=CIRCLE).
	MD2CIRCLECHORD = 1          'CHORD ANGLE.
   
	'INSURE ALL MD-2 SYSTEMS ARE OFF.
	OUT MD2MTRADR12 + 2, &HC
	OUT MD2MTRADR34 + 2, &HC
	OUT MD2MTRADR56 + 2, &HC

	'FIND PARALLEL PRINTER PORTS AND SET VARIABLES.
	OUT MD2MTRADR12, &HAA          'SEND DATA TO PORTS.
	OUT MD2MTRADR34, &HAA          'IF READ BACK THEN AVAILABLE.
	OUT MD2MTRADR56, &HAA
	If INP(MD2MTRADR12) = &HAA Then MD2AVAILABLE12 = -1 Else MD2AVAILABLE12 = 0
	If INP(MD2MTRADR34) = &HAA Then MD2AVAILABLE34 = -1 Else MD2AVAILABLE34 = 0
	If INP(MD2MTRADR56) = &HAA Then MD2AVAILABLE56 = -1 Else MD2AVAILABLE56 = 0
	OUT MD2MTRADR12, &HFF          'TURN OFF ALL PHASES.
	OUT MD2MTRADR34, &HFF
	OUT MD2MTRADR56, &HFF

	'LOAD SPEED CALIBRATION PARAMETERS.
	MD2STATUS = "O"
	On Error GoTo MD2SETUPERROR
	MD2FILENUM = FreeFile
	Open MD2CALFILE For Input As MD2FILENUM
	For MD2ELEMENT = 1 To 24
		If MD2STATUS <> "O" Then Exit For
		Input #MD2FILENUM, MD2DELAY(MD2ELEMENT), MD2VELOCITY(MD2ELEMENT)
		If MD2DELAY(MD2ELEMENT) <= 0 Then MD2STATUS = "F"
		If MD2VELOCITY(MD2ELEMENT) <= 0 Then MD2STATUS = "F"
	Next MD2ELEMENT
	'CLEAR CALIBRATION ARRAYS IF FILE ERROR.
	If MD2STATUS <> "O" Then
		For MD2ELEMENT = 1 To 16
			MD2DELAY(MD2ELEMENT) = 0: MD2VELOCITY(MD2ELEMENT) = 0
		Next MD2ELEMENT
	End If
	Close MD2FILENUM
	On Error GoTo 0
	'SET MD2CALIBRATED FLAG.
	If MD2VELOCITY(1) = 0 Then
		MD2CALIBRATED = 0
		MD2STATUS = "C"
	Else
		MD2CALIBRATED = -1
	End If
	Exit Sub

'FILE ERROR HANDLER.
MD2SETUPERROR:
	MD2STATUS = "F"
	Resume Next
	
End Sub

Sub MD2STANDBYOFF ()
  
	'---------------------------------------------------------------------
	'NAME:      MD2STANDBYOFF
	'DESC:      THE MD2STANDBYOFF PROCEDURE RETURNS AN MD-2 TO FULL
	'           CURRENT MODE.
	'           THE MD-2 IS SELECTED BY THE MD2MOTOR PARAMETER.
	'           BOTH MOTORS ON THE MD-2 IS AFFECTED
	'           SOME MD-2 SYSTEMS DO NOT HAVE THIS FEATURE, CHECK WITH
	'           YOUR MANUAL FOR MORE INFORMATION.
	'USAGE:     USE TO RETURN AN MD-2 TO FULL CURRENT MODE BEFORE MOVING
	'           THEM FOR MAXIMUM PERFORMANCE.
	'INPUTS:    MOTOR # DETERMINES MD-2.
	'OUTPUTS:   NONE.
	'---------------------------------------------------------------------

	'CHECK FOR SETUP.
	If MD2MTRADR12 <> &H3BC Then MD2STATUS = "S": Exit Sub

	'CHECK FOR VALID MOTOR #.
	MD2STATUS = "B"
	If MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 3 Then MD2STATUS = "O"
	If MD2MOTOR = 4 Or MD2MOTOR = 5 Or MD2MOTOR = 6 Then MD2STATUS = "O"
	If MD2MOTOR = 12 Or MD2MOTOR = 34 Or MD2MOTOR = 56 Then MD2STATUS = "O"
	If MD2STATUS <> "O" Then Exit Sub

	'CHECK FOR PORT AVAILABILITY.
	MD2STATUS = "P"
	If (MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 12) And Not MD2AVAILABLE12 Then Exit Sub
	If (MD2MOTOR = 3 Or MD2MOTOR = 4 Or MD2MOTOR = 34) And Not MD2AVAILABLE34 Then Exit Sub
	If (MD2MOTOR = 5 Or MD2MOTOR = 6 Or MD2MOTOR = 56) And Not MD2AVAILABLE56 Then Exit Sub

	'CHECK FOR ENABLED MD-2.
	MD2STATUS = "E"
	If (MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 12) And Not MD2ENABLED12 Then Exit Sub
	If (MD2MOTOR = 3 Or MD2MOTOR = 4 Or MD2MOTOR = 34) And Not MD2ENABLED34 Then Exit Sub
	If (MD2MOTOR = 5 Or MD2MOTOR = 6 Or MD2MOTOR = 56) And Not MD2ENABLED56 Then Exit Sub
	
	'TURN STANDBY MODE OFF.
	If MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 12 Then
		OUT MD2MTRADR12 + 2, INP(MD2MTRADR12 + 2) Or &H4
	End If
	If MD2MOTOR = 3 Or MD2MOTOR = 4 Or MD2MOTOR = 34 Then
		OUT MD2MTRADR34 + 2, INP(MD2MTRADR34 + 2) Or &H4
	End If
	If MD2MOTOR = 5 Or MD2MOTOR = 6 Or MD2MOTOR = 56 Then
		OUT MD2MTRADR56 + 2, INP(MD2MTRADR56 + 2) Or &H4
	End If
												  
	MD2STATUS = "O"

End Sub

Sub MD2STANDBYON ()
   
	'---------------------------------------------------------------------
	'NAME:      MD2STANDBYON
	'DESC:      THE MD2STANDBYON PROCEDURE PUTS AN MD-2 INTO STANDBY MODE
	'           WHICH REDUCES THE MOTOR CURRENT BY 50%.  THIS RESULTS
	'           IN LOWER HEAT DISSIPATION DURING STANDSTILL WITHOUT
	'           LOSSING ALL HOLDING TORQUE.
	'           THE MD-2 SYSTEM IS SELECTED BY THE MD2MOTOR PARAMETER.
	'           BOTH MOTORS ON AN MD-2 ARE AFFECTED.
	'           MAKE SURE THAT MOTORS ARE ENERGIZED BY FIRST MOVING WITH
	'           MD2HOLD=-1 OR STANDBY WILL NOT WORK.
	'           SOME MD-2 SYSTEMS DO NOT HAVE THIS FEATURE, CHECK WITH
	'           YOUR MANUAL FOR MORE INFORMATION.
	'USAGE:     USE AFTER A MOTION TO MAINTAIN SOME HOLDING TORQUE WHILE
	'           REDUCING HEAT.
	'INPUTS:    MOTOR # DETERMINES MD-2.
	'OUTPUTS:   NONE
	'---------------------------------------------------------------------

	'CHECK FOR SETUP.
	If MD2MTRADR12 <> &H3BC Then MD2STATUS = "S": Exit Sub

	'CHECK FOR VALID MOTOR #.
	MD2STATUS = "B"
	If MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 3 Then MD2STATUS = "O"
	If MD2MOTOR = 4 Or MD2MOTOR = 5 Or MD2MOTOR = 6 Then MD2STATUS = "O"
	If MD2MOTOR = 12 Or MD2MOTOR = 34 Or MD2MOTOR = 56 Then MD2STATUS = "O"
	If MD2STATUS <> "O" Then Exit Sub

	'CHECK FOR PORT AVAILABILITY.
	MD2STATUS = "P"
	If (MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 12) And Not MD2AVAILABLE12 Then Exit Sub
	If (MD2MOTOR = 3 Or MD2MOTOR = 4 Or MD2MOTOR = 34) And Not MD2AVAILABLE34 Then Exit Sub
	If (MD2MOTOR = 5 Or MD2MOTOR = 6 Or MD2MOTOR = 56) And Not MD2AVAILABLE56 Then Exit Sub

	'CHECK FOR ENABLED MD-2.
	MD2STATUS = "E"
	If (MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 12) And Not MD2ENABLED12 Then Exit Sub
	If (MD2MOTOR = 3 Or MD2MOTOR = 4 Or MD2MOTOR = 34) And Not MD2ENABLED34 Then Exit Sub
	If (MD2MOTOR = 5 Or MD2MOTOR = 6 Or MD2MOTOR = 56) And Not MD2ENABLED56 Then Exit Sub

	'TURN STANDBY MODE ON.
	If MD2MOTOR = 1 Or MD2MOTOR = 2 Or MD2MOTOR = 12 Then
		OUT MD2MTRADR12 + 2, INP(MD2MTRADR12 + 2) And &HFB
	End If
	If MD2MOTOR = 3 Or MD2MOTOR = 4 Or MD2MOTOR = 34 Then
		OUT MD2MTRADR34 + 2, INP(MD2MTRADR34 + 2) And &HFB
	End If
	If MD2MOTOR = 5 Or MD2MOTOR = 6 Or MD2MOTOR = 56 Then
		OUT MD2MTRADR56 + 2, INP(MD2MTRADR56 + 2) And &HFB
	End If

	MD2STATUS = "O"

End Sub

